"""Contains logic for issuing a certificate for a host."""

import logging

from sepelib.core.constants import MINUTE_SECONDS
from sepelib.core.exceptions import LogicalError
from walle.clients.certificator import Certificate, CertificatorPersistentError, CertificatorPathNotFoundError
from walle.fsm_stages.common import (
    get_current_stage,
    commit_stage_changes,
    fail_current_stage,
    get_parent_stage,
    complete_current_stage,
    register_stage,
    generate_stage_handler,
    get_stage_deploy_configuration,
)
from walle.stages import Stages

_STATUS_PREPARE = "preparing"
_STATUS_REQUESTING = "requesting"
_STATUS_WAITING = "waiting"
_STATUS_FETCHING = "fetching"

_CERTIFICATOR_POLLING_TIMEOUT = 10
_CERTIFICATE_REPLICA_WAIT_TIMEOUT = 10 * MINUTE_SECONDS  # I don't really know how long does it take.

log = logging.getLogger(__name__)


def _handle_prepare(host):
    parent_stage = get_parent_stage(host)

    configuration = get_stage_deploy_configuration(parent_stage)
    if not configuration.certificate:
        return complete_current_stage(host)

    if parent_stage.has_data("certificate"):
        return complete_current_stage(host)

    commit_stage_changes(host, _STATUS_REQUESTING, check_now=True)


def _handle_requesting(host):
    stage = get_current_stage(host)
    parent_stage = get_parent_stage(host)

    # transition code. safe to remove.
    if parent_stage.has_data("certificate"):
        return complete_current_stage(host)

    try:
        certificate = Certificate.request(host.name)
    except CertificatorPersistentError as e:
        return fail_current_stage(host, str(e))

    _next_step(host, stage, certificate)


def _handle_waiting(host):
    stage = get_current_stage(host)
    certificate = Certificate.from_dict(stage.get_temp_data("certificate_info"))

    try:
        certificate = certificate.refresh_info()
    except CertificatorPersistentError as e:
        return fail_current_stage(host, str(e))

    _next_step(host, stage, certificate)


def _handle_fetching(host):
    stage = get_current_stage(host)
    certificate = Certificate.from_dict(stage.get_temp_data("certificate_info"))

    try:
        allow_secondary = stage.get_temp_data("offload_fetching_to_secondary", False)
        certificate_pem = certificate.fetch(force_primary=not allow_secondary)
        log.info("%s: successfully issued and obtained host certificate %s.", host.human_id(), certificate.url)
    except CertificatorPathNotFoundError as e:
        if stage.timed_out(_CERTIFICATE_REPLICA_WAIT_TIMEOUT):
            return fail_current_stage(host, str(e))

        stage.set_temp_data("offload_fetching_to_secondary", True)
        return commit_stage_changes(host, error=str(e), check_after=_CERTIFICATOR_POLLING_TIMEOUT)
    except CertificatorPersistentError as e:
        return fail_current_stage(host, str(e))

    parent_stage = get_parent_stage(host)
    parent_stage.set_data("certificate", certificate_pem)
    complete_current_stage(host)


def _next_step(host, stage, certificate):
    stage.set_temp_data("certificate_info", certificate.to_dict())
    log.debug("%s: handling certificate %s [%s].", host.human_id(), certificate.url, certificate.status)

    if certificate.issued:
        commit_stage_changes(host, status=_STATUS_FETCHING, check_now=True)

    elif certificate.requested and stage.status == _STATUS_WAITING:
        # don't bump stage status time.
        commit_stage_changes(host, check_after=_CERTIFICATOR_POLLING_TIMEOUT)

    elif certificate.requested:
        commit_stage_changes(host, status=_STATUS_WAITING, check_after=_CERTIFICATOR_POLLING_TIMEOUT)

    else:
        raise LogicalError


# Attention:
# this stage writes data into the parent stage.
register_stage(
    Stages.ISSUE_CERTIFICATE,
    generate_stage_handler(
        {
            _STATUS_PREPARE: _handle_prepare,
            _STATUS_REQUESTING: _handle_requesting,
            _STATUS_WAITING: _handle_waiting,
            _STATUS_FETCHING: _handle_fetching,
        }
    ),
    initial_status=_STATUS_PREPARE,
)
