"""Polls BOT API for host preorder status with distribution retries.
Completes when distribution is finished (host setup status becomes "distributed").

Polls status every 3 minutes.
Retries /_distribute call every 15 minutes.
"""

import logging

from sepelib.core import constants
from walle.clients import bot
from walle.fsm_stages.common import (
    register_stage,
    complete_current_stage,
    get_current_stage,
    commit_stage_changes,
    retry_current_stage,
)
from walle.stages import Stages

log = logging.getLogger(__name__)

STATUS_POLL_INTERVAL = 3 * constants.MINUTE_SECONDS
ACQUIRE_INTERVAL = 15 * constants.MINUTE_SECONDS


def _wait_for_bot_distributed_status(host):
    stage = get_current_stage(host)

    status, order_id = bot.get_host_preorder_status(host.inv)
    if status == bot.SetupRequestStatus.DISTRIBUTED:
        log.info("Host #%s is acquired from preorder and ready for operation.", host.human_id())
        return complete_current_stage(host)
    elif status in bot.SetupRequestStatus.ALL_NO_NEED_TO_WAIT:
        log.info("Host #%s has never been preordered or has been distributed in some other way.", host.human_id())
        return complete_current_stage(host)

    if status == bot.SetupRequestStatus.DELIVERED and stage.timed_out(ACQUIRE_INTERVAL):
        bot.acquire_preordered_host(order_id, host.inv, host.name)
        error = "Retrying host #{} distribution by BOT.".format(host.human_id())
        retry_current_stage(host, error=error, check_after=STATUS_POLL_INTERVAL)

    log.info("Waiting for host #%s to be distributed by BOT.", host.human_id())
    commit_stage_changes(host, check_after=STATUS_POLL_INTERVAL)


register_stage(Stages.WAIT_FOR_BOT_PREORDER, _wait_for_bot_distributed_status)
