"""DNS settings stage."""

import json
import logging

from sepelib.core import config
from sepelib.core.constants import MINUTE_SECONDS, DAY_SECONDS
from sepelib.yandex.dns_api import DnsApiError, DnsApiTemporaryError
from walle import network, audit_log
from walle.clients import dns as dns_clients
from walle.dns.dns_lib import get_operations_for_dns_records, get_delete_operations_for_fqdns
from walle.errors import InvalidHostConfiguration, NoInformationError
from walle.fsm_stages.common import (
    register_stage,
    complete_current_stage,
    commit_stage_changes,
    fail_current_stage,
    get_current_stage,
    generate_stage_handler,
    increase_configurable_error_count,
    retry_current_stage,
)
from walle.hosts import DnsConfiguration
from walle.models import timestamp
from walle.network import DnsRecord
from walle.stages import Stages
from walle.statbox.loggers import dns_operations_logger
from walle.util import misc, notifications
from walle.util.misc import drop_none

log = logging.getLogger(__name__)

_STATUS_PREPARE = "preparing-dns-request"
_STATUS_SEND_REQUEST = "sending-dns-request"
_STATUS_COMPLETE = "completing-dns-request"

STAGE_RETRY_INTERVAL = 20 * MINUTE_SECONDS
"""Use this retry interval when synchronous operation fails."""

STAGE_RETRY_TIMEOUT = 30 * DAY_SECONDS
"""Fail stage after a month of retries."""

DNS_STATUS_COMPLETED = "COMPLETED"
DNS_STATUS_ERROR = "ERROR"


def _set_dns_operations_to_stage(stage, records: list[DnsRecord], operations: list[dns_clients.DnsApiOperation]):
    stage.set_temp_data("dns_records", [record._asdict() for record in records])
    stage.set_temp_data("dns_operations", [operation.to_dict() for operation in operations])


def _get_dns_operations_from_stage(stage):
    records = stage.get_temp_data("dns_records")
    operations = [
        dns_clients.DnsApiOperation.from_dict(operation) for operation in stage.get_temp_data("dns_operations")
    ]

    return records, operations


def _peek_dns_operation_from_stage(stage):
    try:
        return dns_clients.DnsApiOperation.from_dict(stage.get_temp_data("dns_operations")[0])
    except IndexError:
        return None


def _pop_dns_operation_from_stage(stage):
    stage.set_temp_data("dns_operations", stage.get_temp_data("dns_operations")[1:])


def _format_http_request(when, request):
    """Formats HTTP request to string before saving it to stage data"""
    when = misc.format_time(when)
    request_text = "Request at {when}:\n{method} {path}\n{body}".format(
        when=when, method=request.method, path=request.path_url, body=request.body or ""
    )
    return request_text


def _format_http_response(when, response):
    """Formats HTTP response to string before saving it to stage data"""
    when = misc.format_time(when)
    response_text = "Response at {when}:\nHTTP Status {status} {reason}\n{body}".format(
        when=when,
        status=response.status_code,
        reason=response.reason,
        body=response.content or "",
    )
    return response_text


def _save_request_response(stage, request_time, response_time, dns_api_client):
    """
    Save HTTP request and response to stage, so it can be logged somewhere later if necessary
    :param stage: Task's stage object
    :param request_time:  Time at which request has been made
    :param response_time: Time at which response has been received
    :param dns_api_client: DnsApiClient
    :return: Nothing, modifies stage temp data.
    """
    request = dns_api_client.network_client.last_request
    response = dns_api_client.network_client.last_response
    # There may be no request and response objects in the network_client.
    # It happens only if  *very* low-level HTTP errors took place.
    # The task will be failed and there's not much to show to slayer@ anyway.
    if request is None or response is None:
        return

    request_text = _format_http_request(request_time, request)
    response_text = _format_http_response(response_time, response)

    http_session = stage.get_temp_data("dns_api_http_session", [])
    http_session.append(request_text)
    http_session.append(response_text)

    stage.set_temp_data("dns_api_http_session", http_session)


def _clear_request_response(stage):
    stage.del_temp_data("dns_api_http_session")


def _email_http_session(host, stage):
    if stage.get_data("dns_api_http_session_sent", False):
        return

    log = stage.get_temp_data("dns_api_http_session", [])
    if not log:
        return
    subject = "DNS-API session for host {}".format(host.human_id())
    body = "\n\n".join(log)
    recipients = config.get_value("notifications.dns_api_session_to")
    notifications.send_email(recipients, subject, body)

    # stage retries many times before failing. We don't want to send every retry.
    stage.set_data("dns_api_http_session_sent", True)


def _retries_exceeded(host, error):
    return not increase_configurable_error_count(
        host, get_current_stage(host), "dns_api_errors", "dns_api.max_errors", error, fail_stage=False
    )


def _prepare_operation(host):
    # Attention: We imply here that the host has been profiled or synced some reasonable time ago,
    # so either Einstellung or our cache has both host's active MAC and switch correct.

    stage = get_current_stage(host)
    statbox_logger = dns_operations_logger(walle_action="fsm_dns_get_operations_for_host")

    try:
        if stage.get_param("clear", False):
            _prepare_delete_operation(host, stage, statbox_logger)
        else:
            _prepare_update_operation(host, stage, statbox_logger)
    except (NoInformationError, InvalidHostConfiguration, dns_clients.DnsError, network.NoNetworkOnSwitch) as e:
        log.exception("Failed to prepare DNS operation:")
        error = "Failed to prepare DNS operation: {}".format(str(e))

        if stage.timed_out(STAGE_RETRY_TIMEOUT):
            return fail_current_stage(host, error)
        else:
            commit_stage_changes(host, error=error, check_after=STAGE_RETRY_INTERVAL)
    else:
        commit_stage_changes(host, status=_STATUS_SEND_REQUEST, check_now=True)


def _prepare_delete_operation(host, stage, statbox_logger):
    fqdns = network.get_host_fqdns(host)
    operations = get_delete_operations_for_fqdns(fqdns, statbox_logger)
    _set_dns_operations_to_stage(stage, records=[], operations=operations)


def _prepare_update_operation(host, stage, statbox_logger):
    project = host.get_project(fields=network.DNS_REQUIRED_PROJECT_FIELDS)

    current_switch = network.get_current_host_switch_port(host)
    switch = current_switch.switch

    vlan_config = network.get_host_expected_vlans(host, project)
    if vlan_config is None:
        raise InvalidHostConfiguration("project is not set up for DNS auto-configuration.")

    active_mac_info = host.get_active_mac()
    if active_mac_info is None:
        raise InvalidHostConfiguration(
            "Unable to determine active MAC address of the host. "
            "Please, add Wall-E.agent into your setup configuration."
        )
    else:
        active_mac = active_mac_info.mac

    dns_records = network.get_host_dns_records(host, project, host.name, switch, active_mac)
    dns_operations = get_operations_for_dns_records(dns_records, statbox_logger)
    _set_dns_operations_to_stage(stage, dns_records, dns_operations)

    stage.set_temp_data("switch", switch)
    stage.set_temp_data("switch_source", current_switch.source)
    stage.set_temp_data("mac", active_mac_info.mac)
    stage.set_temp_data("mac_source", active_mac_info.source)
    stage.set_temp_data("vlans", vlan_config.vlans)
    stage.set_temp_data("vlan_scheme", project.vlan_scheme)

    log.debug(
        "%s: Going to fix dns records for host with switch=%s, mac=%s, vlans=%s, vlan scheme=%s. Dns operations: %s",
        host.human_id(),
        switch,
        active_mac,
        vlan_config.vlans,
        project.vlan_scheme,
        dns_operations,
    )


def _send_request(host):
    stage = get_current_stage(host)

    dns_records, operations = _get_dns_operations_from_stage(stage)
    if not operations:
        return commit_stage_changes(host, status=_STATUS_COMPLETE, check_now=True)

    is_clear_stage = stage.get_param("clear", False)
    action = "clear" if is_clear_stage else "setup"
    log.debug("%s: Running %s DNS records operations for the host.", host.human_id(), action)

    if _host_records_updated(stage, operations, host.name):
        stage.set_temp_data("dns_update_time", timestamp())

    request_time = timestamp()
    dns_client = host.get_dns()

    audit_entry = _create_audit_log(host, stage, dns_records, operations)

    try:
        with audit_entry:
            dns_client.apply_operations(operations)

    except DnsApiError as e:
        error_message = _parse_dns_api_error_response(e)
        error = "Failed to {action} records: DNS API returned an error: {error}".format(
            action=action, error=error_message
        )
        log.error("%s: Failed to %s records: DNS API returned an error: %s", host.human_id(), action, error_message)

        log.debug(
            "%s: DNS API primitives failed to apply: %s",
            host.human_id(),
            repr([op.to_slayer_api_primitive() for op in operations]),
        )
        if e.response:
            log.debug("%s: DNS API returned an error: %s", host.human_id(), repr(e.response.content))

        _statbox_log_operations(operations, result="error", error_message=error_message)

        retry_interval = config.get_value("dns_api.dns_sync_wait_timeout")
        if not isinstance(e, DnsApiTemporaryError):
            if _retries_exceeded(host, error):
                # TODO(rocco66): generalize dns here
                _save_request_response(stage, request_time, timestamp(), dns_client._dns_api_client)
                _email_http_session(host, stage)  # this will only send message once.
                retry_interval = STAGE_RETRY_INTERVAL

        if stage.timed_out(STAGE_RETRY_TIMEOUT):
            return fail_current_stage(host, error)

        retry_current_stage(host, error=error, check_after=retry_interval)

    else:
        _statbox_log_operations(operations, result="ok")
        commit_stage_changes(host, status=_STATUS_COMPLETE, check_now=True)


def _parse_dns_api_error_response(error):
    result = json.loads(error.response.content)
    http_message = None
    zone = None
    for entry in result.get('errors', []):
        if 'error' in entry:
            return entry['error']
        if 'http_message' in entry and not http_message:
            http_message = entry['http_message']
            if 'zone' in entry and not zone:
                zone = entry['zone']
    message = (
        "{http_message} (for zone: {zone})".format(http_message=http_message, zone=zone)
        if http_message and zone
        else http_message
    )
    return message if message else str(error)


def _create_audit_log(host, stage, dns_records, operations):
    is_clear_stage = stage.get_param("clear", False)

    if is_clear_stage:
        return audit_log.on_clear_dns_records(
            host.project,
            host.inv,
            host.name,
            host.uuid,
            scenario_id=host.scenario_id,
            operations=[operation.to_dict() for operation in operations],
        )
    else:
        return audit_log.on_fix_dns_records(
            host.project,
            host.inv,
            host.name,
            host.uuid,
            operations=[operation.to_dict() for operation in operations],
            records=dns_records,
            switch=stage.get_temp_data("switch"),
            switch_source=stage.get_temp_data("switch_source"),
            mac=stage.get_temp_data("mac"),
            mac_source=stage.get_temp_data("mac_source"),
            scenario_id=host.scenario_id,
        )


def _complete_operation(host):
    stage = get_current_stage(host)
    is_clear_stage = stage.get_param("clear", False)

    action = "clear" if is_clear_stage else "setup"
    log.debug("%s: %s DNS records operation complete.", host.human_id(), action)

    if host.dns is None:
        if not is_clear_stage or stage.has_temp_data("dns_update_time"):
            host.dns = DnsConfiguration()
    else:
        del host.dns.error_time

    if not is_clear_stage:
        host.dns.switch = stage.get_temp_data("switch")
        host.dns.mac = stage.get_temp_data("mac")
        host.dns.vlan_scheme = stage.get_temp_data("vlan_scheme")
        host.dns.vlans = stage.get_temp_data("vlans")
        host.dns.project = host.project
        host.dns.check_time = timestamp()

    if stage.has_temp_data("dns_update_time"):
        host.dns.update_time = stage.get_temp_data("dns_update_time")

    commit_stage_changes(host, extra_fields=["dns"])
    return complete_current_stage(host)


def _statbox_log_operations(operations, result, error_message=None):
    logger = dns_operations_logger(
        walle_action="fsm_dns_synchronous_operation", **drop_none(dict(result=result, error_message=error_message))
    )

    for op in operations:
        logger.log(**op.to_statbox())


def _host_records_updated(stage, operations, host_name):
    has_deletes = any(op.operation == dns_clients.DnsApiOperation.DNS_API_DELETE for op in operations)
    if stage.get_param("create", False) and not has_deletes:
        return False

    for op in operations:
        if op.name == host_name and op.type in ("AAAA", "A"):
            return True

    return False


register_stage(
    Stages.SETUP_DNS,
    generate_stage_handler(
        {
            _STATUS_PREPARE: _prepare_operation,
            _STATUS_SEND_REQUEST: _send_request,
            _STATUS_COMPLETE: _complete_operation,
        }
    ),
    initial_status=_STATUS_PREPARE,
)
