"""Clean CMS task on host if on maintenance. Needed for certain operations with host"""

import logging

from sepelib.core import constants
from sepelib.core.exceptions import LogicalError
from walle.clients.cms import CmsError
from walle.clients.tvm import TvmApiError
from walle.fsm_stages.common import register_stage, complete_current_stage, get_current_stage, commit_stage_changes
from walle.stages import Stages
from walle.statbox.contexts import host_context
from walle.statbox.loggers import cms_logger

_RETRY_PERIOD = constants.MINUTE_SECONDS

log = logging.getLogger(__name__)


def _drop_cms_task(host):
    stage = get_current_stage(host)
    cms_task_id = stage.get_param("cms_task_id")

    context_logger = cms_logger(**host_context(host))
    cms_api_clients = host.get_cms_clients(context_logger)

    if cms_api_clients:
        cms_error_ocurred = False
        message = ""

        for cms_api_client in cms_api_clients:
            log.info("%s: Remove obsolete task %s from CMS.", host.human_id(), cms_task_id)
            try:
                cms_api_client.delete_task(cms_task_id)
            except (CmsError, TvmApiError) as e:
                log.error("%s: Failed to remove task %s from CMS: %s", host.human_id(), cms_task_id, e)
                message += "{}: Failed to remove task {} from CMS: {}".format(host.human_id(), cms_task_id, e)
                cms_error_ocurred = True

        if cms_error_ocurred:
            return commit_stage_changes(host, error=message, check_after=_RETRY_PERIOD)
        else:
            return complete_current_stage(host)

    # no CMS
    raise LogicalError


register_stage(Stages.DROP_CMS_TASK, _drop_cms_task)
