"""Contains all logic for processing host profiling using Einstellung."""

import logging
from contextlib import contextmanager

import walle.admin_requests.request as admin_requests
import walle.hosts
import walle.profile_stat
import walle.stages
from sepelib.core import config, constants
from sepelib.core.exceptions import Error
from walle.admin_requests.severity import admin_request_severity_tag_factory, RequestSource
from walle.clients import eine, otrs, startrek
from walle.clients.eine import EineProfileStatus, EinePersistentError, EineProfileTags
from walle.constants import EMAIL_DOMAIN, EINE_NOP_PROFILE, EINE_PROFILES_WITH_DC_SUPPORT, NETWORK_SOURCE_EINE
from walle.fsm_stages.common import (
    register_stage,
    generate_stage_handler,
    get_current_stage,
    fail_current_stage,
    commit_stage_changes,
    increase_configurable_error_count,
    complete_current_stage,
    retry_parent_stage,
    get_parent_stage,
    terminate_current_stage,
    push_host_ticket,
)
from walle.fsm_stages.constants import EineProfileOperation
from walle.hosts import HostNetwork
from walle.models import timestamp
from walle.stages import Stages, StageTerminals

log = logging.getLogger(__name__)

_EAAS_TAG = "eaas"
"""This tag turns on EaaS mode (DC engineers support). This tag should be local for running operation.
See description here: https://wiki.yandex-team.ru/einstellung/common-profiles
NB: there is no description for eaas tag there :-)
"""

_CHECK_PERIOD = 15
"""Profiling process check period."""

_MAX_ASSIGN_DELAY = 2 * constants.HOUR_SECONDS
"""Silently wait for other profile to finish when trying to assign profile. Light up an error after timeout."""

_PERSISTENT_ERROR_RETRY_PERIOD = 30 * constants.MINUTE_SECONDS
"""Client error or other types of persistent errors are expected to be fixed eventually by developers."""

_ERROR_RETRY_DELAY = 5 * constants.MINUTE_SECONDS
"""Retry period for failed stage."""

_LONG_RETRY_PERIOD = constants.DAY_SECONDS
"""Wait until host owner recover error. Retry stage every day just in case it suddenly help."""

_IN_PROCESS_TIMEOUT = 6 * constants.HOUR_SECONDS
"""Profiling timeout."""

_IN_PROCESS_EXTENDED_TIMEOUT = 18 * constants.HOUR_SECONDS
"""Profiling timeout when running extended test suite (e.g. common-profile with extra-highload-test tag)."""

_STOPPED_TIMEOUT = 2 * constants.HOUR_SECONDS
"""Timeout for waiting while somebody restarts profiling when running without DC support."""

_WAIT_FOR_TICKET_TIMEOUT = 1 * constants.HOUR_SECONDS
"""Timeout for Eine to create ticket for failing host."""

_TICKET_CLOSING_TIMEOUT = 20 * constants.MINUTE_SECONDS
"""Timeout for Eine to close ticket after failing host had been fixed."""

_SERVICE_TIMEOUT = 6 * constants.HOUR_SECONDS
"""Time which we give to DC engineers to debug a failed profile with DC support."""

_DC_SUPPORT_CHECK_PERIOD = 5 * constants.MINUTE_SECONDS
"""Check period for failed profile with DC support."""

_TOTAL_TIMEOUT = constants.WEEK_SECONDS
"""Total timeout for profile operation including delays and waiting for user to resolve the problem."""

_HOST_STATE_TAGS = {"lan-is-flashed"}
"""Maps profile names to a list of tags that store host state instead of configuring profiling actions."""


SUPPORT_SYSTEM = {eine.TicketType.OTRS: otrs, eine.TicketType.STARTREK: startrek}

_STATUS_ASSIGN_PROFILE = "assign-profile"
_STATUS_PROFILING = "profiling"


class _EineOperationFailure(Error):
    def __init__(self, operation):
        message = "Failed to {}."
        super().__init__(message, operation)


@contextmanager
def _eine_operation(host, operation_title):
    try:
        yield
    except EinePersistentError as e:
        log.error("%s: Failed to {}: {}".format(operation_title, str(e)), host.human_id())
        raise _EineOperationFailure(operation_title)


class StageHandler:
    def __init__(self, host):
        self.host = host
        self.stage = get_current_stage(host)
        self.client = eine.get_client(eine.get_eine_provider(self.host.get_eine_box()))

    @classmethod
    def as_handler(cls):
        return lambda host: cls(host).handle()

    def handle(self):
        raise NotImplementedError


class EineAssignProfileHandler(StageHandler):
    def __init__(self, host):
        super().__init__(host)

        self.operation = self.stage.get_param("operation")
        self.profile = self.stage.get_param("profile")
        self.tags = set(self.stage.get_param("profile_tags", []))
        self.vlans_stage_id = self.stage.get_param("vlans_stage", None)
        self.repair_request_severity = self.stage.get_param("repair_request_severity", None)

    def handle(self):
        host = self.host

        try:
            eine_host = self._get_eine_host()
            if not eine_host.in_use():
                return self._try_later("The host is not in use in Einstellung. Please, deal with it.")

            if eine_host.has_profile() and eine_host.in_process() and eine_host.profile() != EINE_NOP_PROFILE:
                message = "eine is currently running '{}' profile on the host. Waiting for it to finish.".format(
                    eine_host.profile()
                )
                return self._wait(message)
            host_network = HostNetwork.get_or_create(host.uuid)
            if eine_host.has_staled_location_info(host_network):
                self._set_host_location()

            self._assign_host_tags()
            self._assign_eine_profile()
        except _EineOperationFailure as e:
            return self._try_later(str(e))
        else:
            # Eine always returns data from MySQL secondary, so it may be outdated right after profile assigning.
            # That's why we don't handle profiling immediately after assigning.
            commit_stage_changes(host, status=_STATUS_PROFILING, check_after=_CHECK_PERIOD)

    def _get_eine_host(self):
        with _eine_operation(self.host, "get host from eine"):
            return self.client.get_host_status(self.host.inv, profile=True, location=True)

    def _wait(self, message):
        """Try proceed later, this is not an error, but we can not proceed now."""

        log.info("Host %s: {}".format(message), self.host.human_id())
        if self.stage.timed_out(_MAX_ASSIGN_DELAY):
            commit_stage_changes(self.host, error=message, check_after=_CHECK_PERIOD)
        else:
            commit_stage_changes(self.host, status_message=message, check_after=_CHECK_PERIOD)

    def _try_later(self, error):
        """Retry on a persistent error, allow bigger period because the problem needs to be fixed by people."""
        log.error("Host %s: {}".format(error), self.host.human_id())  # send to sentry
        commit_stage_changes(self.host, error=error, check_after=_PERSISTENT_ERROR_RETRY_PERIOD)

    def _set_host_location(self):
        host = self.host
        log.info(
            "%s: Eine location info is old and incorrect, setting location info from %s.",
            host.human_id(),
            host.location.network_source,
        )

        with _eine_operation(host, "push proper host location into eine"):
            self.client.set_host_location(host.inv, host.location.switch, host.location.port)
            self.tags.add("_set:do-not-use-snmp")

    def _assign_host_tags(self):
        if self.vlans_stage_id is not None:
            vlan_stage = walle.stages.get_by_uid(self.host.task.stages, self.vlans_stage_id)
            if vlan_stage.get_data('vlan_success', False) and self.host.platform_ipxe_supported():
                self.tags.add(EineProfileTags.SKIP_VLAN_SWITCH)

        if additional_platform_specific_tags := self.host.get_platform_eine_tags():
            self.tags.update(additional_platform_specific_tags)

        # TODO: get rid of that eine.get_tags call, tags can be fetched with eine.get_host_status.
        tags_to_add, tags_to_remove = self._get_tags_operations(
            expected_tags=self.tags, current_tags=self.client.get_tags(self.host.inv)
        )

        if tags_to_add or tags_to_remove:
            with _eine_operation(self.host, "set tags for the host in eine"):
                self.client.update_tags(self.host.inv, add=sorted(tags_to_add), remove=sorted(tags_to_remove))

    def _get_tags_operations(self, expected_tags, current_tags):
        current_tags = set(current_tags)
        tags_to_add = expected_tags - current_tags
        keep_tags = ("PERF:", "_set:")
        remove_tags = current_tags - expected_tags - _HOST_STATE_TAGS
        tags_to_remove = [tag for tag in remove_tags if not tag.startswith(keep_tags)]

        return tags_to_add, tags_to_remove

    def _assign_eine_profile(self):
        if self._dc_support_enabled():
            self.stage.set_temp_data("dc_support", True)
            cc_logins = self._get_cc_logins()
            if not self.repair_request_severity:
                severity_tag = admin_request_severity_tag_factory(self.host, RequestSource.EINE)
            else:
                severity_tag = self.repair_request_severity

            local_tags = [_EAAS_TAG, severity_tag]
        else:
            local_tags, cc_logins = None, None

        self.stage.set_temp_data("profile_assign_time", timestamp())

        with _eine_operation(self.host, "assign profile to host in eine"):
            self.client.assign_profile(
                self.host.inv,
                self.profile,
                local_tags=local_tags,
                cc_logins=cc_logins,
                assigned_for=self.host.task.owner,
            )

    def _get_cc_logins(self):
        login_email_suffix = "@" + EMAIL_DOMAIN
        initiator, to_emails, cc_emails = admin_requests.get_request_owners(
            self.host.project, self.host.task.owner, initiated_by_issuer=True
        )

        cc_logins = [
            email[: -len(login_email_suffix)] for email in to_emails | cc_emails if email.endswith(login_email_suffix)
        ]

        return cc_logins

    def _dc_support_enabled(self):
        """Return True if dc support (a.k.a. EaaS) is enabled for this host profile operation."""
        return (
            self._admin_requests_enabled()
            and self.profile in EINE_PROFILES_WITH_DC_SUPPORT
            and self.operation != EineProfileOperation.DEPLOY
        )

    def _admin_requests_enabled(self):
        return config.get_value("hardware.enable_admin_requests") and not self.host.task.disable_admin_requests


class EineProfileStageHandler(StageHandler):
    """Deal with host profile when eaas (a.k.a DC support) is not enabled."""

    def __init__(self, host):
        super().__init__(host)

        self.profile = self.stage.get_param("profile")
        self.tags = self.stage.get_param("profile_tags", [])
        self.vlans_stage_id = self.stage.get_param("vlans_stage", None)

    def handle(self):
        try:
            eine_host = self._get_eine_host()
        except EinePersistentError as e:
            return self._try_later(str(e))

        if not self._profile_started(eine_host):
            log.error("%s: Eine failed to start the profiling process: %s.", self.host.human_id(), eine_host)
            return self._retry_and_wait_forever("Eine failed to start the profiling process.")

        self._amend_host_status(self.host, eine_host)

        if eine_host.profile() not in self._equal_profiles(self.profile):
            return self._handle_profile_changed(eine_host)

        return self._handle_profile_status(eine_host)

    def _handle_profile_status(self, eine_host):
        log.debug(
            "%s: handling eine profile %s:%s.", self.host.human_id(), eine_host.profile(), eine_host.profile_status()
        )

        try:
            status_handler = {
                EineProfileStatus.QUEUED: self._handle_in_process,
                EineProfileStatus.STOPPED: self._handle_stopped,
                EineProfileStatus.COMPLETED: self._handle_completed,
                EineProfileStatus.FAILED: self._handle_failed,
            }[eine_host.profile_status()]
        except KeyError:
            self._try_later("Profiling process got an unexpected status '{}'.".format(eine_host.profile_status()))
        else:
            return status_handler(eine_host)

    def _handle_profile_changed(self, eine_host):
        message = "Someone changed current host profile in Eine from {} to {}.".format(
            self.profile, eine_host.profile()
        )
        return self._on_profile_changed(eine_host, message, timeout=_STOPPED_TIMEOUT, period=_CHECK_PERIOD)

    def _on_profile_changed(self, eine_host, message, timeout, period):
        """Handle sudden profile switching made by third person."""

        # Give them some time to switch it back. Then switch ourselves.
        # Do not drop their profile if it is running, this may turn server into a brick.
        if eine_host.profile_status() != EineProfileStatus.QUEUED and self._stage_status_timed_out(timeout):
            message += " Switching back."
            return self._retry_and_wait_forever(message)
        else:
            # Not exactly an error. Apparently, humans may do this.
            message += " Waiting for it to complete."
            commit_stage_changes(self.host, status_message=message, check_after=period)

    def _handle_in_process(self, eine_host):
        """Wait for profile to finish."""
        if EineProfileTags.EXTRA_LOAD in self.tags:
            """This tags starts extended test suite which we know run a lot longer then normal profiling does."""
            timeout = _IN_PROCESS_EXTENDED_TIMEOUT
        else:
            timeout = _IN_PROCESS_TIMEOUT

        timeout_message = "Profiling process has timed out."
        self._handle_wait(
            eine_host,
            timeout,
            timeout_message,
            on_timeout=lambda error: commit_stage_changes(self.host, error=error, check_after=_CHECK_PERIOD),
        )

    def _handle_stopped(self, eine_host):
        """Handle sudden profile stop made by third person."""
        timeout = _STOPPED_TIMEOUT
        timeout_message = "Profiling process has been stopped."

        self._handle_wait(eine_host, timeout, timeout_message, on_timeout=self._retry_and_wait_forever)

    def _handle_wait(self, eine_host, timeout, timeout_message, on_timeout):
        """Give some timeout to stage status and retry.
        `on_timeout` is expected to be a function receiving timeout message as only parameter.
        """
        if self.stage.timed_out(_TOTAL_TIMEOUT):
            return fail_current_stage(self.host, timeout_message)

        if self._stage_status_timed_out(timeout):
            on_timeout(timeout_message)
        else:
            commit_stage_changes(self.host, check_after=_CHECK_PERIOD)

    def _handle_completed(self, eine_host):
        """Make some cleanup, record last profile time."""
        if self._is_initial_tune_up():
            log.info("%s: Updating host profiling time.", self.host.human_id())
            _save_network_location(self.host, eine_host.switch(), eine_host.active_mac())
            walle.profile_stat.update_profile_time(self.host.inv, eine_host.profile_updated_timestamp())

        if self.stage.get_data("failure_detected", False):
            complete_current_stage(self.host)
        else:
            terminate_current_stage(StageTerminals.NO_ERROR_FOUND, self.host)

    def _handle_failed(self, eine_host):
        """Handle stage failure."""
        error = "Profile failed - it got {}/{} status: {}".format(
            eine_host.profile_status(), eine_host.get_stage_description(), eine_host.profile_message()
        )

        deploy_failure = self._is_deploy_failure(self.stage, eine_host.profile_status())
        return self._retry_or_fail(error, terminal=StageTerminals.DEPLOY_FAILED if deploy_failure else None)

    def _try_later(self, error):
        """Retry on a persistent error, allow bigger period because the problem needs to be fixed by people."""
        log.error("Host %s: {}".format(error), self.host.human_id())  # send to sentry
        commit_stage_changes(self.host, error=error, check_after=_PERSISTENT_ERROR_RETRY_PERIOD)

    def _retry_or_fail(self, error, terminal=None):
        """Either retry parent stage or terminate current stage with given terminal (default is FAIL)."""
        if self._can_retry_more():
            log.warning("%s: {}".format(error), self.host.human_id())
            retry_parent_stage(self.host, check_after=_ERROR_RETRY_DELAY)
        else:
            self._terminate_current_stage(error, terminal)

    def _terminate_current_stage(self, error, terminal=None):
        # N.B: This stage may be terminated with DEPLOY_FAILED terminal, which default action is to retry parent stage
        # but it may be changed to HIGHLOAD_AND_REDEPLOY.
        if terminal is None:
            terminal = StageTerminals.FAIL

        error = "Too many errors occurred during processing '{}' stage of '{}' task. Last error: {}".format(
            self.stage.name, self.host.status, error
        )

        terminate_current_stage(terminal, self.host, error)

    def _retry_and_wait_forever(self, error):
        """Retry parent stage to reduce the bad luck syndrome."""
        if self._can_retry_more():
            log.warning("%s: {}".format(error), self.host.human_id())
            retry_parent_stage(self.host, check_after=_ERROR_RETRY_DELAY)
        else:
            self._retry_every_day(error)

    def _retry_every_day(self, error):
        """Try just wait for the problem to be resolved by people.
        We have to persist the error message between runs.
        """
        error = "Too many errors occurred during processing '{}' stage of '{}' task. Last error: {}".format(
            self.stage.name, self.host.status, error
        )

        if self._stage_status_timed_out(_LONG_RETRY_PERIOD):
            log.info(
                "%s: {} No retries left, but hope is still alive. Retrying stage.'.".format(error), self.host.human_id()
            )
            retry_parent_stage(self.host, error=error, check_after=_ERROR_RETRY_DELAY)
        else:
            log.info("%s: {} No retries left.".format(error), self.host.human_id())
            error_message = "Waiting until somebody run profile '{}' and it completes successfully. {}".format(
                self.profile, error
            )
            commit_stage_changes(self.host, error=error_message, check_after=_PERSISTENT_ERROR_RETRY_PERIOD)

    def _can_retry_more(self):
        return increase_configurable_error_count(
            self.host, self.stage, name="eine_errors", limit_name="profiling.max_errors", error=None, fail_stage=False
        )

    def _get_eine_host(self):
        return self.client.get_host_status(self.host.inv, profile=True, location=True)

    def _profile_started(self, eine_host):
        """Return True if profile that we scheduled started successfully."""
        # Eine always returns data from MySQL secondary, so it may be outdated. If we don't detect it we can decide that
        # profile is completed if Eine return us data from previous profile operation.
        minimal_assign_time = self.stage.get_temp_data("profile_assign_time") - 1

        return eine_host.has_profile() and eine_host.profile_assigned_timestamp() >= minimal_assign_time

    @staticmethod
    def _equal_profiles(profile):
        # A dirty hack that was done by a request from Eine engineers: sometimes when a node fails on common-firmware DC
        # engineers can fix this only by assigning another version of common-firmware which is called firmware-legacy.
        # Wall-E shouldn't somehow react on this change and treat it as if node is profiling by common-firmware.
        if profile in EINE_PROFILES_WITH_DC_SUPPORT:
            return EINE_PROFILES_WITH_DC_SUPPORT + ["firmware-legacy"]
        else:
            return [profile]

    def _is_initial_tune_up(self):
        return (
            self.stage.get_param("operation") in (EineProfileOperation.PREPARE, EineProfileOperation.PROFILE)
            and self.stage.get_param("profile") in EINE_PROFILES_WITH_DC_SUPPORT
        )

    @staticmethod
    def _is_deploy_failure(stage, eine_status):
        return stage.get_param("operation") == EineProfileOperation.DEPLOY and eine_status == EineProfileStatus.FAILED

    @staticmethod
    def _amend_host_status(host, eine_host):
        """Show eine stage description/status in UI."""
        # TODO: Looks like a hack. Consider to introduce substatuses or any other common way to add an additional status
        host.task.status = host.task.stage_name + ":" + eine_host.get_stage_description()

    def _stage_status_timed_out(self, timeout):
        """Return True if current stage lasts too long."""
        return self.stage.timed_out(timeout)


class EineProfileEaasStageHandler(EineProfileStageHandler):
    """Additional logic for profiling hosts with eaas support."""

    def _handle_profile_status(self, eine_host):
        self._set_current_status(eine_host.profile_status())
        super()._handle_profile_status(eine_host)

    def _handle_completed(self, eine_host):
        ticket_id = eine_host.ticket_id()

        if ticket_id is None:
            super()._handle_completed(eine_host)

        elif self._stage_status_timed_out(_TICKET_CLOSING_TIMEOUT):
            log.warning(
                "%s: EaaS profile completed but ticket has not been closed during timeout. Finishing task.",
                self.host.human_id(),
            )
            super()._handle_completed(eine_host)

        else:
            message = "EaaS profile completed. Waiting for ticket to close."
            commit_stage_changes(self.host, status_message=message, check_after=_CHECK_PERIOD, extra_fields=["ticket"])

    def _handle_profile_changed(self, eine_host):
        """Handle sudden profile switching made by third person."""
        eine_profile = eine_host.profile()
        self._set_current_status("{}:{}".format(eine_host.profile(), eine_host.profile_status()))

        if eine_profile in eine.EINE_SERVICE_PROFILES:
            self._handle_on_service(eine_host)
        else:
            super()._handle_profile_changed(eine_host)

    def _handle_on_service(self, eine_host):
        """Handle case when DC people switch profile to one of service profiles."""
        message = "Someone has switched the host to '{}' service profile.".format(eine_host.profile())
        return super()._on_profile_changed(
            eine_host, message=message, timeout=_SERVICE_TIMEOUT, period=_DC_SUPPORT_CHECK_PERIOD
        )

    def _handle_failed(self, eine_host):
        """Handle stage failure."""
        message = "Profile failed - it got {}/{} status: {}".format(
            eine_host.profile_status(), eine_host.get_stage_description(), eine_host.profile_message()
        )

        if self.stage.timed_out(_TOTAL_TIMEOUT):
            error_message = "Host profiling process (with DC support) has timed out: {}".format(message)
            return fail_current_stage(self.host, error_message)

        ticket_id = eine_host.ticket_id()

        if ticket_id is None and self._stage_status_timed_out(_WAIT_FOR_TICKET_TIMEOUT):
            # Show ticket absence as an error. Stage failure is not an error, DC engineers gonna handle it.
            if _EAAS_TAG in eine_host.local_tags():
                return self._where_is_my_eaas_ticket(message)
            else:
                # Somebody dropped tag from einestellung - that's why there is no ticket.
                # Restart operation with tags. NB: ITDC guys don't like when Wall-E restarts profiles when they work,
                # but there is no ticket associated with this host, so we kind of assume they don't work on it.
                return self._retry_and_wait_forever(message)

        if ticket_id:
            message += " (see {})".format(startrek.get_ticket_url_by_id(ticket_id))
            push_host_ticket(self.host, ticket_id)
            self.stage.setdefault_data("failure_detected", True)

        commit_stage_changes(
            self.host, status_message=message, check_after=_DC_SUPPORT_CHECK_PERIOD, extra_fields=["ticket"]
        )

    def _handle_wait(self, eine_host, timeout, timeout_message, on_timeout):
        """Give some timeout to stage status and retry."""
        ticket_id = eine_host.ticket_id()

        if ticket_id and not self.stage.timed_out(_TOTAL_TIMEOUT):
            # Do not try to restart profile until DC engineers done.
            push_host_ticket(self.host, ticket_id)
            commit_stage_changes(self.host, check_after=_CHECK_PERIOD, extra_fields=["ticket"])

        else:
            super()._handle_wait(eine_host, timeout, timeout_message, on_timeout)

    def _set_current_status(self, status):
        if self.stage.get_temp_data("profile_status", None) != status:
            self.stage.set_temp_data("profile_status", status)
            self.stage.set_temp_data("profile_status_time", timestamp())

    def _stage_status_timed_out(self, timeout):
        """Return True if current profile status lasts more then we allow it to."""
        return self.stage.timed_out(timeout, "profile_status_time")

    def _where_is_my_eaas_ticket(self, failure_message):
        """Show ticket absence as an error."""
        log.error("%s: Host profiling failed but DC support ticket hasn't been created.", self.host.human_id())
        error = failure_message + " but no eaas ticket created."
        return commit_stage_changes(self.host, error=error, check_after=_DC_SUPPORT_CHECK_PERIOD)


def _handle_profile(host):
    stage = get_current_stage(host)
    if stage.get_temp_data("dc_support", False):
        return EineProfileEaasStageHandler(host).handle()
    else:
        return EineProfileStageHandler(host).handle()


def _cancel_stage(host, stage):
    parent_stage = get_parent_stage(host, stage)
    if not walle.stages.is_descendant(get_current_stage(host), parent_stage):
        return

    log.info("%s: Cancelling host profiling stage...", host.human_id())
    try:
        _drop_profile(host)
    except Exception as e:
        log.error("%s: Failed to cancel host profiling: %s", host.human_id(), e)


def _handle_drop_profile(host):
    try:
        _drop_profile(host)
    except EinePersistentError as e:
        fail_current_stage(host, str(e))
    else:
        complete_current_stage(host)


def _drop_profile(host, eine_host=None):
    client = eine.get_client(eine.get_eine_provider(host.get_eine_box()))
    if eine_host is None:
        eine_host = client.get_host_status(host.inv, profile=True)

    if eine_host.has_profile() and eine_host.in_process() and eine_host.profile() != EINE_NOP_PROFILE:
        log.info("%s: Cancelling profile %s.", host.human_id(), eine_host.profile())
        client.assign_profile(host.inv, EINE_NOP_PROFILE, assigned_for=host.task.owner)


def _save_network_location(host, switch_info, mac_info):
    host_network = HostNetwork.get_or_create(host.uuid)
    if switch_info is not None:
        walle.hosts.update_network_location(
            host, host_network, switch_info.switch, switch_info.port, switch_info.timestamp, NETWORK_SOURCE_EINE
        )
    if mac_info is not None:
        walle.hosts.update_eine_active_mac(host, host_network, mac_info.active, mac_info.timestamp)


# Attention: These stages retry parent stage on error.
register_stage(
    Stages.EINE_PROFILE,
    handler=generate_stage_handler(
        {
            _STATUS_ASSIGN_PROFILE: EineAssignProfileHandler.as_handler(),
            _STATUS_PROFILING: _handle_profile,
        }
    ),
    initial_status=_STATUS_ASSIGN_PROFILE,
    cancellation_handler=_cancel_stage,
)

register_stage(Stages.DROP_EINE_PROFILE, _handle_drop_profile)
