"""Host fqdn deinvalidation stage."""

import logging

from walle import audit_log
from walle import authorization
from walle.clients import bot
from walle.fsm_stages.common import complete_current_stage, fail_current_stage
from walle.fsm_stages.common import register_stage, get_current_stage
from walle.hosts import Host
from walle.stages import Stages

log = logging.getLogger(__name__)


def _handle(host):
    log.info("%s: Change host's fqdn", host.human_id())

    old_name = host.name
    new_name = bot.get_host_info(host.inv)['name']

    if Host.objects(name=new_name).count():
        message = "Host with fqdn {} already exists in Wall-e.".format(new_name)
        return fail_current_stage(host, error=message)

    if new_name != old_name:
        stage = get_current_stage(host)

        reason = stage.get_param("reason", None)

        log_entry = audit_log.on_host_fqdn_changed(
            issuer=authorization.ISSUER_WALLE,
            project_id=host.project,
            inv=host.inv,
            host_uuid=host.uuid,
            old_fqdn=old_name,
            new_fqdn=new_name,
            reason=reason,
        )

        with log_entry:
            host.modify(set__name=new_name)

        log.info("%s: Successfully changed host's fqdn from %s to %s", host.human_id(), old_name, new_name)

    else:
        log.error(
            "%s: Failed to change host's fqdn, old %s and new %s fqdns are the same",
            host.human_id(),
            old_name,
            new_name,
        )

    complete_current_stage(host)


register_stage(Stages.FQDN_DEINVALIDATION, _handle)
