"""Contains common logic for hardware error fixing."""

import logging

import walle.expert.constants as expert_constants
from sepelib.core import constants
from sepelib.core.exceptions import Error
from walle.admin_requests import request as admin_requests
from walle.clients import startrek
from walle.expert.decision import Decision
from walle.expert.types import WalleAction
from walle.fsm_stages.common import (
    MONITORING_PERIOD,
    ADMIN_REQUEST_CHECK_INTERVAL,
    commit_stage_changes,
    fail_current_stage,
    get_current_stage,
    cancel_task,
    generate_stage_handler,
    push_host_ticket,
    complete_current_stage,
    cancel_task_considering_operation_limits,
)
from walle.fsm_stages.constants import StageStatus
from walle.hosts import TaskType
from walle.models import timestamp
from walle.clients import dmc

log = logging.getLogger(__name__)

TICKET_RESOLVE_CHECK_INTERVAL = 2 * constants.MINUTE_SECONDS
"""Interval for checking ticket resolve."""

STARTREK_RESOLUTION_FIELD = "resolution"

_MONITORING_TIMEOUT = max(
    constants.HOUR_SECONDS,
    expert_constants.HW_WATCHER_CHECK_PERIOD
    + expert_constants.HW_WATCHER_CHECK_ACCURACY
    + expert_constants.MONITORING_TIMEOUT,
)
_LOST_TICKET_TIMEOUT = 2 * constants.HOUR_SECONDS

REQUEST_ID_STAGE_FIELD_NAME = "request_id"
TICKET_ID_STAGE_FIELD_NAME = "ticket_id"
REDEPLOY_STAGE_FIELD_NAME = "redeploy"


class HwErrorsStageHandler:
    operation = None
    check = None

    def __init__(self, host):
        self._host = host
        self._stage = get_current_stage(host)

        if self.operation is None:
            raise Error("Stage handler {} implemented incorrectly")

    @classmethod
    def as_handler(cls):
        return generate_stage_handler(
            {
                StageStatus.HW_ERRORS_PENDING: lambda host: cls(host)._handle_pending(),
                StageStatus.HW_ERRORS_WAITING_DC: lambda host: cls(host)._handle_wait(),
                StageStatus.HW_ERRORS_WAITING_RESOLVE_TICKET: lambda host: cls(host)._handle_wait_ticket(),
            }
        )

    def handle_action(self, decision):
        # these methods provided for older stage handlers (change disk, repair link) to be able to cut in.
        # in newer stages we don't use this decision params, we use the original decision.
        self._stage.set_temp_data("decision_params", decision.params)
        self._stage.set_temp_data("decision_reason", decision.reason)
        self.handle_create()

    def handle_create(self):
        host = self._host
        stage = self._stage

        decision_params = stage.get_param("decision_params")
        reason = stage.get_param("decision_reason", "")

        # for "backwards compatibility": allow this handler to be used with current rules for memory and link problems.
        # this should be dropped eventually.
        decision_params.pop("failure_type", None)
        decision_params.pop("reboot", None)

        request_type = admin_requests.RequestTypes.by_typename(decision_params.pop("request_type"))
        if request_type in admin_requests.RequestTypes.ALL_NOC:
            ticket_id = admin_requests.create_noc_request(host, request_type, reason, **decision_params)
            stage.set_temp_data(TICKET_ID_STAGE_FIELD_NAME, ticket_id)
            if push_host_ticket(self._host, ticket_id):
                commit_stage_changes(self._host, extra_fields=["ticket"])
            commit_stage_changes(self._host, status=StageStatus.HW_ERRORS_WAITING_RESOLVE_TICKET, check_now=True)
            return
        request_id, ticket_id = admin_requests.create_admin_request(host, request_type, reason, **decision_params)

        stage.set_temp_data(REQUEST_ID_STAGE_FIELD_NAME, request_id)
        stage.set_temp_data(TICKET_ID_STAGE_FIELD_NAME, ticket_id)
        commit_stage_changes(host, status=StageStatus.HW_ERRORS_WAITING_DC, check_now=True)

    def handle_completed(self):
        complete_current_stage(self._host)

    def _handle_pending(self):
        stage = get_current_stage(self._host)
        orig_decision_params = self._stage.get_data("orig_decision", {}).get("params", {})
        if REQUEST_ID_STAGE_FIELD_NAME in orig_decision_params:
            stage.set_temp_data(REQUEST_ID_STAGE_FIELD_NAME, orig_decision_params[REQUEST_ID_STAGE_FIELD_NAME])
            stage.set_temp_data(TICKET_ID_STAGE_FIELD_NAME, orig_decision_params[TICKET_ID_STAGE_FIELD_NAME])
            stage.set_temp_data(REDEPLOY_STAGE_FIELD_NAME, orig_decision_params.get(REDEPLOY_STAGE_FIELD_NAME, False))
            # NOTE(rocco66): we have already known admin requests ID, so just wait it
            commit_stage_changes(self._host, status=StageStatus.HW_ERRORS_WAITING_DC, check_now=True)
            return

        if self._host.task.type == TaskType.AUTOMATED_HEALING:
            _, decision = dmc.get_decisions_from_handler(
                self._host, decision_params=dmc.DecisionParams(checks={self.check} if self.check else set())
            )

            if decision.action == WalleAction.WAIT:
                self._set_monitoring_time()
                return self._handle_monitoring(decision.reason)

            if decision.action == WalleAction.HEALTHY:
                if cancel_task_considering_operation_limits(self._host):
                    return
                # cannot cancel anymore, continue healing with original decision
                elif self._stage.has_data("orig_decision"):
                    decision = Decision(**self._stage.get_data("orig_decision"))
                else:  # TODO backward compatibility, remove after all tasks w/o orig_decision finish
                    return
        else:
            decision = Decision.healthy("Host is healthy.")

        self._reset_monitoring_time()
        self.handle_action(decision)

    def _handle_wait(self):
        request_id = self._stage.get_temp_data(REQUEST_ID_STAGE_FIELD_NAME)

        request = admin_requests.get_request_status(request_id)

        if request.get("ticket"):
            if push_host_ticket(self._host, request["ticket"]):
                commit_stage_changes(self._host, extra_fields=["ticket"])

        if self._request_completed(request):
            commit_stage_changes(self._host, status=StageStatus.HW_ERRORS_WAITING_RESOLVE_TICKET, check_now=True)

    def _handle_wait_ticket(self):
        ticket_id = self._stage.get_temp_data(TICKET_ID_STAGE_FIELD_NAME, None)
        if ticket_id is None or ticket_id == "":
            self.handle_completed()
            return

        if self._ticket_resolved(ticket_id):
            self.handle_completed()

    def _handle_monitoring(self, message):
        error = "Unable to check actuality of {} operation: {}".format(self.operation, message)
        if self._stage.timed_out(_MONITORING_TIMEOUT, "monitoring_start_time"):
            cancel_task(self._host, error)
        else:
            commit_stage_changes(self._host, error=error, check_after=MONITORING_PERIOD)

    def _request_completed(self, request):
        host = self._host
        request_id = request["request_id"]

        if request["status"] == admin_requests.STATUS_PROCESSED:
            return True

        elif request["status"] == admin_requests.STATUS_IN_PROCESS:
            commit_stage_changes(
                host,
                check_after=ADMIN_REQUEST_CHECK_INTERVAL,
                status_message="Waiting for #{} BOT admin request to complete.".format(request_id),
            )
        elif request["status"] == admin_requests.STATUS_DELETED:
            log.warn(
                "%s: BOT request {} was deleted (status {}). "
                "Re-trying bot admin request.".format(request_id, request["info"]),
                host.human_id(),
            )
            self.handle_create()
        elif request["status"] == admin_requests.STATUS_NOT_EXIST:
            stage = get_current_stage(host)
            error = "BOT request {} does not exist (got error {})".format(request_id, request["info"])

            if not stage.has_temp_data("request_gone_timestamp"):
                stage.set_temp_data("request_gone_timestamp", timestamp())

            if timestamp() - stage.get_temp_data("request_gone_timestamp") <= _LOST_TICKET_TIMEOUT:
                log.info("%s: {}. Admin request is lost. Wait for it to came back.".format(error), host.human_id())
                commit_stage_changes(host, error=error)
            else:
                log.warn("%s: {}. Admin request is lost. Re-trying.".format(error), host.human_id())
                self.handle_create()

        else:
            fail_current_stage(
                host, "BOT admin request #{} got unexpected status {}.".format(request_id, request["info"])
            )

        return False

    def _ticket_resolved(self, ticket_id):
        try:
            client = startrek.get_client()
            info = client.get_issue(ticket_id)
        except startrek.StartrekClientRequestError as e:
            if e.response.status_code == 404:  # ticket does not exist
                log.warn("Startrek ticket %s does not exist.", ticket_id)
                return True
            else:
                self._handle_check_ticket_error(ticket_id, str(e))
                return False
        except Exception as e:
            self._handle_check_ticket_error(ticket_id, str(e))
            return False

        if STARTREK_RESOLUTION_FIELD in info:
            return True

        commit_stage_changes(
            self._host,
            check_after=TICKET_RESOLVE_CHECK_INTERVAL,
            status_message="Waiting for Startrek ticket {} to resolve.".format(ticket_id),
        )
        return False

    def _handle_check_ticket_error(self, ticket_id, error):
        error_message = "Failed to get status for Startrek ticket {}: {}".format(ticket_id, error)
        log.error(error_message)
        commit_stage_changes(self._host, check_after=TICKET_RESOLVE_CHECK_INTERVAL, error=error_message)

    def _set_monitoring_time(self):
        self._stage.setdefault_temp_data("monitoring_start_time", timestamp())

    def _reset_monitoring_time(self):
        self._stage.del_temp_data("monitoring_start_time")
