"""Stage that updates network location from a few known sources."""
import logging
from collections import namedtuple

import mongoengine
import walle.stages


from sepelib.core import constants
from walle.clients import eine
from walle.clients.network import racktables_client
from walle.constants import (
    NETWORK_SOURCE_SNMP,
    NETWORK_SOURCE_EINE,
    VLAN_SCHEMES_THAT_DONT_WAIT_IN_NETWORK_LOCATION_STAGES,
)
from walle.fsm_stages.common import (
    complete_current_stage,
    get_current_stage,
    register_stage,
    fail_current_stage,
    commit_stage_changes,
    complete_parent_stage,
    generate_stage_handler,
)
from walle.host_network import HostNetwork
from walle.hosts import Host, update_eine_active_mac, update_snmp_active_mac, update_network_location
from walle.models import timestamp
from walle.stages import Stages

log = logging.getLogger(__name__)

_NETMAP_WAIT_TIMEOUT = constants.HOUR_SECONDS + 20 * constants.MINUTE_SECONDS
_CHECK_PERIOD = constants.MINUTE_SECONDS

_STATUS_CHECK = "checking"
_STATUS_UPDATE_FROM_RACKTABLES = "updating-from-racktables"
_STATUS_UPDATE_FROM_EINE = "updating-from-eine"
_STATUS_WAIT_FOR_NETMAP = "waiting-for-netmap"

_STATUS_CHECK_SWITCH_PORT_IN_WALLE = "check-switch-port-in-walle"
_STATUS_CHECK_POWER_ON_NEED = "check-power-on-need"


update_received = namedtuple("UpdateReceived", ["mac_updated", "location_updated"])


def _handle_check(host):
    stage = get_current_stage(host)
    force = stage.get_param("force", False)

    if not force and host.get_active_mac() and host.get_current_network_location():
        _complete_update_stage(host)
    else:
        log.debug("%s: host will be have recent network location info", host.human_id())
        commit_stage_changes(host, status=_STATUS_UPDATE_FROM_RACKTABLES, check_now=True)


def _handle_update_from_racktables_snmp(host):
    if all(_update_from_racktables_snmp(host)):
        _complete_update_stage(host)
    else:
        log.debug("%s: no recent network location from racktables snmp", host.human_id())
        commit_stage_changes(host, status=_STATUS_UPDATE_FROM_EINE, check_now=True)


def _handle_update_from_eine(host):
    if all(_update_from_eine(host)):
        _complete_update_stage(host)
    else:
        log.debug("%s: no recent network location from eine", host.human_id())
        commit_stage_changes(host, status=_STATUS_WAIT_FOR_NETMAP, check_now=True)


def _handle_wait_for_netmap(host):
    stage = get_current_stage(host)

    if host.get_active_mac() and host.get_current_network_location():
        return _complete_update_stage(host)

    if not _should_wait_for_netmap(host):
        return fail_current_stage(host, error="Failed to determine network location and active mac address.")

    if stage.timed_out(_NETMAP_WAIT_TIMEOUT):
        error_message = "Can't determine host's network location, giving up after {} seconds".format(
            _NETMAP_WAIT_TIMEOUT
        )
        fail_current_stage(host, error=error_message)
    else:
        commit_stage_changes(host, check_after=_CHECK_PERIOD)


def _should_wait_for_netmap(host):
    stage = get_current_stage(host)
    if not stage.get_param("wait_for_racktables", False):
        return False

    project = host.get_project(fields=("vlan_scheme",))
    if project.vlan_scheme in VLAN_SCHEMES_THAT_DONT_WAIT_IN_NETWORK_LOCATION_STAGES:
        return False

    return True


def _handle_compatibility_mode(host):
    stage = get_current_stage(host)
    if stage.status is None:
        log.debug("%s: Legacy stage %s started, falling into compatibility mode", host.human_name(), stage.name)
        return commit_stage_changes(host, status=_STATUS_CHECK, check_now=True)

    log.debug("%s: Legacy stage %s, using compatibility mode", host.human_name(), stage.name)
    return _handle_network_location_stage(host)


def _update_from_racktables_snmp(host):
    active_mac_info = None
    active_mac_address = None

    mac_updated = False
    location_updated = False

    for mac in host.macs:
        mac_info = racktables_client.get_mac_status(mac)
        if mac_info is None:
            continue

        if active_mac_info is None or mac_info["Port_timestamp"] > active_mac_info["Port_timestamp"]:
            active_mac_info = mac_info
            active_mac_address = mac

    if active_mac_info is not None:
        port_timestamp = active_mac_info["Port_timestamp"]

        try:
            host_network = HostNetwork.get_or_create(host.uuid)
            mac_updated = update_snmp_active_mac(host, host_network, active_mac_address, port_timestamp)

            location_updated = update_network_location(
                host,
                host_network,
                active_mac_info["Switch"],
                active_mac_info["Port"],
                port_timestamp,
                NETWORK_SOURCE_SNMP,
            )
        except mongoengine.DoesNotExist:
            log.error('Error while getting host network info for host %s', host.human_id())

    else:
        log.info('Racktables did not return active MAC addr for host %s (choices were %s)', host.human_id(), host.macs)

    return update_received(mac_updated, location_updated)


def _update_from_eine(host):
    try:
        client = eine.get_client(eine.get_eine_provider(host.get_eine_box()))
        eine_host = client.get_host_status(host.inv, location=True)
    except (eine.EineHostDoesNotExistError, AttributeError) as e:
        log.error("Failed to get host #%s information from Eine: %s", str(e))
        return False

    mac_updated = False
    location_updated = False
    timestamp_threshold = timestamp() - (constants.DAY_SECONDS + constants.HOUR_SECONDS)

    mac_info = eine_host.active_mac()

    try:
        host_network = HostNetwork.get_or_create(host.uuid)
    except mongoengine.DoesNotExist:
        log.error("Failed to get host network info %s", host.human_id())
        return False

    if mac_info and mac_info.timestamp > timestamp_threshold:
        mac_updated = update_eine_active_mac(host, host_network, mac_info.active, mac_info.timestamp)
    else:
        log.info("Active mac address received from eine for host %s is too old", host.human_id())

    location_info = eine_host.switch()
    if location_info and location_info.timestamp > timestamp_threshold:
        location_updated = update_network_location(
            host, host_network, location_info.switch, location_info.port, location_info.timestamp, NETWORK_SOURCE_EINE
        )
    else:
        log.info("Location info received from eine for host %s is too old", host.human_id())

    return update_received(mac_updated, location_updated)


def _complete_update_stage(host: Host):
    get_current_stage(host).set_data("update_network_success", True)
    return complete_current_stage(host)


def _handle_check_switch_port_in_walle(host: Host):
    host_network = host.get_current_network_location()
    if host_network:
        query = mongoengine.Q(
            network_switch=host_network.network_switch,
            network_port=host_network.network_port,
            network_timestamp__gte=host_network.network_timestamp,
        )
        for found_host_network in HostNetwork.objects(query):
            if found_host_network.uuid != host_network.uuid:
                return fail_current_stage(
                    host,
                    "found more recent host on {}/{}; subsequent operations are dangerous".format(
                        host_network.network_switch, host_network.network_port
                    ),
                )
    return commit_stage_changes(host, status=_STATUS_CHECK_POWER_ON_NEED, check_now=True)


def _handle_check_power_on_need(host: Host):
    # If updating network on previous stage was not success,
    # need to power on the host by IMPI for detecting his location,
    # so continue parent stage.
    # Otherwise may break parent stage and move host to Eine

    stage = get_current_stage(host)
    update_network_stage_id = stage.get_param("update_network_stage_id", None)
    update_network_stage = walle.stages.get_by_uid(host.task.stages, update_network_stage_id)
    if update_network_stage.get_data("update_network_success", False):
        return complete_parent_stage(host, stage)

    return complete_current_stage(host)


_handle_network_location_stage = generate_stage_handler(
    {
        _STATUS_CHECK: _handle_check,
        _STATUS_UPDATE_FROM_RACKTABLES: _handle_update_from_racktables_snmp,
        _STATUS_UPDATE_FROM_EINE: _handle_update_from_eine,
        _STATUS_WAIT_FOR_NETMAP: _handle_wait_for_netmap,
    }
)


_handle_verify_switch_port = generate_stage_handler(
    {
        _STATUS_CHECK_SWITCH_PORT_IN_WALLE: _handle_check_switch_port_in_walle,
        _STATUS_CHECK_POWER_ON_NEED: _handle_check_power_on_need,
    }
)


register_stage(Stages.WAIT_FOR_ACTIVE_MAC, initial_status=_STATUS_CHECK, handler=_handle_compatibility_mode)
register_stage(Stages.WAIT_FOR_SWITCH_PORT, initial_status=_STATUS_CHECK, handler=_handle_compatibility_mode)
register_stage(Stages.UPDATE_NETWORK_LOCATION, initial_status=_STATUS_CHECK, handler=_handle_network_location_stage)


register_stage(
    Stages.VERIFY_SWITCH_PORT, initial_status=_STATUS_CHECK_SWITCH_PORT_IN_WALLE, handler=_handle_verify_switch_port
)
