"""Contains all logic for handling host rebooting via IPMI."""

import logging
from functools import partial

import walle.admin_requests.request as admin_requests
import walle.fsm_stages.common
from sepelib.core import config
from sepelib.core.constants import MINUTE_SECONDS
from sepelib.core.exceptions import LogicalError
from walle.clients import ipmiproxy
from walle.expert import dmc
from walle.expert.types import CheckType, CheckStatus
from walle.fsm_stages import ipmi_errors
from walle.fsm_stages.common import (
    register_stage,
    get_current_stage,
    complete_current_stage,
    commit_stage_changes,
    retry_parent_stage,
    generate_stage_handler,
    complete_parent_stage,
    get_parent_stage,
    retry_action,
)
from walle.fsm_stages.power_post_util import wait_post_complete, STATUS_WAIT_POST_COMPLETE, goto_post_code_check
from walle.host_platforms import create_platform_for_host
from walle.stages import Stages

log = logging.getLogger(__name__)


_HARDWARE_CHECK_PERIOD = 5
"""Power on/off check period."""

_POWER_ON_OF_TIMEOUT = 30
"""Host powering on/off timeout."""

_SOFT_POWER_OFF_TIMEOUT = 10 * MINUTE_SECONDS
"""Allow host a few minutes to power off gracefully."""

_STATUS_SOFT_POWER_OFF = "soft-power-off"
"""Soft power off is scheduled."""

_STATUS_SOFT_POWERING_OFF = "soft-powering-off"
"""Host is powering off (soft)."""

_STATUS_POWER_OFF = "power-off"
"""Power off is scheduled."""

_STATUS_POWERING_OFF = "powering-off"
"""Host is powering off."""

_STATUS_POWER_ON = "power-on"
"""Power on is scheduled."""

_STATUS_POWERING_ON = "powering-on"
"""Host is powering on."""


#
# Power-off stage handlers
#


def _power_off_handler(host, handler):
    """Abstract power-off handler."""

    stage = get_current_stage(host)

    if _handle_retry_on_existing_admin_request(host, stage):
        return

    ipmi_client = host.get_ipmi_client()
    if not ipmi_client.is_power_on():
        log.info("Host %s has powered off.", host.human_id())
        return complete_current_stage(host)

    return handler(host)


def _handle_soft_poweroff(host):
    ipmi_client = host.get_ipmi_client()

    try:
        ipmi_client.soft_power_off()
    except ipmiproxy.BrokenIpmiCommandError as e:
        # Some nodes always reject `power soft` command but successfully accept `power off` command
        # (see https://st.yandex-team.ru/RND-340). Use hard power off as a temporary workaround.
        message = "%s rejected soft power off request: %s Falling back to hard power off..."
        log.warning(message, host.human_id(), e)
        commit_stage_changes(host, status=_STATUS_POWER_OFF, check_now=True)
    else:
        stage = get_current_stage(host)
        stage.set_temp_data("soft_poweroff_timeout", _get_soft_poweroff_timeout(host))
        commit_stage_changes(host, status=_STATUS_SOFT_POWERING_OFF, check_after=_HARDWARE_CHECK_PERIOD)


def _wait_soft_poweroff(host):
    stage = get_current_stage(host)
    timeout = stage.get_temp_data("soft_poweroff_timeout", _POWER_ON_OF_TIMEOUT)

    if stage.timed_out(timeout):
        log.warning("Software power off timeout exceeded for host %s.", host.human_id())
        commit_stage_changes(host, status=_STATUS_POWER_OFF, check_now=True)
    else:
        commit_stage_changes(host, check_after=_HARDWARE_CHECK_PERIOD)


def _handle_hard_poweroff(host):
    ipmi_client = host.get_ipmi_client()

    ipmi_client.power_off()
    commit_stage_changes(host, status=_STATUS_POWERING_OFF, check_after=_HARDWARE_CHECK_PERIOD)


def _wait_hard_poweroff(host):
    stage = get_current_stage(host)

    if stage.timed_out(_POWER_ON_OF_TIMEOUT):
        error = "Failed to power off the host: the host hasn't powered off during power off timeout."
        ipmi_errors.handle_power_on_off_timeout(host, error, _STATUS_POWER_OFF)
    else:
        commit_stage_changes(host, check_after=_HARDWARE_CHECK_PERIOD)


def _get_soft_poweroff_timeout(host):
    reasons = dmc.get_host_reasons(host, checks=CheckType.ALL_AVAILABILITY)

    if not reasons or all(reasons[check]["status"] == CheckStatus.FAILED for check in CheckType.ALL_AVAILABILITY):
        # We do not expect this command to succeed, don't waste too much time.
        return _POWER_ON_OF_TIMEOUT
    else:
        return _SOFT_POWER_OFF_TIMEOUT


#
# Power-on stage handlers
#


def _power_on_handler(host, handler, on_powered_on):
    """Abstract power-on handler."""
    stage = get_current_stage(host)

    if _handle_retry_on_existing_admin_request(host, stage):
        return

    ipmi_client = host.get_ipmi_client()
    if ipmi_client.is_power_on():
        return on_powered_on(host)

    return handler(host)


def _log_power_on_and_wait_post_if_supported(host):
    log.info("Host %s has powered on.", host.human_id())
    stage = get_current_stage(host)

    if stage.get_param("check_post_code", False):
        platform = create_platform_for_host(host)

        if platform.provides_post_code():
            return goto_post_code_check(host)
        else:
            return complete_parent_stage(host, stage)
    else:
        return complete_current_stage(host)


def _retry_if_need_pxe(host):
    stage = get_current_stage(host)

    if stage.get_param("pxe", False):
        error_message = "Got powered on %s host in '%s' status. Power off it first..."
        log.error(error_message, host.human_id(), stage.status)

        if stage.get_param("check_post_code", False):
            # retry parent of power-on-composite to retry whole reboot action
            parent_stage = get_parent_stage(host, stage)
            return retry_action(host, parent_stage)
        else:
            return retry_parent_stage(host, check_after=_HARDWARE_CHECK_PERIOD)
    else:
        _log_power_on_and_wait_post_if_supported(host)


def _handle_power_on(host):
    stage = get_current_stage(host)
    ipmi_client = host.get_ipmi_client()

    pxe = stage.get_param("pxe", False)
    ipmi_client.power_on(pxe=pxe)
    commit_stage_changes(host, status=_STATUS_POWERING_ON, check_after=_HARDWARE_CHECK_PERIOD)


def _wait_power_on(host):
    stage = get_current_stage(host)

    if stage.timed_out(_POWER_ON_OF_TIMEOUT):
        error = "Failed to power on the host: the host hasn't powered on during power on timeout."
        ipmi_errors.handle_power_on_off_timeout(host, error, _STATUS_POWER_ON)
    else:
        commit_stage_changes(host, check_after=_HARDWARE_CHECK_PERIOD)


def _get_power_off_initial_status(stage):
    if stage.get_param("soft", False):
        return _STATUS_SOFT_POWER_OFF
    else:
        return _STATUS_POWER_OFF


def _handle_retry_on_existing_admin_request(host, stage):
    # TODO: It's an experimental feature. Not sure yet whether we need it and how exactly we should handle existing
    # admin requests.
    #
    # TODO: Doesn't covered by tests at all.

    if not config.get_value("hardware.dont_touch_hosts_with_pending_admin_requests"):
        return False

    for request_type in admin_requests.RequestTypes.ALL_IPMI:
        request = admin_requests.get_last_request_status(request_type, host.inv)
        if request is None or request["status"] != admin_requests.STATUS_IN_PROCESS:
            continue

        error = "There is an active '{}' admin request. Waiting when it will be processed.".format(request_type.type)
        log.info("%s: %s", host.human_id(), error)

        ipmi_errors.reset_ipmi_errors(host)

        retry_status_mapping = {
            _STATUS_SOFT_POWERING_OFF: _STATUS_SOFT_POWER_OFF,
            _STATUS_POWERING_OFF: _STATUS_POWER_OFF,
            _STATUS_POWERING_ON: _STATUS_POWER_ON,
        }

        if stage.status in (_STATUS_SOFT_POWER_OFF, _STATUS_POWER_OFF, _STATUS_POWER_ON):
            commit_stage_changes(host, error=error, check_after=walle.fsm_stages.common.ADMIN_REQUEST_CHECK_INTERVAL)
        elif stage.status in retry_status_mapping:
            commit_stage_changes(
                host,
                error=error,
                status=retry_status_mapping[stage.status],
                check_after=walle.fsm_stages.common.ADMIN_REQUEST_CHECK_INTERVAL,
            )
        else:
            raise LogicalError

        return True

    return False


def _on_post_ok(host):
    stage = get_current_stage(host)
    return complete_parent_stage(host, stage)


def wait_post_complete_ipmi(host):
    wait_post_complete(host, _on_post_ok)


# Attention: power on stage with pxe=True parameter may retry parent stage on errors
register_stage(
    Stages.POWER_OFF,
    generate_stage_handler(
        {
            _STATUS_SOFT_POWER_OFF: partial(_power_off_handler, handler=_handle_soft_poweroff),
            _STATUS_SOFT_POWERING_OFF: partial(_power_off_handler, handler=_wait_soft_poweroff),
            _STATUS_POWER_OFF: partial(_power_off_handler, handler=_handle_hard_poweroff),
            _STATUS_POWERING_OFF: partial(_power_off_handler, handler=_wait_hard_poweroff),
        }
    ),
    initial_status=_get_power_off_initial_status,
)

register_stage(
    Stages.POWER_ON,
    generate_stage_handler(
        {
            _STATUS_POWER_ON: partial(_power_on_handler, handler=_handle_power_on, on_powered_on=_retry_if_need_pxe),
            _STATUS_POWERING_ON: partial(
                _power_on_handler, handler=_wait_power_on, on_powered_on=_log_power_on_and_wait_post_if_supported
            ),
            STATUS_WAIT_POST_COMPLETE: wait_post_complete_ipmi,
        }
    ),
    initial_status=_STATUS_POWER_ON,
)
