import logging

from walle.fsm_stages.common import (
    complete_current_stage,
    terminate_current_stage,
    complete_parent_stage,
    get_current_stage,
    commit_stage_changes,
)
from walle.host_platforms.platform import PlatformProblems
from walle.host_platforms.platform_manager import create_platform_for_host
from walle.stages import StageTerminals
from walle.stats import stats_manager

log = logging.getLogger(__name__)


POST_CHECK_PERIOD = 15
"""Host POST status check period"""

POST_CHECK_DELAY = 60
"""Delay before checking POST code for the first time"""

POST_CHECK_TIMEOUT = 240
"""Host POST status check timeout"""

POST_SEEN_STABLE_COUNT = 3
"""POST code should be analyzed if we have seen it more than this times"""

STATUS_WAIT_POST_COMPLETE = "waiting-post-complete"
"""Waiting for POST to complete."""


def goto_post_code_check(host, delay=POST_CHECK_DELAY):
    stats_manager.increment_counter(("hosts", "post_code_check"))
    return commit_stage_changes(host, status=STATUS_WAIT_POST_COMPLETE, check_after=delay)


def handle_post_code(host, stage, platform, code, on_post_ok):
    upgrade_to_profile = stage.get_param("upgrade_to_profile", False)

    if platform.get_post_problem_for_code(code) == PlatformProblems.POST_OK:
        log.info("POST reported OK for host %s completing power-on-composite stage", host.human_id())
        # if post code reported OK complete always parent stage
        return on_post_ok(host)

    elif platform.get_post_problem_for_code(code) == PlatformProblems.POST_MEMORY_PROBLEM:
        stats_manager.increment_counter(("hosts", "post_code_check_memory_problem"))
        if upgrade_to_profile:
            stats_manager.increment_counter(("hosts", "post_code_check_memory_problem_profiles"))
            log.info("POST reported MEMORY PROBLEM for host %s upgrading to profile", host.human_id())
            return terminate_current_stage(StageTerminals.PROFILE, host, "POST reported memory problem, profiling host")
        else:
            log.info("POST reported MEMORY PROBLEM for host %s doing power-cycle", host.human_id())
            return complete_current_stage(host)

    elif platform.get_post_problem_for_code(code) == PlatformProblems.POST_UNKNOWN:
        if upgrade_to_profile:
            log.info("POST reported UNKNOWN PROBLEM for host %s upgrading to profile", host.human_id())
            return terminate_current_stage(
                StageTerminals.PROFILE, host, "POST reported unknown problem, profiling host"
            )
        else:
            log.info("POST reported UNKNOWN PROBLEM for host %s doing power-cycle", host.human_id())
            return complete_current_stage(host)


def wait_post_complete(host, on_post_ok):
    stage = get_current_stage(host)
    platform = create_platform_for_host(host)

    # for now we're checking only current post code
    if stage.timed_out(POST_CHECK_TIMEOUT):
        log.info("Waiting for POST status for host %s timed out. Completing stage.", host.human_id())

        post_code = platform.get_current_post_code(host.get_ipmi_client())

        return handle_post_code(host, stage, platform, post_code, on_post_ok)

    # for now we're completing whole task
    if not platform.provides_post_code():
        log.error(
            "Host %s has generic platform. Platform.board=%s, platform.system=%s Completing stage.",
            host.human_id(),
            platform.board,
            platform.system,
        )
        return complete_parent_stage(host, stage)

    code = platform.get_current_post_code(host.get_ipmi_client())
    log.debug("Got %s POST code for %s", code, platform)

    if not stage.has_temp_data("last_post_code_seen"):
        stage.set_temp_data("last_post_code_seen", code)
        stage.set_temp_data("last_post_code_seen_times", 1)

        log.debug("Set last_post_code_seen for %s to %s, commiting changes", code, platform)

        return commit_stage_changes(host, check_after=POST_CHECK_PERIOD)

    last_code = stage.get_temp_data("last_post_code_seen")
    last_code_seen_times = stage.get_temp_data("last_post_code_seen_times")

    log.debug("Got last_post_code_seen for %s code: %s times: %s", platform, last_code, last_code_seen_times)

    if code == last_code:
        last_code_seen_times += 1
        if last_code_seen_times > POST_SEEN_STABLE_COUNT:
            # post code is stable, we can handle it now
            log.info(
                "POST code %s seen %s > %s times, handling it", last_code, last_code_seen_times, POST_SEEN_STABLE_COUNT
            )

            return handle_post_code(host, stage, platform, code, on_post_ok)
        else:
            stage.set_temp_data("last_post_code_seen_times", last_code_seen_times)
            return commit_stage_changes(host, check_after=POST_CHECK_PERIOD)
    else:
        log.debug("POST code for %s changed since last poll from %s to %s", platform, last_code, code)
        stage.set_temp_data("last_post_code_seen", code)
        stage.set_temp_data("last_post_code_seen_times", 1)
        return commit_stage_changes(host, check_after=POST_CHECK_PERIOD)
