"""Grants host access to EXP teams for diagnostics."""

import logging

from sepelib.core import config

from walle.clients import idm
from walle.fsm_stages.common import register_stage, complete_current_stage
from walle.stages import Stages

log = logging.getLogger(__name__)


def _provide_diagnostic_host_access(host):
    rnd_groups = config.get_value("host_diagnostics_access.groups")

    with idm.BatchRequest() as br:
        for group_name in rnd_groups:
            _request_access_for_group(br, host, group_name)
        br.execute()

    log.info("Successfully requested EXP access to %s for groups %s", host.name, ", ".join(rnd_groups))
    complete_current_stage(host)


def _request_access_for_group(batch, host, group):
    _request_cauth_role(batch, group, host.name, "ssh", role_parameters={"root": False})
    _request_cauth_role(batch, group, host.name, "sudo", role_parameters={"role": "ALL=(ALL) ALL"})


def _request_cauth_role(batch, group_name, host_name, role_name, role_parameters):
    batch.request_role(
        system="cauth",
        group=group_name,
        path=["dst", host_name, "role", role_name],
        fields_data=role_parameters,
        deprive_after_days=config.get_value("host_diagnostics_access.ttl_days"),
    )


register_stage(Stages.PROVIDE_DIAGNOSTIC_HOST_ACCESS, _provide_diagnostic_host_access)
