"""Report is a stage that creates startrek reports about host failures."""

from walle.failure_reports.project_reports import ticket_params_for_project, NoAssigneeException
from walle.failure_reports.startrek import TicketParams
from walle.fsm_stages import startrek_report
from walle.fsm_stages.common import register_stage, get_current_stage
from walle.stages import Stages
from walle.util.text import enumeration_join


class CanNotCreateProjectReports(startrek_report.CanNotCreateReports):
    def __init__(self, project_id, reason):
        super().__init__("Can not create reports for project {}: {}", project_id, reason)


def _handler(host):
    handler = startrek_report.ReportStageHandler(host, _report_getter)
    handler.handle()


def _report_getter(handler):
    project = handler.project
    fqdn = handler.host.name
    task_id = handler.task_id
    check_names = handler.checks
    stage = get_current_stage(handler.host)

    if stage.has_temp_data('ticket_params'):
        ticket_params = TicketParams.from_dict(stage.get_temp_data('ticket_params'))
    else:
        ticket_params = _ticket_params(project, task_id, fqdn, *check_names)
        stage.set_temp_data('ticket_params', ticket_params.to_dict())

    if stage.has_temp_data('report_summary'):
        report_summary = stage.get_temp_data('report_summary')
    else:
        report_summary = _create_summary(ticket_params.get("summary", None), fqdn, check_names)
        stage.set_temp_data('report_summary', report_summary)

    return startrek_report.get_report(ticket_params, report_summary)


def _ticket_params(project, task_id, *tags):
    try:
        tp = ticket_params_for_project(project)
    except NoAssigneeException as e:
        raise CanNotCreateProjectReports(project.id, str(e))

    if tp.get("tags", None) is not None:
        tags = tuple(tags) + tuple(tp.get("tags"))

    return tp.append(tags=tags, unique_salt=task_id)


def _create_summary(summary, fqdn, check_names):
    ticket_summary = "Host {fqdn} suffers from {checks} check{pl} failure".format(
        fqdn=fqdn, checks=enumeration_join(sorted(check_names)), pl="s" if len(check_names) > 1 else ""
    )
    summary_parts = filter(None, [ticket_summary, summary])
    return " | ".join(summary_parts)


# beware! this stage has internal states defined outside this register_stage call
register_stage(Stages.REPORT, _handler)
