"""Sets downtime for the host in Juggler."""

import logging

from sepelib.core.constants import MINUTE_SECONDS
from walle.clients.juggler import JugglerClient, JugglerDowntimeName, JugglerDowntimeTypes
from walle.fsm_stages.common import (
    register_stage,
    commit_stage_changes,
    complete_current_stage,
    get_current_stage,
    retry_current_stage,
    increase_configurable_error_count,
)
from walle.stages import Stages

log = logging.getLogger(__name__)


STAGE_RETRY_INTERVAL = 2 * MINUTE_SECONDS
ERROR_TEMPLATE = "Failed to set downtime in Juggler: {}"


def _set_downtime(host):
    stage = get_current_stage(host)

    if host.name is not None:
        juggler_downtime_name = stage.get_param("juggler_downtime_name", JugglerDowntimeName.DEFAULT)
        dt_params = JugglerDowntimeTypes.by_suffix_name(juggler_downtime_name)
        log.debug("%s: Set downtime in Juggler{}.".format(dt_params.log_message), host.human_id())

        try:
            juggler = JugglerClient()
            previous_downtime_ids = juggler.get_fqdn_to_downtimes_map(fqdn=host.name).get(host.name)
            downtime_id = juggler.set_downtime(
                host.name,
                end_time=dt_params.end_time,
                description="Wall-E is processing '{}' task on the host".format(host.status),
                suffix=dt_params.source_suffix,
            )
            if previous_downtime_ids:
                juggler.remove_downtimes(previous_downtime_ids)
            host.on_downtime = True
            commit_stage_changes(host, extra_fields=["on_downtime"])
        except Exception as e:
            error_message = ERROR_TEMPLATE.format(e)
            log.error("%s: %s", host.human_id(), error_message)
            retry_or_complete_stage(host, stage, error_message)
            return
        else:
            log.info(
                "%s: Successfully set downtime id %s in Juggler{}.".format(dt_params.log_message),
                host.human_id(),
                downtime_id,
            )

    complete_current_stage(host)


def retry_or_complete_stage(host, stage, error_message):
    counter_name = "retries"
    limit_name = "set_downtime.max_retries_count"

    if increase_configurable_error_count(host, stage, counter_name, limit_name, error=error_message, fail_stage=False):
        return retry_current_stage(host, error=error_message, check_after=STAGE_RETRY_INTERVAL)
    return complete_current_stage(host)


register_stage(Stages.SET_DOWNTIME, _set_downtime)
