"""Contains common logic for startrek tasks handling in stages."""

# See stage description on wiki
# https://docs.yandex-team.ru/wall-e/stages/report

from contextlib import contextmanager

from sepelib.core.constants import MINUTE_SECONDS, WEEK_SECONDS, DAY_SECONDS
from sepelib.core.exceptions import Error
from walle.expert import juggler, decisionmakers
from walle.expert.types import WalleAction
from walle.failure_reports import observers
from walle.failure_reports.base import (
    ErrorHostsReport,
    ReportSection,
    HostFailureRotationStrategy,
    StreamKey,
    ReportedHost,
    SectionFormatter,
    ReportFormatter,
)
from walle.failure_reports.reports import ReportContents
from walle.failure_reports.startrek import StarTrekReportPublisher
from walle.fsm_stages.common import (
    get_current_stage,
    commit_stage_changes,
    push_host_ticket,
    fail_current_stage,
    complete_current_stage,
)
from walle.models import timestamp
from walle.util.optional import Either
from walle.clients import dmc

_CHECK_POLLING_PERIOD = 10 * MINUTE_SECONDS

_TICKET_WAIT_TIMEOUT = WEEK_SECONDS
_HEALTH_WAIT_TIMEOUT = DAY_SECONDS

_STATUS_WAITING_HEALTH = "waiting-health"
_STATUS_WAITING_TICKET = "waiting-ticket"
_STATUS_ERROR = "error"


class CanNotCreateReports(Error):
    """Raised when reports can not be used for some obscure reason"""

    pass


class HealthDataMissing(Error):
    """raised when required health data is not present."""

    pass


@contextmanager
def get_report(ticket_params, report_summary, formatter=None):
    with _create_report(ticket_params) as report:
        yield ReportFacade(report_summary, report, formatter)


def _create_report(ticket_params):
    publisher = StarTrekReportPublisher(ticket_params)

    rotation_strategy = HostFailureRotationStrategy(rotate=False)
    stream_key = StreamKey(publisher.get_stream_key(), publisher.name, rotation_strategy.name)

    observers_group = observers.collect_group(stream_key)

    return ErrorHostsReport(stream_key.wrapped_key(), publisher, rotation_strategy, observers_group)


class ReportFacade:
    """Simple facade for reports, only allow to add problem/solved hosts manually."""

    def __init__(self, summary, report, report_formatter=None):
        """
        :type summary: str
        :type report: ErrorHostsReport
        :type report_formatter: ReportFormatter
        """

        section_formatter = SectionFormatter(summary)
        report_section = CheckFailureSection("CheckFailureSection", section_formatter)

        if report_formatter is None:
            report_formatter = ReportFormatter()
        report_content = ReportContents(report_formatter)
        report_content.add_section(report_section)

        report.set_content(summary, report_content)

        self.report = report
        self.report_section = report_section

    def add_host(self, host, failure_reason):
        reported_host = self._reported_host(host, failure_reason)
        self.report_section.add_problem_host(reported_host)

    def remove_host(self, host, failure_reason):
        reported_host = self._reported_host(host, failure_reason)
        self.report_section.add_solved_host(reported_host)

    def get_report_key(self):
        return self.report.get_report_key()

    @staticmethod
    def _reported_host(host, failure_reason):
        stage = get_current_stage(host)
        tickets = stage.get_data("tickets", None)

        return ReportedHost(
            inv=host.inv,
            name=host.name,
            host_uuid=host.uuid,
            status=host.status,
            project=host.project,
            reason=failure_reason,
            tickets=tickets,
            report_timestamp=timestamp(),
        )


class CheckFailureSection(ReportSection):
    def merge_hosts(self, previous_hosts):
        """Merge host from previous report runs with current run.
        Add all hosts from previous run as problem hosts.
        Mark hosts as solved if they have been explicitly marked as such.

        This method is called automatically by report.

        :param previous_hosts - dict of inv -> host data from previous run
        :type previous_hosts: dict
        """

        for inv, host in previous_hosts.items():
            if host.solved:
                self.add_solved_host(host)
            else:
                if inv in self.solved_hosts:
                    continue
                self.add_problem_host(host)


class ReportStageHandler:
    def __init__(self, host, report_getter):
        self.host = host
        self.task_id = host.task.task_id
        self.stage = get_current_stage(host)
        self.checks = self.stage.get_param("checks")
        self.reason = self.stage.get_param("reason")
        self.project = self.host.get_project()

        self.get_report = report_getter

    def handle(self):
        try:
            decision = self._get_decision()

            if decision.action == WalleAction.HEALTHY:
                self._close_report()
                return complete_current_stage(self.host)
            else:
                self._open_report()
                return self._wait_ticket("Waiting for ticket to be processed.")

        except HealthDataMissing as e:
            return self._wait_health(str(e))

        except CanNotCreateReports as e:
            return self._wait_error(error=str(e))

    def _open_report(self):
        with self.get_report(self) as report:
            report.add_host(self.host, self.reason)

        push_host_ticket(self.host, report.get_report_key())

    def _close_report(self):
        with self.get_report(self) as report:
            report.remove_host(self.host, self.reason)

    def _get_decision(self):
        decision_maker = decisionmakers.get_decision_maker(self.project, self.checks)
        failures = juggler.get_host_health_reasons(self.host, decision_maker)
        if not failures:
            raise HealthDataMissing("Health information is not available.")

        _, decision = dmc.get_decisions_from_handler(
            self.host, decision_params=dmc.DecisionParams(self.checks if self.checks else set())
        )
        decision = decision_maker.make_decision(self.host, failures)
        if decision.action == WalleAction.WAIT:
            raise HealthDataMissing("Health information is not complete.")

        return decision

    def _wait_health(self, reason):
        return Either(self._wait(_HEALTH_WAIT_TIMEOUT, _STATUS_WAITING_HEALTH, reason)).or_else(
            lambda: self._fail_task(reason)
        )

    def _wait_ticket(self, reason):
        return Either(
            self._wait(_TICKET_WAIT_TIMEOUT, _STATUS_WAITING_TICKET, reason, extra_fields=["ticket"])
        ).or_else(lambda: self._fail_task("Have been waiting too long for ticket to be processed. Timed out."))

    def _wait_error(self, error):
        return Either(self._wait(_TICKET_WAIT_TIMEOUT, _STATUS_ERROR, error=error)).or_else(
            lambda: self._fail_task(error)
        )

    def _wait(self, timeout, status, reason=None, error=None, extra_fields=None):
        # Wait for a given timeout. Return True if is is going to wait, False otherwise.

        if self.stage.status == status and self.stage.timed_out(timeout):
            return False

        # do not reset stage status every time, resetting status also resets status_time and that affects stage timeout.
        commit_stage_changes(
            status=status if status != self.stage.status else None,
            host=self.host,
            status_message=reason,
            error=error,
            check_after=_CHECK_POLLING_PERIOD,
            extra_fields=extra_fields,
        )
        return True

    def _fail_task(self, reason):
        self._close_report()
        return fail_current_stage(self.host, reason)
