"""Change project ID for task in default CMS"""

import logging

import mongoengine

from sepelib.core.exceptions import LogicalError
from walle.default_cms import CmsTask, CmsProject
from walle.fsm_stages.common import register_stage, complete_current_stage, get_current_stage
from walle.stages import Stages

log = logging.getLogger(__name__)


def _switch_default_cms_project(host):
    stage = get_current_stage(host)
    cms_task_id = stage.get_param("cms_task_id")
    source_project_id = stage.get_param("source_project_id")
    target_project_id = stage.get_param("target_project_id")

    log.info("%s: Switch project for task %s in default CMS.", host.human_id(), cms_task_id)

    query = {"id": cms_task_id, "project_id": source_project_id}

    try:
        cms_task = CmsTask.objects(**query).get()
    except mongoengine.DoesNotExist:
        log.error(
            "%s: CMS task with id %s does not exist for project %s.", host.human_id(), cms_task_id, source_project_id
        )
        return complete_current_stage(host)

    if len(cms_task.hosts) > 1:
        # Cannot switch project for CMS task with more than one host
        raise LogicalError

    query["hosts__size"] = 1
    cms_task.modify(query, project_id=target_project_id)

    CmsProject(id=source_project_id, tasks=cms_task_id).modify(pull__tasks=cms_task_id)
    CmsProject(id=target_project_id).modify(add_to_set__tasks=cms_task_id, upsert=True)

    complete_current_stage(host)


register_stage(Stages.SWITCH_DEFAULT_CMS_PROJECT, _switch_default_cms_project)
