"""Switches a host to another project."""

import logging

import walle.projects
import walle.tasks
from sepelib.core.exceptions import Error
from walle.fsm_stages.common import register_stage, get_current_stage, fail_current_stage, complete_current_stage
from walle.locks import ProjectInterruptableLock
from walle.restrictions import strip_restrictions
from walle.stages import Stages
from walle.util.misc import fix_mongo_set_kwargs

log = logging.getLogger(__name__)


def _switch_project(host):
    stage = get_current_stage(host)
    project_id = stage.get_param("project")
    host_restrictions = stage.get_param("host_restrictions", None)

    with ProjectInterruptableLock(project_id):
        try:
            project = walle.projects.get_by_id(project_id, fields=("logical_datacenter",))
            logical_datacenter = project.logical_datacenter
        except walle.projects.ProjectNotFoundError:
            return fail_current_stage(host, "Project '{}' has been deleted.".format(project_id))

        update_kwargs = dict(
            set__project=project_id, inc__task__revision=1, set__location__logical_datacenter=logical_datacenter
        )
        if host_restrictions is not None:
            update_kwargs.update(
                fix_mongo_set_kwargs(set__restrictions=strip_restrictions(host_restrictions, strip_to_none=True))
            )
        if host.modify(walle.tasks.host_query(host), **update_kwargs):
            return complete_current_stage(host)

    raise Error("Unable to commit host state change: it doesn't have an expected state already.")


register_stage(Stages.SWITCH_PROJECT, _switch_project)
