"""Contains all logic for VLAN switching stage."""

import logging

import walle.projects
from sepelib.core import constants
from sepelib.core.exceptions import LogicalError
from walle import constants as walle_constants
from walle import network
from walle.clients import racktables, vlan_toggler
from walle.constants import NetworkTarget, PROFILING_VLAN, PARKING_VLAN
from walle.errors import InvalidHostConfiguration, NoInformationError, UnauthorizedError
from walle.fsm_stages.common import (
    register_stage,
    generate_stage_handler,
    get_current_stage,
    complete_current_stage,
    commit_stage_changes,
    fail_current_stage,
    terminate_current_stage,
    retry_current_stage,
    DEFAULT_RETRY_TIMEOUT,
    get_parent_stage,
    increase_configurable_error_count,
    get_stage_deploy_configuration,
)
from walle.stages import Stages, StageTerminals

log = logging.getLogger(__name__)


_STATUS_PREPARING = "preparing"
_STATUS_VLAN_AVAILABILITY = "checking-vlan-availability"
_STATUS_CHECKING_VLANS = "checking-vlans"
_STATUS_CHECKING_PROJECT = "checking-project-id"
_STATUS_CHECKING_YC_STATE = "checking-yc-state"
_STATUS_SWITCHING_VLANS = "switching-vlans"
_STATUS_SWITCHING_PROJECT = "switching-project-id"
_STATUS_SWITCHING_YC_STATE = "switching-yc-state"
_STATUS_WAITING_VLANS_SYNC = "waiting-vlans-sync"
_STATUS_WAITING_PROJECT_SYNC = "waiting-project-sync"
_STATUS_WAITING_YC_STATE = "waiting-yc-state"
_STATUS_WAITING_NETWORKS = "waiting-networks"

_SYNC_POLLING_PERIOD = 10
_INCREASED_SYNC_POLLING_PERIOD = 2 * constants.MINUTE_SECONDS
"""Normally check vlans/project sync every 10 seconds, but after waiting too long switch to longer intervals."""

_ULTIMATE_WAIT_TIMEOUT = 2 * constants.DAY_SECONDS
"""If nobody wants it then don't burn the fuel on it. Give up waiting for sync after this timeout."""


_SWITCHING_TIMEOUT = constants.HOUR_SECONDS
_NETWORK_ASSIGNING_TIMEOUT = 3 * constants.HOUR_SECONDS
# the files we check have a very long update intervals:
# l3-tors updates every 10 minutes
# l2-segments updates every hour.
# We don't want to wait too long having a faster option, therefore we have different timeouts for l2 and l3.
_L3_NETWORKS_POLLING_PERIOD = 2 * constants.MINUTE_SECONDS
_L2_NETWORKS_POLLING_PERIOD = 10 * constants.MINUTE_SECONDS

_MINIMUM_SYNC_DELAY = constants.MINUTE_SECONDS
"""Sync routine starts every 60 seconds. When switch is updated, it lands into a queue for the second next run.
So in theory every switch waits at least 60 seconds before sync."""
# Note: In practice, however, we sometime update switches that are already in the queue for the next run.
# For them our modification apply faster. About 30% of our requests sync faster then 60 seconds
# and we wait more the we need to.

_VLAN_TOGGLER_RETRY_DELAY = constants.MINUTE_SECONDS
_VLAN_TOGGLER_RETRY_TIMEOUT = 12 * constants.HOUR_SECONDS


def _handle_preparing(host):
    def _vlan_config_handler(host, stage, vlan_config_data):
        stage.set_temp_data("switched_port_info", vlan_config_data)

        if "target_yc_state" in vlan_config_data:
            return commit_stage_changes(host, status=_STATUS_CHECKING_YC_STATE, check_now=True)
        elif vlan_config_data["mtn_project_id"] and not set(vlan_config_data['vlans']).issubset(
            {PARKING_VLAN, PROFILING_VLAN}
        ):
            return commit_stage_changes(host, status=_STATUS_CHECKING_PROJECT, check_now=True)
        else:
            return commit_stage_changes(host, status=_STATUS_CHECKING_VLANS, check_now=True)

    _handle_stage_config(host, _vlan_config_handler)


def _handle_stage_config(host, vlan_config_handler):
    stage = get_current_stage(host)
    project = host.get_project(fields=("vlan_scheme", "native_vlan", "extra_vlans", "hbf_project_id"))
    need_service_network_for_pxe = host._need_service_network_for_pxe()

    if not project.vlan_scheme and not stage.has_param("vlans"):
        log.info("%s: project is not configured for VLAN auto-configuration.", host.human_id())
        return complete_stage(host)

    # first check that we know host location at all.
    try:
        network_location = network.get_current_host_switch_port(host)
    except NoInformationError:
        message = "Can't switch vlans: host's switch is unknown."
        if project.vlan_scheme == walle_constants.VLAN_SCHEME_CLOUD:
            return retry_current_stage(host, error=str(message), check_after=DEFAULT_RETRY_TIMEOUT)
        return terminate_current_stage(StageTerminals.SWITCH_MISSING, host, message)

    switch, port = network_location.switch, network_location.port

    if project.vlan_scheme == walle_constants.VLAN_SCHEME_CLOUD:
        if not stage.has_param("network"):
            return fail_current_stage(
                host, "Project with Yandex.Cloud's VLAN schema is not supported switch to explicit VLAN"
            )
        vlan_config_data = {
            "switch": switch,
            "port": port,
            "target_yc_state": network.get_yc_state_from_target(stage.get_param("network")),
        }
        return vlan_config_handler(host, stage, vlan_config_data)

    try:
        vlan_config = network.get_host_expected_vlans(host, project=project)
    except InvalidHostConfiguration as e:
        # Infinite retry. If we ever need to limit it, will do it inside retry_current_stage function.
        return retry_current_stage(host, error=str(e), check_after=DEFAULT_RETRY_TIMEOUT)

    if vlan_config is None:
        log.info("%s: project is not configured for VLAN auto-configuration.", host.human_id())
        return complete_stage(host)

    mtn_prj_id = vlan_config.mtn_project_id

    if need_service_network_for_pxe and stage.get_param("network") == NetworkTarget.DEPLOY:
        vlans, native_vlan = [PROFILING_VLAN], PROFILING_VLAN
        mtn_prj_id = None

    elif mtn_prj_id is not None:
        vlans, native_vlan = vlan_config.vlans, vlan_config.native_vlan

    elif stage.has_param("vlans"):
        vlans, native_vlan = stage.get_param("vlans"), stage.get_param("native_vlan", None)

    else:
        target_network = stage.get_param("network")
        if target_network == NetworkTarget.DEPLOY:
            # this is only valid in deploy stages, which strongly rely on presence of parent stages.
            configuration = get_stage_deploy_configuration(get_parent_stage(host))
            target_network = configuration.network

        if target_network == NetworkTarget.SERVICE:
            vlans, native_vlan = [PROFILING_VLAN], PROFILING_VLAN
        elif target_network == NetworkTarget.PARKING:
            vlans, native_vlan = [PARKING_VLAN], PARKING_VLAN
        elif target_network == NetworkTarget.PROJECT:
            vlans, native_vlan = vlan_config.vlans, vlan_config.native_vlan

            extra_vlans = stage.get_param("extra_vlans", [])
            if extra_vlans:
                vlans = sorted(set(vlans) | set(extra_vlans))
        else:
            raise LogicalError

    try:
        walle.projects.authorize_vlans(host.project, vlans)
    except UnauthorizedError as e:
        # either task was created manually or vlans was removed from project's owned vlans but left in vlan scheme.
        # either way, task shall be terminated, not retried.
        # Shall we mark host as dead, though?
        return fail_current_stage(
            host, "Failed to switch the host to {}: {}".format(_get_vlans_operation_description(vlans, native_vlan), e)
        )

    vlan_config_data = {
        "switch": switch,
        "port": port,
        "vlans": vlans,
        "native_vlan": native_vlan,
        "mtn_project_id": mtn_prj_id,
    }

    return vlan_config_handler(host, stage, vlan_config_data)


def _handle_checking_mtn_project(host):
    stage = get_current_stage(host)
    info = stage.get_temp_data("switched_port_info")

    switch = info["switch"]
    port = info["port"]
    mtn_project_id = info["mtn_project_id"]

    description = _get_switch_operation_description(info)
    try:
        current_project_id, synced = _get_port_project_status(switch, port, mtn_project_id is not None)
    except racktables.PersistentRacktablesError as e:
        message = "Can not switch {}: {}".format(description, e)
        return terminate_current_stage(StageTerminals.SWITCH_MISSING, host, message)

    if current_project_id == mtn_project_id:
        if synced:
            return _proceed_to_vlans(host, stage)
        else:
            return commit_stage_changes(host, status=_STATUS_WAITING_PROJECT_SYNC, check_after=_SYNC_POLLING_PERIOD)
    else:
        stage.set_temp_data("current_project_id", current_project_id)
        return commit_stage_changes(host, status=_STATUS_SWITCHING_PROJECT, check_now=True)


def _handle_checking_yc_state(host):
    stage = get_current_stage(host)
    info = stage.get_temp_data("switched_port_info")

    switch = info["switch"]
    port = info["port"]
    target = info["target_yc_state"]

    try:
        resp = vlan_toggler.get_port_state(switch, port)
    except vlan_toggler.VlanTogglerError as e:
        message = "Can not switch {}: {}".format(_get_switch_operation_description(info), e)
        if isinstance(e, vlan_toggler.VlanTogglerPersistentError):
            return fail_current_stage(host, message)
        return _yc_retry_or_fail_stage(host, stage, message)

    if resp.state == target:
        complete_stage(host)
    else:
        commit_stage_changes(host, status=_STATUS_SWITCHING_YC_STATE, check_now=True)


def _get_port_project_status(switch, port, mtn_support_required):
    try:
        current_project_id, synced = racktables.get_port_project_status(switch, port)
    except racktables.MtnNotSupportedForSwitchError:
        if mtn_support_required:
            # this branch should raise the exception further, but we've got some weird issues
            # when this error was returned for switches that certainly supported mtn,
            # so we ignore the error until issue resolved.
            current_project_id, synced = None, True
        else:
            current_project_id, synced = None, True

    return current_project_id, synced


def _handle_switching_mtn_project(host):
    stage = get_current_stage(host)
    info = stage.get_temp_data("switched_port_info")
    switch, port, mtn_project_id = info["switch"], info["port"], info["mtn_project_id"]
    current_project_id = stage.get_temp_data("current_project_id", None)
    description = _get_switch_operation_description(info)

    try:
        if mtn_project_id:
            racktables.switch_mtn_project(switch, port, mtn_project_id)
        elif current_project_id:
            racktables.delete_mtn_project(switch, port, current_project_id)
    except racktables.PersistentRacktablesError as e:
        # hope the error is because we failed to determine correct switch.
        # see description in _handle_check
        message = "Can not switch {}: {}".format(description, e)
        return terminate_current_stage(StageTerminals.SWITCH_MISSING, host, message)

    commit_stage_changes(host, status=_STATUS_WAITING_PROJECT_SYNC, check_after=_SYNC_POLLING_PERIOD)


def _handle_switching_yc_state(host):
    stage = get_current_stage(host)
    info = stage.get_temp_data("switched_port_info")

    switch = info["switch"]
    port = info["port"]
    target = info["target_yc_state"]

    try:
        resp = vlan_toggler.switch_port_state(switch, port, target)
        if resp.state != target:
            raise vlan_toggler.VlanTogglerPersistentError("Returned state '{}'".format(resp.state))
    except vlan_toggler.VlanTogglerError as e:
        message = "Can not switch {}: {}".format(_get_switch_operation_description(info), e)
        if isinstance(e, vlan_toggler.VlanTogglerPersistentError):
            return fail_current_stage(host, message)
        return _yc_retry_or_fail_stage(host, stage, message)

    commit_stage_changes(host, status=_STATUS_WAITING_YC_STATE, check_after=_MINIMUM_SYNC_DELAY)


def _handle_waiting_project_sync(host):
    stage = get_current_stage(host)
    info = stage.get_temp_data("switched_port_info")
    switch, port, mtn_project_id = info["switch"], info["port"], info["mtn_project_id"]

    description = _get_switch_operation_description(info)
    try:
        current_project_id, synced = racktables.get_port_project_status(switch, port)
    except racktables.PersistentRacktablesError as e:
        # We messed up, but we can fix it, right? Wait until fix is released.
        return retry_current_stage(
            host, "Failed to switch {}: {}".format(description, e), check_after=DEFAULT_RETRY_TIMEOUT
        )

    if current_project_id != mtn_project_id:
        error_msg = "Failed to switch {}: port changed it's project to unexpected value '{}'.".format(
            description, current_project_id
        )
        return retry_or_fail_stage(host, stage, error_msg)

    if synced:
        return _proceed_to_vlans(host, stage)

    timeout_error_message = _timeout_message(action="MTN project switching process", operation=description)
    if not _wait_timeout(host, stage, timeout_error_message, _SWITCHING_TIMEOUT):
        return commit_stage_changes(host, check_after=_SYNC_POLLING_PERIOD)


def _handle_waiting_yc_state(host):
    stage = get_current_stage(host)
    info = stage.get_temp_data("switched_port_info")

    switch = info["switch"]
    port = info["port"]
    target = info["target_yc_state"]

    try:
        resp = vlan_toggler.get_port_state(switch, port)
    except vlan_toggler.VlanTogglerError as e:
        message = "Can not switch {}: {}".format(_get_switch_operation_description(info), e)
        if isinstance(e, vlan_toggler.VlanTogglerPersistentError):
            return fail_current_stage(host, message)
        return _yc_retry_or_fail_stage(host, stage, message)

    if resp.state == target:
        return complete_stage(host)

    timeout_error_message = _timeout_message(
        action="YC state switching process",
        operation=_get_switch_operation_description(info),
        component="vlan toggler",
        email="cloud-netinfra@yandex-team.ru",
    )
    if not _wait_timeout(host, stage, timeout_error_message, _SWITCHING_TIMEOUT):
        return commit_stage_changes(host, check_after=_SYNC_POLLING_PERIOD)


def _handle_checking_vlans(host):
    stage = get_current_stage(host)
    info = stage.get_temp_data("switched_port_info")

    switch, port, vlans, native_vlan = info["switch"], info["port"], info["vlans"], info["native_vlan"]
    description = _get_switch_operation_description(info)
    log.info("%s: Switching %s...", host.human_id(), description)

    try:
        current_vlans, current_native_vlan, synced = racktables.get_port_vlan_status(switch, port)
    except racktables.PersistentRacktablesError as e:
        # We can't distinguish persistent RackTables errors now but it looks like all persistent errors are caused
        # by an attempt to switch VLAN on unexisting switch or assign a VLAN where it can't be assigned. Both of
        # these errors may be caused by our fail in determining the actual host switch. So we use the
        # "SWITCH_MISSING" stage terminal to enable host profiling task to ignore all persistent RackTables errors
        # in a hope that this exact error is caused by our outdated switch info
        # and by ignoring this error we allow host profiling operation to complete and give us the actual
        # information about host switch.
        message = "Can not switch {}: {}".format(description, e)
        return terminate_current_stage(StageTerminals.SWITCH_MISSING, host, message)

    if current_native_vlan == native_vlan and current_vlans == vlans:
        if synced:
            commit_stage_changes(host, status=_STATUS_WAITING_NETWORKS, check_now=True)
        else:
            commit_stage_changes(host, status=_STATUS_WAITING_VLANS_SYNC, check_after=_SYNC_POLLING_PERIOD)
    else:
        commit_stage_changes(host, status=_STATUS_VLAN_AVAILABILITY, check_now=True)


def _handle_vlan_availability(host):
    stage = get_current_stage(host)
    if stage.get_param("network", None) in NetworkTarget.AVAILABLE:
        return commit_stage_changes(host, status=_STATUS_SWITCHING_VLANS, check_now=True)

    info = stage.get_temp_data("switched_port_info")
    switch, vlans = info["switch"], info["vlans"]
    log.info("%s: Checking vlan availability for switch %s...", host.human_id(), switch)
    for vlan in vlans:
        if not racktables.is_vlan_available(switch, vlan):
            message = (
                "VLAN {} is not available in the domain of {} switch. "
                "This is most probably an error in network configuration. "
                "Please, contact with nocrequests@yandex-team.ru to find out "
                "why VLAN is not available on the switch "
                "(vlan is not present in net-layout.xml).".format(vlan, switch)
            )
            # Infinite retry. If we ever need to limit it, will do it inside retry_current_stage function.
            return retry_current_stage(host, error=message, check_after=DEFAULT_RETRY_TIMEOUT)

    commit_stage_changes(host, status=_STATUS_SWITCHING_VLANS, check_now=True)


def _handle_switching_vlans(host):
    stage = get_current_stage(host)
    info = stage.get_temp_data("switched_port_info")
    switch, port, vlans, native_vlan = info["switch"], info["port"], info["vlans"], info["native_vlan"]
    description = _get_switch_operation_description(info)

    try:
        racktables.switch_vlans(switch, port, vlans, native_vlan)
    except racktables.PersistentRacktablesError as e:
        # hope the error is because we failed to determine correct switch.
        # see description in _handle_check
        message = "Can not switch {}: {}".format(description, e)
        return terminate_current_stage(StageTerminals.SWITCH_MISSING, host, message)

    commit_stage_changes(host, status=_STATUS_WAITING_VLANS_SYNC, check_after=_MINIMUM_SYNC_DELAY)


def _handle_waiting_vlans_sync(host):
    stage = get_current_stage(host)
    info = stage.get_temp_data("switched_port_info")
    switch, port, vlans, native_vlan = info["switch"], info["port"], info["vlans"], info["native_vlan"]
    description = _get_switch_operation_description(info)

    try:
        current_vlans, current_native_vlan, synced = racktables.get_port_vlan_status(switch, port)
    except racktables.PersistentRacktablesError as e:
        # We messed up, but we can fix it, right? Wait until fix is released.
        return retry_current_stage(
            host, error="Failed to switch {}: {}".format(description, e), check_after=DEFAULT_RETRY_TIMEOUT
        )

    if current_vlans != vlans or current_native_vlan != native_vlan:
        error = "Failed to switch {}: the port has suddenly changed it's assigned VLANs to {}.".format(
            description, _get_vlans_operation_description(current_vlans, current_native_vlan)
        )

        return retry_or_fail_stage(host, stage, error)

    if synced:
        return commit_stage_changes(host, status=_STATUS_WAITING_NETWORKS, check_now=True)

    timeout_error_message = _timeout_message(action="VLAN switching process", operation=description)
    if not _wait_timeout(host, stage, timeout_error_message, _SWITCHING_TIMEOUT):
        return commit_stage_changes(host, check_after=_SYNC_POLLING_PERIOD)


def _handle_waiting_networks(host):
    # Networks are allocated for each (switch, VLAN) pair.
    # Network is allocated and assigned to switch when VLAN is assigned to switch the first time.
    # Network allocation/assigning is an asynchronous process so we must wait it to be sure that host will get its
    # prefix via RA and we won't break the next stages that may rely on assigned networks (address allocation for
    # example).

    stage = get_current_stage(host)
    if stage.get_param("network", None) in NetworkTarget.AVAILABLE:
        return _proceed_to_project_id(host, stage)

    info = stage.get_temp_data("switched_port_info")
    switch, vlans, = (
        info["switch"],
        info["vlans"],
    )
    assigned_vlans = stage.setdefault_temp_data("assigned_vlans", [])
    pending_vlans = set(vlans) - set(assigned_vlans)

    for vlan in list(pending_vlans):
        try:
            vlan_network = racktables.get_vlan_networks(switch, vlan)
        except racktables.RacktablesError as e:
            return retry_current_stage(
                host, error="Failed to get vlan networks: {}".format(e), check_after=DEFAULT_RETRY_TIMEOUT
            )
        if vlan_network is not None:
            assigned_vlans.append(vlan)
            pending_vlans.remove(vlan)

    if not pending_vlans:
        return _proceed_to_project_id(host, stage)

    # This may return false negative for l3 switch without any networks. That is ok, it just increases polling interval.
    try:
        is_l3_switch = racktables.is_l3_switch(switch)
    except racktables.RacktablesError as e:
        return retry_current_stage(
            host, error="Failed to check switch {}: {}".format(switch, e), check_after=DEFAULT_RETRY_TIMEOUT
        )
    polling_period = _L3_NETWORKS_POLLING_PERIOD if is_l3_switch else _L2_NETWORKS_POLLING_PERIOD

    # it's a three hours timeout, but the files update asynchronously, so let it wait for one more polling period.
    assign_timeout = _NETWORK_ASSIGNING_TIMEOUT + polling_period
    timeout_error_message = _timeout_message(
        action="Network assigning for {} VLANs".format(", ".join(str(vlan) for vlan in pending_vlans)),
        operation=_get_switch_operation_description(info),
    )
    if not _wait_timeout(host, stage, timeout_error_message, assign_timeout, increased_period=polling_period):
        return commit_stage_changes(host, check_after=polling_period)


def _handle_racktables_error(host):
    """Check stage running circumstances.
    Return True if stage can be handled further with the default handler, false if not.
    """

    stage = get_current_stage(host)
    if stage.status == _STATUS_PREPARING:
        return True  # proceed with the default handler

    def _vlan_config_handler(host, stage, vlan_config_data):
        """
        Check if network configuration has changed.
        Return True if FSM should proceed with default stage handler (i.e. config has not changed).
        """
        if vlan_config_data == stage.get_temp_data("switched_port_info"):
            return True  # proceed with the default stage handler

        return retry_current_stage(host, check_now=True)

    return _handle_stage_config(host, _vlan_config_handler)


def _proceed_to_vlans(host, stage):
    stage.set_temp_data("project_id_synced", True)

    if stage.get_temp_data("vlans_synced", False):
        complete_stage(host)
    else:
        commit_stage_changes(host, status=_STATUS_CHECKING_VLANS, check_now=True)


def _proceed_to_project_id(host, stage):
    stage.set_temp_data("vlans_synced", True)

    if stage.get_temp_data("project_id_synced", False):
        complete_stage(host)
    else:
        commit_stage_changes(host, status=_STATUS_CHECKING_PROJECT, check_now=True)


def _get_switch_operation_description(info):
    port = info["port"]
    switch = info["switch"]

    if info.get("mtn_project_id", None):
        return "'{}' port of '{}' switch to '{}' mtn project".format(port, switch, info["mtn_project_id"])

    if "target_yc_state" in info:
        return "'{}' port of '{}' switch to '{}' state".format(port, switch, info["target_yc_state"])

    description = _get_vlans_operation_description(info["vlans"], info["native_vlan"])
    return "'{}' port of '{}' switch to {}".format(port, switch, description)


def _get_vlans_operation_description(vlans, native_vlan=None):
    return "{} VLANs {}".format(
        ",".join(str(vlan) for vlan in vlans),
        "without native VLAN" if native_vlan is None else "with native VLAN set to {}".format(native_vlan),
    )


def retry_or_fail_stage(host, stage, error_message):
    counter_name = "retries"
    limit_name = "vlan_switching.max_errors"

    if increase_configurable_error_count(host, stage, counter_name, limit_name, error=error_message, fail_stage=True):
        return retry_current_stage(host, error_message + " Retrying.", check_after=DEFAULT_RETRY_TIMEOUT)


def complete_stage(host):
    get_current_stage(host).set_data('vlan_success', True)
    complete_current_stage(host)


def _wait_timeout(host, stage, error, timeout, increased_period=_INCREASED_SYNC_POLLING_PERIOD):
    if stage.timed_out(_ULTIMATE_WAIT_TIMEOUT):
        fail_current_stage(host, error=error)
        return True

    if stage.timed_out(timeout):
        commit_stage_changes(host, error=error, check_after=increased_period)
        return True

    return False


def _yc_retry_or_fail_stage(host, stage, error):
    if stage.timed_out(_VLAN_TOGGLER_RETRY_TIMEOUT):
        fail_current_stage(host, error=error)
    else:
        retry_current_stage(host, error=error + " Retrying.", check_after=_VLAN_TOGGLER_RETRY_DELAY)


def _timeout_message(action, operation, component="racktables", email="nocdev-bugs@yandex-team.ru"):
    return (
        "{action} is taking too long. "
        "It is probably a failure on the {component}'s side. "
        "Please, contact with {email}. "
        "Operation is switching {operation}".format(
            action=action, operation=operation, component=component, email=email
        )
    )


register_stage(
    Stages.SWITCH_VLANS,
    generate_stage_handler(
        {
            _STATUS_PREPARING: _handle_preparing,
            _STATUS_CHECKING_VLANS: _handle_checking_vlans,
            _STATUS_CHECKING_PROJECT: _handle_checking_mtn_project,
            _STATUS_CHECKING_YC_STATE: _handle_checking_yc_state,
            _STATUS_VLAN_AVAILABILITY: _handle_vlan_availability,
            _STATUS_SWITCHING_VLANS: _handle_switching_vlans,
            _STATUS_SWITCHING_PROJECT: _handle_switching_mtn_project,
            _STATUS_SWITCHING_YC_STATE: _handle_switching_yc_state,
            _STATUS_WAITING_VLANS_SYNC: _handle_waiting_vlans_sync,
            _STATUS_WAITING_PROJECT_SYNC: _handle_waiting_project_sync,
            _STATUS_WAITING_YC_STATE: _handle_waiting_yc_state,
            _STATUS_WAITING_NETWORKS: _handle_waiting_networks,
        }
    ),
    error_handler=_handle_racktables_error,
    initial_status=_STATUS_PREPARING,
)
