"""Controls host finite-state machine (local or remote)."""

import logging

from walle.constants import DEFAULT_TIER_ID
from walle.util import mongo
from walle.util.cloud_tools import get_tier

log = logging.getLogger(__name__)


SHARDS_NUM_PATH = "task_processing.shards_num"
MAX_CONCURRENCY = "task_processing.max_concurrency"

_FSM = None
"""Host FSM instance."""


def start(settings):
    """Starts Host FSM.

    Function is not thread-safe.
    """

    global _FSM
    if _FSM is not None:
        return

    # Import FSM logic only if we are going to start local FSM
    from walle.host_fsm.fsm import HostFsm

    tier_id = DEFAULT_TIER_ID
    try:
        tier_id = get_tier()
    except Exception as e:
        log.error(
            "Couldn't get tier for fsm instance, default value will be used: %s. Unexpected error happened: %s",
            e,
            DEFAULT_TIER_ID,
        )

    partitioner = mongo.MongoPartitionerService(f"hosts-fsm-tier-{tier_id}")
    fsm = HostFsm(settings, SHARDS_NUM_PATH, MAX_CONCURRENCY, partitioner)
    fsm.start()

    _FSM = fsm


def stop():
    """Stops Host FSM.

    Function is not thread-safe.
    """

    global _FSM

    fsm, _FSM = _FSM, None
    if fsm:
        fsm.stop()
