"""Global tasks for FSM handbrake."""
import logging

from walle import audit_log
from walle.application import app
from walle.authorization import ISSUER_WALLE
from walle.models import timestamp
from walle.projects import Project
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import drop_none

log = logging.getLogger(__name__)


def _gc_global_fsm_handbrake():
    settings = app.settings()
    if settings.fsm_handbrake and settings.fsm_handbrake.timeout_time < timestamp():
        log.info("Resetting expired global fsm handbrake.")
        if not unset_fsm_handbrake(ISSUER_WALLE, settings, "FSM handbrake expired"):
            log.error("Failed to reset expired global fsm handbrake for, probably raced out.")


def _gc_project_fsm_handbrake():
    """Find projects with expired handbrake and re-enable fsm for them."""
    projects_qs = Project.objects(fsm_handbrake__timeout_time__lt=timestamp()).only("id", "fsm_handbrake")

    for project in gevent_idle_iter(projects_qs):
        log.info(
            "Resetting expired fsm handbrake for project %s issued by %s.", project.id, project.fsm_handbrake.issuer
        )

        if not unset_fsm_handbrake(ISSUER_WALLE, project, "FSM handbrake expired"):
            log.info("Failed to reset expired fsm handbrake for project %s, probably raced out.", project.id)


def unset_fsm_handbrake(issuer, scope_obj, reason):
    original_audit_log_id = scope_obj.fsm_handbrake.audit_log_id

    if scope_obj.modify({"fsm_handbrake__audit_log_id": original_audit_log_id}, unset__fsm_handbrake=True):
        payload = drop_none(
            {
                "unset_issuer": issuer,
                "unset_reason": reason,
            }
        )
        audit_log.complete_request(original_audit_log_id, extra_payload=payload, check_updated=False)
        return True
    else:
        # try to cancel if not finished yet, ignore errors (we are probably racing with other instance).
        audit_log.cancel_request(
            original_audit_log_id,
            check_updated=False,
            error="Unset fsm failed, raced out. Cancelling audit log just in case.",
        )
        return False
