"""Collects information about hosts' MAC addresses."""

import logging

from mongoengine import LongField, StringField, ListField

from sepelib.core import constants
from sepelib.core.exceptions import LogicalError
from walle.models import Document, register_model, timestamp
from walle.util.gevent_tools import gevent_idle_iter

log = logging.getLogger(__name__)

_INFO_EXPIRE_TIME = constants.DAY_SECONDS + constants.HOUR_SECONDS


@register_model
class HostMacs(Document):
    id = StringField(primary_key=True, required=True, help_text="Host ID")
    name = StringField(required=True, help_text="Host name")
    macs = ListField(StringField(), required=True, help_text="Host's MAC addresses")
    first_time = LongField(required=True, help_text="Time when the host has been first seen with these MACs")
    last_time = LongField(required=True, help_text="Time when the host has been seen with these MACs last time")

    meta = {
        "collection": "host_macs",
        "indexes": [{"name": "last_time", "fields": ["last_time"]}, {"name": "host_name", "fields": ["name"]}],
    }


def save_macs_info(name, macs):
    if not macs:
        raise LogicalError()

    macs = sorted(macs)
    host_id = name + ":" + "|".join(mac.replace(":", "") for mac in macs)
    info_time = timestamp()

    HostMacs.objects(id=host_id, name=name, macs=macs).update(
        set_on_insert__first_time=info_time, set__last_time=info_time, upsert=True
    )


def host_macs_gc():
    for host_macs in gevent_idle_iter(HostMacs.objects(last_time__lte=timestamp() - _INFO_EXPIRE_TIME)):
        host_macs.delete()
