"""Host network configuration."""

import logging

import mongoengine
from mongoengine import StringField, LongField, ListField

from sepelib.mongo.util import register_model
from walle import constants as walle_constants
from walle.errors import HostNetworkNotFoundError
from walle.models import Document
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.mongo import SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)


@register_model
class HostNetwork(Document):
    """Represents a host network information."""

    uuid = StringField(primary_key=True, required=True)

    active_mac = StringField(help_text="Host's active MAC")
    active_mac_time = LongField(help_text="Time when the active MAC has been learned")
    active_mac_source = StringField(
        choices=walle_constants.MAC_SOURCES, help_text="Source name the active MAC has been learned from"
    )
    ips = ListField(StringField(), help_text="Host's IP addresses")
    ips_time = LongField(help_text="Time when ips has been learned")

    network_switch = StringField(help_text="Switch name")
    network_port = StringField(help_text="Port name")
    network_timestamp = LongField(help_text="Network location updated at")
    network_source = StringField(choices=walle_constants.NETWORK_SOURCES, help_text="Source of network info")

    @classmethod
    def get_or_create(cls, uuid):
        try:
            return HostNetwork.objects(uuid=uuid).get()
        except mongoengine.DoesNotExist:
            return HostNetwork(uuid=uuid).save(force_insert=True)

    @classmethod
    def get_by_uuid(cls, uuid, fields=None):
        """Returns a host network by host's uuid."""

        objects = HostNetwork.objects(uuid=uuid)
        if fields is not None:
            objects = objects.only(*fields)
        try:
            return objects.get()
        except mongoengine.DoesNotExist:
            raise HostNetworkNotFoundError()

    api_fields = (
        "uuid",
        "active_mac",
        "active_mac_time",
        "active_mac_source",
        "ips",
        "ips_time",
        "network_switch",
        "network_port",
        "network_timestamp",
        "network_source",
    )


def _get_with_query(fields, **query):
    objects = HostNetwork.objects(**query)
    if fields is not None:
        objects = objects.only(*fields)

    try:
        return objects.get()
    except mongoengine.DoesNotExist:
        raise HostNetworkNotFoundError(query=query)


def get_host_network_map():
    host_network_map = dict()
    for host_network in gevent_idle_iter(HostNetwork.objects(read_preference=SECONDARY_LOCAL_DC_PREFERRED)):
        host_network_map[host_network.uuid] = host_network
    return host_network_map
