"""Operations on hosts."""

import logging
from contextlib import contextmanager

import attr
import mongoengine
from mongoengine import Q

import walle.host_fsm.control
import walle.host_status
import walle.projects
import walle.util.api
from sepelib.core.exceptions import Error, LogicalError
from walle import audit_log, constants as walle_constants, network, authorization
from walle.clients import deploy, inventory, bot
from walle.clients.juggler import exception_monitor
from walle.constants import HostType, HOST_TYPES_ALLOWED_WITHOUT_IPMI
from walle.errors import (
    BadRequestError,
    ResourceAlreadyExistsError,
    RequestValidationError,
    NoInformationError,
    InvalidHostNameError,
    ResourceConflictError,
    InconsistentBotProjectId,
    InvalidHostStateError,
    BotEmptyNameError,
    InvalidHostConfiguration,
)
from walle.host_network import HostNetwork
from walle.hosts import (
    HostState,
    HostStatus,
    get_host_query,
    HostLocation,
    Host,
    HostPlatform,
    StateExpire,
    HostOperationState,
    get_host_human_id,
    generate_external_inventory_number,
)
from walle.locks import ProjectInterruptableLock, HostInterruptableLock
from walle.models import timestamp
from walle.network import BlockedHostName, get_free_host_name_template
from walle.physical_location_tree import NoShortNameError, get_shortname, LocationNamesMap
from walle.projects import get_by_id
from walle.restrictions import strip_restrictions
from walle.tasks import schedule_add_host
from walle.util import net, notifications
from walle.util.misc import drop_none, dummy_context, is_not_none_arg_validator, LocationSegment
from walle.util.misc import fix_mongo_set_kwargs
from walle.views.helpers.validators import prepare_maintenance_params

log = logging.getLogger(__name__)


def _ensure_host_doesnt_exist(inv=None, name=None) -> bool:
    if not (name or inv):
        raise LogicalError()

    query = Q()
    if inv is not None:
        query |= Q(inv=inv)
    if name is not None:
        query |= Q(name__iexact=name)

    host = Host.objects(query).only("inv", "name", "project", "type").first()
    if host:
        if host.type != HostType.SHADOW_SERVER:
            raise ResourceAlreadyExistsError("Host {} already exists in {} project.", host.human_id(), host.project)
        return True
    return False


def check_host_not_missing_from_preorder(inv):
    # Some people do not collect hosts from preorder but just pass them to Wall-E instead.
    # This leads to some weird behaviour and is just not the desirable usage.
    # We'd prefer users to add preorder to wall-e or at least collect hosts before adding them.
    preordered_hosts = bot.missed_preordered_hosts()

    if inv in preordered_hosts:
        host = preordered_hosts[inv]
        raise ResourceConflictError(
            "Host #{} (name: {}) has not been collected from preorder #{}. "
            "You can add preorder to Wall-E or you can collect it in BOT.",
            inv,
            host.get("fqdn") or "<no name>",
            host["order"],
        )


def _exists_in_dns(fqdn):
    try:
        net.get_host_ips(fqdn)
    except InvalidHostNameError:
        return False
    else:
        return True


def _is_automated_action(issuer):
    return issuer == authorization.ISSUER_WALLE


def check_bot_project_id(host, project, bot_project_id):
    if bot_project_id is None:
        # Host without bot project id does not have owners.
        # It is an incorrect situation, we need to handle every case we meet.
        # People should bring these hosts to WALLESUPPORT.
        raise bot.BotProjectIdNotFound(
            "Host {} does not have bot project id, Wall-E can not check ownership.", host.human_id()
        )

    error_message = None
    if bot_project_id != project.bot_project_id:
        name_or_inv = host.name if host.name else host.inv
        error_message = "BOT project id of host {} ({}) doesn't match its project BOT project id {}".format(
            name_or_inv, bot_project_id, project.bot_project_id
        )
        if project.validate_bot_project_id:
            raise InconsistentBotProjectId(error_message)

    return error_message


def set_host_location(host, host_network, location, logical_datacenter=None):
    host.location = _make_host_location(location, logical_datacenter)

    try:
        network_location = network.get_current_host_switch_port(host)
    except NoInformationError:
        pass
    except Exception as e:
        log.exception("%s: Failed to determine network location for adding host: %s", host.human_id(), e)
    else:
        host.location.switch = network_location.switch
        host.location.port = network_location.port
        host.location.network_source = network_location.source
        host_network.network_switch = network_location.switch
        host_network.network_port = network_location.port
        host_network.network_source = network_location.source
        host_network.network_timestamp = network_location.timestamp


def _make_host_location(location, logical_datacenter=None):
    location_lookup = LocationNamesMap.get_map()
    with exception_monitor(
        "no-physical-locations-short-name",
        err_msg_tmpl="Could not create host location: {exc}",
        exc_classes=[NoShortNameError],
        reraise=True,
    ):
        location = HostLocation(
            country=location.country,
            city=location.city,
            datacenter=location.datacenter,
            queue=location.queue,
            rack=location.rack,
            unit=location.unit,
            logical_datacenter=logical_datacenter,
            short_datacenter_name=get_shortname(
                location_lookup, location, LocationSegment.DATACENTER, logical_datacenter=logical_datacenter
            ),
            short_queue_name=get_shortname(
                location_lookup, location, LocationSegment.QUEUE, logical_datacenter=logical_datacenter
            ),
            physical_timestamp=timestamp(),
        )

    return location


@contextmanager
def _forced_host_status(host, issuer, status, reason):
    if status is not None:
        with audit_log.on_force_host_status(
            issuer, host.project, host.inv, host.name, host.uuid, status, reason=reason, scenario_id=host.scenario_id
        ) as log_entry:
            yield log_entry
    else:
        yield None


@attr.s
class AddHostArgs:
    issuer = attr.ib(validator=[is_not_none_arg_validator])
    project = attr.ib(validator=[is_not_none_arg_validator])
    inv = attr.ib(default=None)
    name = attr.ib(default=None)
    state = attr.ib(default=HostState.ASSIGNED)
    status = attr.ib(default=HostStatus.READY)
    provisioner = attr.ib(default=None)
    config = attr.ib(default=None)
    deploy_config_policy = attr.ib(default=None)
    deploy_tags = attr.ib(default=None)
    deploy_network = attr.ib(default=None)
    preorder_id = attr.ib(default=None)
    disable_admin_requests = attr.ib(default=None)
    reason = attr.ib(default=None)
    host_type = attr.ib(default=walle_constants.HostType.SERVER)

    with_auto_healing = attr.ib(default=None)
    restrictions = attr.ib(default=None)
    check = attr.ib(default=None)
    ignore_cms = attr.ib(default=None)
    dns = attr.ib(default=False)
    instant = attr.ib(default=False)
    default_name = attr.ib(default=None)
    maintenance_properties = attr.ib(default=attr.Factory(dict))


class ParamsValidationStrategy:
    def apply(self, args):
        args.state, args.status = self.validate_state_and_status(args)
        return args

    @staticmethod
    def validate_state_and_status(args):
        state = args.state or HostState.ASSIGNED
        status = args.status or HostStatus.default(state)

        if (state, status) == (HostState.ASSIGNED, HostStatus.MANUAL):
            raise BadRequestError(
                "Status '{}' is forbidden in '{}' state. Use '{}' state instead",
                HostStatus.MANUAL,
                HostState.ASSIGNED,
                HostState.MAINTENANCE,
            )

        if state == HostState.MAINTENANCE and status != HostStatus.default(state):
            raise BadRequestError(
                "Only '{}' status is allowed for hosts in '{}' state.", HostStatus.default(state), state
            )
        if state == HostState.FREE and status not in (HostStatus.default(state), HostStatus.INVALID):
            raise BadRequestError(
                "Only '{}' statuses is allowed for hosts in '{}' state.",
                (HostStatus.default(state), HostStatus.INVALID),
                state,
            )
        return state, status


def prepare_maintenance_properties(args):
    maintenance_properties = args.maintenance_properties.copy()
    timeout_time, timeout_status, ticket_key, operation_state = prepare_maintenance_params(**maintenance_properties)

    maintenance_properties.update(
        {
            "timeout_time": timeout_time,
            "timeout_status": timeout_status,
            "ticket_key": ticket_key,
            "operation_state": operation_state,
        }
    )
    return maintenance_properties


class MaintenanceParamsValidationStrategy(ParamsValidationStrategy):
    def apply(self, args):
        args = super().apply(args)
        args.maintenance_properties = prepare_maintenance_properties(args)
        return args


def raise_on_task_params(args):
    task_parameters = ("ignore_cms", "check", "disable_admin_requests", "with_auto_healing", "dns")
    for parameter_name in task_parameters:
        if getattr(args, parameter_name) is not None:
            raise RequestValidationError(
                "Parameter is not suitable for instant (or in '{}' state) host adding: {}",
                HostState.FREE,
                parameter_name,
            )


def raise_on_free_params(args):
    for parameter_name in ("config", "provisioner", "deploy_config_policy", "deploy_tags", "restrictions"):
        if getattr(args, parameter_name) is not None:
            raise RequestValidationError("Host in '{}' state must not have '{}' field.", HostState.FREE, parameter_name)


class InstantParamsValidationStrategy(ParamsValidationStrategy):
    def apply(self, args):
        raise_on_task_params(args)
        return super().apply(args)


class FreeParamsValidationStrategy(ParamsValidationStrategy):
    def apply(self, args):
        raise_on_task_params(args)
        raise_on_free_params(args)
        return super().apply(args)


class MaintenanceInstantParamsValidationStrategy(ParamsValidationStrategy):
    def apply(self, args):
        raise_on_task_params(args)
        args = super().apply(args)
        args.maintenance_properties = prepare_maintenance_properties(args)
        return args


class ScheduleStrategyInterface:
    @staticmethod
    def schedule(host, args, deploy_configuration, add_host_audit_entry, status_audit_entry):
        raise NotImplementedError

    @staticmethod
    def notify(host, args):
        raise NotImplementedError


class ScheduleTaskStrategy(ScheduleStrategyInterface):
    @staticmethod
    def schedule(host, args, deploy_configuration, add_host_audit_entry, status_audit_entry):
        schedule_add_host(
            host,
            args.issuer,
            deploy_configuration,
            add_host_audit_entry,
            args.dns,
            args.check,
            args.ignore_cms,
            args.disable_admin_requests,
            args.with_auto_healing,
            args.reason,
        )

    @staticmethod
    def notify(host, args):
        notifications.on_task_enqueued(args.issuer, host, args.reason)


class ScheduleInstantStrategy(ScheduleStrategyInterface):
    @staticmethod
    def schedule(host, args, deploy_configuration, add_host_audit_entry, status_audit_entry):

        if host.state != HostState.FREE and deploy_configuration.provisioner == walle_constants.PROVISIONER_LUI:
            if args.host_type in walle_constants.HOST_TYPES_WITH_PARTIAL_AUTOMATION:
                # Attention: If you change the logic here don't forget to sync the changes with host preparing and releasing
                try:
                    lui_client = deploy.get_lui_client(deploy.get_deploy_provider(host.get_eine_box()))
                    lui_client.setup_host(host.name, mac=None, config_name=deploy_configuration.config)
                except deploy.DeployClientError as e:
                    raise Error("Failed to update host's config in LUI: {}", e)

        audit_entry = add_host_audit_entry or status_audit_entry
        host.set_status(
            host.status, args.issuer, audit_entry.id, reason=args.reason, confirmed=status_audit_entry is not None
        )

    @staticmethod
    def notify(host, args):
        pass


class InventoryStrategyInterface:
    @staticmethod
    def validate(args):
        raise NotImplementedError

    def process_inventory_data(self, host, project, host_network, args):
        raise NotImplementedError


class InternalInventoryStrategy(InventoryStrategyInterface):
    def __init__(self, empty_name_handler):
        self.empty_name_handler = empty_name_handler

    @staticmethod
    def validate(args):
        if args.inv is None and args.name is None:
            raise RequestValidationError("Host name or inventory number must be specified.")

    def process_inventory_data(self, host, project, host_network, args):
        (
            host.inv,
            host.name,
            host.ipmi_mac,
            host.macs,
            location,
            bot_project_id,
            platform,
        ) = inventory.get_host_info_and_check_status(args.inv or args.name)
        error_message = check_bot_project_id(host, project, bot_project_id)

        # platform named tuple -> embedded document
        host.platform = HostPlatform(system=platform.system, board=platform.board)

        host.rename_time = timestamp()

        if host.ipmi_mac is None and args.host_type not in HOST_TYPES_ALLOWED_WITHOUT_IPMI:
            raise InvalidHostConfiguration(
                "{} doesn't have IPMI. Wall-E can work only with IPMI hosts.", get_host_human_id(host.inv, host.name)
            )

        if host.name is None:
            self.empty_name_handler(host, args.default_name)

        if args.name is not None and host.name != args.name.lower():
            raise RequestValidationError("#{} points to {}, not {}.", host.inv, host.name, args.name)

        force_insert = _ensure_host_doesnt_exist(host.inv, host.name)
        if not args.preorder_id:
            if not _is_automated_action(args.issuer):
                host.authorize(args.issuer, ignore_maintenance=False)
            check_host_not_missing_from_preorder(host.inv)

        set_host_location(host, host_network, location, logical_datacenter=project.logical_datacenter)
        return error_message, force_insert


class ExternalInventoryStrategy(InventoryStrategyInterface):
    def __init__(self, *args):
        pass

    @staticmethod
    def validate(args):
        if args.inv is not None:
            raise RequestValidationError("Inventory number for external host must be generated automatically.")
        if args.name is None:
            raise RequestValidationError("Host name must be specified.")
        if bot.get_host_info(args.name) is not None:
            raise RequestValidationError("Host '{}' is registered in BOT.", args.name)

    def process_inventory_data(self, host, project, host_network, args):
        host.inv = generate_external_inventory_number()
        host.name = args.name
        host.type = walle_constants.HostType.VM

        host.platform = HostPlatform()
        host.location = HostLocation()
        host.rename_time = timestamp()

        _ensure_host_doesnt_exist(host.inv, host.name)

        # TODO: read location from args

        return None, True


class ShadowInventoryStrategy(InventoryStrategyInterface):
    def __init__(self, empty_name_handler):
        self.empty_name_handler = empty_name_handler

    @staticmethod
    def validate(args):
        if args.inv is None and args.name is None:
            raise RequestValidationError("Host name or inventory number must be specified.")

    def process_inventory_data(self, host, project, host_network, args):
        (
            host.inv,
            host.name,
            host.ipmi_mac,
            host.macs,
            location,
            bot_project_id,
            platform,
        ) = inventory.get_host_info_and_check_status(args.inv or args.name, supress_exceptions=True)
        error_message = check_bot_project_id(host, project, bot_project_id)

        # platform named tuple -> embedded document
        host.platform = HostPlatform(system=platform.system, board=platform.board)

        host.rename_time = timestamp()

        if host.name is None:
            self.empty_name_handler(host, args.default_name)

        if args.name is not None and host.name != args.name.lower():
            raise RequestValidationError("#{} points to {}, not {}.", host.inv, host.name, args.name)

        force_insert = _ensure_host_doesnt_exist(host.inv, host.name)

        set_host_location(host, host_network, location, logical_datacenter=project.logical_datacenter)
        return error_message, force_insert


def raise_on_empty_name(host, *args):
    raise BotEmptyNameError(
        "There is no host name in BOT associated with #{} inventory number. "
        "Hosts without names can be added only with '{}' state.",
        host.inv,
        HostState.FREE,
    )


def set_default_on_empty_name(host, default_name):
    host.name = default_name


def dead_on_empty_dns(host, args):
    if args.status != HostStatus.DEAD and not args.dns and not _exists_in_dns(host.name):
        host.status = HostStatus.DEAD
        return _forced_host_status(host, args.issuer, HostStatus.DEAD, "The host doesn't exist in DNS.")
    else:
        return dummy_context()


def ignore_empty_dns(host, args):
    return dummy_context()


def set_restrictions(host, project, restrictions):
    if restrictions is None:
        host.restrictions = project.default_host_restrictions
    else:
        host.restrictions = strip_restrictions(restrictions, strip_to_none=True)


def ignore_restrictions(*args):
    pass


class AddHostHandler:
    host = None
    host_network = None
    project = None

    error_message = None

    def __init__(
        self,
        validation_strategy,
        scheduling_strategy,
        inventory_strategy,
        empty_name_handler,
        empty_dns_handler,
        restrictions_handler,
        **parameters,
    ):
        self.original_args = AddHostArgs(**parameters)
        self.args = AddHostArgs(**parameters)

        self.validation_strategy = validation_strategy()
        self.host_scheduling_strategy = scheduling_strategy()
        self.inventory_strategy = inventory_strategy(empty_name_handler)
        self.empty_dns_handler = empty_dns_handler
        self.restrictions_handler = restrictions_handler
        self.remove_shadow_host = False

    def execute(self):
        self.args = self.validation_strategy.apply(self.args)
        self.inventory_strategy.validate(self.args)

        self.initialize_host_object()
        self.find_and_set_project()

        self.error_message, self.remove_shadow_host = self.inventory_strategy.process_inventory_data(
            self.host, self.project, self.host_network, self.args
        )
        self.commit_host_save()

        return self.host

    def initialize_host_object(self):
        self.host = Host(project=self.args.project, status=self.args.status, status_author=self.args.issuer)
        self.host_network = HostNetwork(uuid=self.host.uuid)

    def find_and_set_project(self):
        try:
            self.project = walle.projects.get_by_id(self.args.project)
        except walle.projects.ProjectNotFoundError as e:
            raise BadRequestError(str(e))

    def commit_host_save(self):
        with ProjectInterruptableLock(self.host.project):
            self.restrictions_handler(self.host, self.project, self.args.restrictions)
            deploy_configuration = self.set_host_deploy_configuration()
            _forced_status_audit = self.empty_dns_handler(self.host, self.args)
            _add_host_audit = self.create_audit_log_entry()

            with _forced_status_audit as status_audit_entry, _add_host_audit as add_host_audit_entry:
                self.set_host_state(add_host_audit_entry)
                self.set_host_type()
                self.host_scheduling_strategy.schedule(
                    self.host, self.args, deploy_configuration, add_host_audit_entry, status_audit_entry
                )

                if self.remove_shadow_host:
                    Host.objects(type=HostType.SHADOW_SERVER, inv=self.host.inv).delete()

                try:
                    self.host.save(force_insert=True)
                    self.host_network.save(force_insert=True)
                except mongoengine.NotUniqueError:
                    raise ResourceAlreadyExistsError(
                        "Host with the specified inventory number or DNS name already exists in Wall-E database."
                    )

                self.host_scheduling_strategy.notify(self.host, self.args)

    def set_host_state(self, add_host_audit_entry):
        expire = None
        ticket = None
        operation_state = HostOperationState.OPERATION

        if self.args.state == HostState.MAINTENANCE:
            ticket = self.args.maintenance_properties["ticket_key"]
            operation_state = self.args.maintenance_properties["operation_state"]
            expire = StateExpire(
                time=self.args.maintenance_properties["timeout_time"],
                status=self.args.maintenance_properties["timeout_status"],
                ticket=self.args.maintenance_properties["ticket_key"],
                issuer=self.args.issuer,
            )

        self.host.ticket = ticket
        self.host.operation_state = operation_state
        self.host.set_state(
            self.args.state,
            issuer=self.args.issuer,
            audit_log_id=add_host_audit_entry.id,
            expire=expire,
            reason=self.args.reason,
        )

    def set_host_type(self):
        self.host.type = self.project.type

    def set_host_deploy_configuration(self):
        (
            self.host.provisioner,
            self.host.config,
            self.host.deploy_tags,
            self.host.deploy_network,
            self.host.deploy_config_policy,
            deploy_configuration,
        ) = self.host.deduce_deploy_configuration(
            self.args.provisioner,
            self.args.config,
            self.args.deploy_tags,
            self.args.deploy_network,
            self.args.deploy_config_policy,
        )
        return deploy_configuration

    def create_audit_log_entry(self):
        return audit_log.on_add_host(
            self.args.issuer,
            self.host.project,
            self.host.inv,
            self.host.name,
            self.host.uuid,
            self.args.preorder_id,
            user_host=drop_none(attr.asdict(self.original_args)),
            added_host=self.host.to_api_obj(Host.api_fields),
            instant=self.args.instant,
            ignore_cms=self.args.ignore_cms,
            dns=self.args.dns,
            disable_admin_requests=self.args.disable_admin_requests,
            with_auto_healing=self.args.with_auto_healing,
            error=self.error_message,
            reason=self.args.reason,
            scenario_id=self.host.scenario_id,
            state=self.args.state,
            status=self.args.status,
        )


def add_host(
    issuer,
    project,
    inv=None,
    name=None,
    state=None,
    status=None,
    provisioner=None,
    config=None,
    deploy_config_policy=None,
    deploy_tags=None,
    deploy_network=None,
    restrictions=None,
    preorder_id=None,
    check=None,
    ignore_cms=None,
    disable_admin_requests=None,
    with_auto_healing=None,
    dns=None,
    instant=None,
    default_name=None,
    reason=None,
    maintenance_properties=None,
):
    project_obj = get_by_id(project, fields=("type",))

    if project_obj.type == walle_constants.HostType.VM:
        inventory_strategy = ExternalInventoryStrategy
    elif project_obj.type == walle_constants.HostType.SHADOW_SERVER:
        inventory_strategy = ShadowInventoryStrategy
    else:
        inventory_strategy = InternalInventoryStrategy

    if instant and state != HostState.FREE:
        validation_strategy = InstantParamsValidationStrategy
        if state == HostState.MAINTENANCE:
            validation_strategy = MaintenanceInstantParamsValidationStrategy

        strategies = (
            validation_strategy,
            ScheduleInstantStrategy,
            inventory_strategy,
            raise_on_empty_name,
            dead_on_empty_dns,
            set_restrictions,
        )
    elif state == HostState.FREE:
        strategies = (
            FreeParamsValidationStrategy,
            ScheduleInstantStrategy,
            inventory_strategy,
            set_default_on_empty_name,
            ignore_empty_dns,
            ignore_restrictions,
        )
    elif state == HostState.MAINTENANCE:
        strategies = (
            MaintenanceParamsValidationStrategy,
            ScheduleTaskStrategy,
            inventory_strategy,
            raise_on_empty_name,
            dead_on_empty_dns,
            set_restrictions,
        )
    else:
        strategies = (
            ParamsValidationStrategy,
            ScheduleTaskStrategy,
            inventory_strategy,
            raise_on_empty_name,
            dead_on_empty_dns,
            set_restrictions,
        )

    return AddHostHandler(
        *strategies,
        issuer=issuer,
        project=project,
        inv=inv,
        name=name,
        state=state,
        status=status,
        provisioner=provisioner,
        config=config,
        deploy_config_policy=deploy_config_policy,
        deploy_tags=deploy_tags,
        deploy_network=deploy_network,
        restrictions=restrictions,
        preorder_id=preorder_id,
        check=check,
        ignore_cms=ignore_cms,
        disable_admin_requests=disable_admin_requests,
        with_auto_healing=with_auto_healing,
        dns=dns,
        default_name=default_name,
        reason=reason,
        maintenance_properties=maintenance_properties,
        host_type=project_obj.type,
    ).execute()


def instant_delete_host(issuer, host, host_query, ignore_maintenance=False, lui=False, reason=None):
    with audit_log.on_delete_host(
        issuer,
        host.project,
        host.inv,
        host.name,
        host.uuid,
        lui,
        instant=True,
        reason=reason,
        scenario_id=host.scenario_id,
    ):
        allowed_states = HostState.ALL
        allowed_statuses = HostStatus.ALL_STEADY + [HostStatus.INVALID]

        if host.state in allowed_states and host.status in allowed_statuses:
            if lui:
                deploy.get_client(deploy.get_deploy_provider(host.get_eine_box())).remove(host.name)

            name_template = get_free_host_name_template()
            if host.name and not name_template.fqdn_matches(host.name):
                BlockedHostName.store(host.name)

            query = get_host_query(issuer, ignore_maintenance, allowed_states, allowed_statuses, **host_query)
            return Host.objects(**query).only("inv").modify(remove=True)

        # If all was ok, we shouldn't get here.
        raise InvalidHostStateError(host, allowed_states=allowed_states, allowed_statuses=allowed_statuses)


def change_host_inventory_number(host, new_inv, ipmi_mac, reason):
    log_entry = audit_log.on_host_inv_changed(
        issuer=authorization.ISSUER_WALLE,
        project_id=host.project,
        old_inv=host.inv,
        new_inv=new_inv,
        host_uuid=host.uuid,
        name=host.name,
        reason=reason,
    )

    with log_entry, HostInterruptableLock(host.uuid, host.tier):
        if shadow_host := Host.objects(type=HostType.SHADOW_SERVER, inv=new_inv).first():
            shadow_host.delete()
        Host.objects(inv=host.inv).update(**fix_mongo_set_kwargs(set__inv=new_inv, set__ipmi_mac=ipmi_mac))

    if host.status == HostStatus.INVALID:
        walle.host_status.force_status(
            authorization.ISSUER_WALLE,
            host,
            HostStatus.default(host.state),
            only_from_current_status_id=True,
            ignore_maintenance=True,
            reason=reason,
        )
