import logging

from walle.host_platforms.errors import PlatformAlreadyExistsError
from walle.host_platforms.platform import Platform

log = logging.getLogger(__name__)


class PlatformStore:
    def __init__(self):
        self.system_idx = {}
        self.board_idx = {}

    def add_platform(self, system, board, platform):
        if system in self.system_idx:
            raise PlatformAlreadyExistsError('platform with system model {} already registered'.format(system))
        self.system_idx[system] = platform

        if board is not None:
            if board in self.board_idx:
                raise PlatformAlreadyExistsError('platform with board model {} already registered'.format(board))
            self.board_idx[board] = platform

    def get_platform(self, system, board):
        if system in self.system_idx:
            return self.system_idx[system]
        if board in self.board_idx:
            return self.board_idx[board]
        return None


class PlatformRegistry:
    registered_platforms = PlatformStore()

    @classmethod
    def register_platform(cls, klass):
        cls.registered_platforms.add_platform(klass.system, klass.board, klass)
        log.info("registered ipmi platform for system=%s board=%s", klass.system, klass.board)
        return klass

    @classmethod
    def get_platform_for_host(cls, host):
        return cls.registered_platforms.get_platform(host.platform.system, host.platform.board)


def create_platform_for_host(host):
    # try to get supported platform from registry or return generic platform
    platform_cls = PlatformRegistry.get_platform_for_host(host)
    platform_cls = platform_cls if platform_cls else Platform

    return platform_cls(host)
