import logging

from walle.host_platforms.errors import PlatformIPMIError
from walle.host_platforms.platform import Platform, PlatformProblems

log = logging.getLogger(__name__)


class PlatformGigabyte(Platform):
    POST_CODE_IDX = None
    POST_CODES_MEMORY_PROBLEM = None
    POST_CODE_OK = None
    POST_GET_COMMAND = None

    def __init__(self, host):
        super().__init__(host)

    def get_current_post_code(self, ipmi_client):
        ipmi_result = ipmi_client.raw_command(self.POST_GET_COMMAND)
        if not ipmi_result['success']:
            raise PlatformIPMIError("ipmi raw command was not successful")

        post_line = ipmi_result['message'].strip()
        log.debug("Got post line %s for %s", post_line, self.fqdn)

        try:
            post_code = int(post_line.split(" ")[self.POST_CODE_IDX], 16)

        # return none if we failed to parse POST-code
        except (IndexError, ValueError):
            post_code = None

        return post_code

    def get_post_problem_for_code(self, code):
        # workaround for non-parsable POST-codes
        if code is None:
            return PlatformProblems.POST_OK
        else:
            if code in self.POST_CODES_MEMORY_PROBLEM:
                return PlatformProblems.POST_MEMORY_PROBLEM
            else:
                return PlatformProblems.POST_OK

    def provides_post_code(self):
        return True


class PlatformGigabyteV2(PlatformGigabyte):
    POST_CODE_IDX = 0
    POST_CODES_MEMORY_PROBLEM = [0xB7, 0xB9, 0xBA, 0xBD]
    POST_CODE_OK = 0x01
    POST_GET_COMMAND = '0x34 0xcc'

    def __init__(self, host):
        super().__init__(host)


class PlatformGA7PTSH(PlatformGigabyteV2):
    # platform info overrides
    system = 'T17I-234'
    board = 'GA-7PTSH'


class PlatformGA7PPSH(PlatformGigabyteV2):
    # platform info overrides
    system = 'GS-CFZ1/GS-CCZx'
    board = 'GA-7PPSH'

    def provides_post_code(self):
        return False  # temporarily disable POST codes, see WALLE-2958


class PlatformGigabyteV3(PlatformGigabyte):
    POST_CODE_IDX = 2
    POST_CODES_MEMORY_PROBLEM = [0xB7, 0xB9, 0xBA, 0xBD]
    POST_CODE_OK = 0xB2
    POST_GET_COMMAND = '0x04 0x2d 0x9d'

    def __init__(self, host):
        super().__init__(host)


class PlatformMY70EX0Y3N(PlatformGigabyteV3):
    # platform info overrides
    system = 'T174-N40'
    board = 'MY70-EX0-Y3N'


class PlatformMH60RE2Y2(PlatformGigabyteV3):
    # platform info overrides
    system = 'T173-1M1-Y2'
    board = 'MH60-RE2-Y2'


class PlatformMH60RE0Y2(PlatformGigabyteV3):
    # platform info overrides
    system = 'T173-138-Y2'
    board = 'MH60-RE0-Y2'


class PlatformAMD(PlatformGigabyte):
    POST_CODE_IDX = -1
    POST_CODES_MEMORY_PROBLEM = [0x10]

    POST_CODE_OK = 0xAF
    POST_GET_COMMAND = '0x32 0x73 0x0'


class PlatformMZ81EX0Y3N(PlatformAMD):
    board = 'MZ81-EX0-Y3N'
    system = 'MZ81-EX0-Y3N'


def register_gigabyte_platforms(manager):
    manager.register_platform(PlatformGA7PPSH)
    manager.register_platform(PlatformGA7PTSH)
    manager.register_platform(PlatformMY70EX0Y3N)
    manager.register_platform(PlatformMH60RE0Y2)
    manager.register_platform(PlatformMH60RE2Y2)
    manager.register_platform(PlatformMZ81EX0Y3N)
