"""Host management."""

import logging
import typing as tp
from collections import namedtuple
from uuid import uuid4

import mongoengine
from mongoengine import (
    EmbeddedDocument,
    StringField,
    IntField,
    LongField,
    ListField,
    EmbeddedDocumentField,
    DictField,
    BooleanField,
    Q,
    MapField,
)
from pymongo import DESCENDING

import sepelib
import walle.clients.ipmiproxy
import walle.clients.ssh
import walle.expert.automation
from sepelib.core import config, constants
from sepelib.core.exceptions import Error, LogicalError
from sepelib.mongo import monotonic
from sepelib.mongo.util import register_model
from walle import audit_log, constants as walle_constants, projects, restrictions
from walle.authorization import blackbox, has_iam
from walle.clients import dns as dns_clients
from walle.clients.cms import CmsNamespace
from walle.clients.eine import ProfileMode
from walle.clients.network import racktables_client
from walle.constants import (
    EINE_PROFILES_WITH_DC_SUPPORT,
    NETWORK_SOURCE_RACKTABLES,
    NETWORK_SOURCE_LLDP,
    NetworkTarget,
    FLEXY_EINE_PROFILE,
    DEFAULT_TIER_ID,
    HostType,
)
from walle.errors import (
    RequestValidationError,
    InvalidProfileNameError,
    BadRequestError,
    HostUnderMaintenanceError,
    HostNotFoundError,
    HostNetworkNotFoundError,
    OperationRestrictedForHostTypeError,
)
from walle.fsm_stages.constants import StageStatus
from walle.host_network import HostNetwork
from walle.host_platforms import create_platform_for_host
from walle.idm.project_role_managers import ProjectRole
from walle.models import timestamp, Document
from walle.operations_log.constants import OPERATION_HOST_STATUSES, Operation
from walle.stages import Stage, Stages
from walle.util.deploy_config import DeployConfigPolicies
from walle.util.juggler import get_aggregate_name
from walle.util.misc import fix_mongo_set_kwargs, format_time, get_location_path
from walle.restrictions import EXCLUDE_FOR_MACS

log = logging.getLogger(__name__)

ActiveMacAddressInfo = namedtuple("ActiveMacAddressInfo", ["mac", "source", "timestamp"])
ProfileConfiguration = namedtuple("ProfileConfiguration", ["profile", "tags", "modes"])
DeployConfiguration = namedtuple(
    "DeployConfiguration", ["provisioner", "config", "tags", "certificate", "network", "ipxe", "deploy_config_policy"]
)

_LOCATION_STALE_TIMEOUT = constants.DAY_SECONDS


HostNetworkLocationInfo = namedtuple("HostNetworkLocationInfo", ["switch", "port", "source", "timestamp"])

MAX_BOT_INVENTORY_NUMBER = (1 << 31) - 1


class NonIpmiHostError(Error):
    def __init__(self):
        super().__init__("The host doesn't have IPMI.")


class MissingDefaultProfileConfigurationError(BadRequestError):
    def __init__(self):
        super().__init__(
            "Default Einstellung profile is not specified for the project, "
            "so you must explicitly specify a profile which should be assigned to the host."
        )


class HostState:
    FREE = "free"
    """Host is free and ready for assigning."""

    ASSIGNED = "assigned"
    """Host is assigned to a role."""

    MAINTENANCE = "maintenance"
    """Host is under maintenance."""

    PROBATION = "probation"
    """Host is being prepared for assignment."""

    ALL_ASSIGNED = [PROBATION, ASSIGNED, MAINTENANCE]
    """Host is assigned and prepared for production."""

    ALL_DMC = [PROBATION, ASSIGNED]
    """Host can be processed by DMC."""

    ALL_DOWNTIME = [MAINTENANCE, PROBATION]
    """Host is being worked on and needs to be on downtime."""

    ALL_IGNORED_LIMITS_COUNTING = [MAINTENANCE, PROBATION]
    """Host is being worked on and should not be counted when check failure limits"""

    ALL = [FREE, PROBATION, ASSIGNED, MAINTENANCE]
    """A list of states host can be in."""

    MAC_SYNC_STATES = [FREE, PROBATION, ASSIGNED, MAINTENANCE]
    """List of states, in which we can perform macs update"""


class HostOperationState:
    OPERATION = "operation"
    """Host is in operation."""

    DECOMMISSIONED = "decommissioned"
    """Host had been decommissioned."""

    ALL_INUSE = [OPERATION]
    """Host is in production."""

    ALL_NOTINUSE = [DECOMMISSIONED]
    """Host is not in production."""

    ALL = [OPERATION, DECOMMISSIONED]
    """A list of states host operation status can be in."""


class HostStatus:
    READY = "ready"
    """The host is ready."""

    MANUAL = "manual"
    """The host is marked for manual operation by administrator."""

    DEAD = "dead"
    """Host is dead."""

    INVALID = "invalid"
    """Host has an inconsistent state which must be handled manually."""

    ALL_STEADY = [READY, MANUAL, DEAD]
    """A list of statuses in which host stays in constant predictable state."""

    ALL_ASSIGNED = [READY, DEAD]
    """A list of statuses in which host can be when it have an assigned state."""

    ALL_TASK = OPERATION_HOST_STATUSES
    """A list of statuses host which is processing some task can be in."""

    ALL_RENAMING = [
        Operation.PREPARE.host_status,
        Operation.SWITCH_PROJECT.host_status,
        Operation.CHECK_DNS.host_status,
    ]
    """A list of statuses in which host can be renamed or its DNS records can be changed."""

    ALL_TIMEOUT = [READY, DEAD]
    """A list of statuses host can be automatically switched to on timeout."""

    ALL = ALL_STEADY + ALL_TASK + [INVALID]
    """A list of statuses host can be in."""

    FILTER_STEADY = "steady"
    """Filter all idle hosts from UI or API."""

    FILTER_TASK = "task"
    """Filter all host that currently being processed by Wall-E."""

    FILTER_ERROR = "error"
    """Filter all host that have any errors."""

    ALL_FILTERS = [FILTER_STEADY, FILTER_TASK, FILTER_ERROR]
    """All group status filters to be exposed to the constants UI."""

    DEFAULT_STATUSES = {
        HostState.FREE: READY,
        HostState.ASSIGNED: READY,
        HostState.PROBATION: READY,
        HostState.MAINTENANCE: MANUAL,
    }

    assert sorted(DEFAULT_STATUSES) == sorted(HostState.ALL)

    @classmethod
    def default(cls, state):
        return cls.DEFAULT_STATUSES[state]


class DecisionStatus:
    HEALTHY = "healthy"
    FAILURE = "failure"

    ALL = [HEALTHY, FAILURE]


class TaskType:
    MANUAL = "manual"
    """Task queued by user."""

    AUTOMATED_ACTION = "automated-action"
    """Task that has been queued automatically."""

    AUTOMATED_HEALING = "automated-healing"
    """Task that has been queued automatically by host healing logic."""

    ALL_AUTOMATED = [AUTOMATED_ACTION, AUTOMATED_HEALING]
    """A list of available automated task types."""

    ALL = [MANUAL] + ALL_AUTOMATED
    """A list of available task types."""


class HostLocation(EmbeddedDocument):
    """Location of a host (network or physical)."""

    switch = StringField(help_text="Switch name")
    port = StringField(help_text="Port name")
    network_source = StringField(choices=walle_constants.NETWORK_SOURCES, help_text="Source of information")

    country = StringField(help_text="Country")
    city = StringField(help_text="City")
    datacenter = StringField(help_text="Datacenter")
    queue = StringField(help_text="Datacenter's queue")
    rack = StringField(help_text="Rack")
    unit = StringField(help_text="Unit")
    physical_timestamp = LongField(help_text="Physical location updated at")

    # WALLESUPPORT-1806
    logical_datacenter = StringField(help_text="Logical datcenter identifier")

    # We've got some weird names for datacenters and queues, which is not something we want to expose to humans.
    short_datacenter_name = StringField(help_text="Short datacenter name")
    short_queue_name = StringField(help_text="Short datacenter's queue name")

    def get_path(self):
        return get_location_path(self)

    @staticmethod
    def split_physical_location_string_to_fields(physical_location: str):
        # NB: Does not support 'unit' field yet.
        fields = ["country", "city", "datacenter", "queue", "rack"]
        location_bits = physical_location.split("|")
        if len(location_bits) > len(fields):
            raise ValueError("Path with more than %d bits: %r" % (len(fields), location_bits))

        return {field_name: field_value for field_name, field_value in zip(fields, location_bits)}


class StateExpire(EmbeddedDocument):
    """State expiration configuration.
    After state expiration host will be switched into assigned state with specified status."""

    time = LongField(required=False, help_text="Time when the host will be switched to timeout status")
    ticket = StringField(required=True, help_text="Startrek ticket key for host state")
    status = StringField(required=True, help_text="Status the host will be switched to at the timeout time")
    issuer = StringField(required=True, help_text="Request issuer")


class HostMessage(EmbeddedDocument):
    SEVERITY_INFO = "info"
    SEVERITY_ERROR = "error"
    SEVERITIES = [SEVERITY_INFO, SEVERITY_ERROR]

    severity = StringField(
        choices=SEVERITIES, required=True, help_text="severity is used by UI to adjust message appearance"
    )
    message = StringField(required=True, help_text="message text")

    @classmethod
    def info(cls, message):
        return cls(severity=cls.SEVERITY_INFO, message=message)

    @classmethod
    def error(cls, message):
        return cls(severity=cls.SEVERITY_ERROR, message=message)


class Decision(EmbeddedDocument):
    """Data represents decision made by DMC"""

    action = StringField(required=True, help_text="decision action")
    params = DictField(required=False, default=None, help_text="decision params")
    checks = ListField(StringField(), default=None, help_text="checks that failed")
    failures = ListField(StringField(), default=None, help_text="list of failures that should be repaired")
    reason = StringField(required=True, help_text="human readable reason")
    restrictions = ListField(StringField(), help_text="list of restrictions applied to the operation")
    failure_type = StringField(required=False, default=None, help_text="failure type")
    failure_check_info = DictField(required=False, default=None, help_text="failure check state")
    counter = LongField(default=0, help_text="Automatically incremented decision counter")
    rule_name = StringField(required=False, default="", help_text="failed rule name (for new DMC migration)")

    def to_dict(self):
        return self.to_mongo(use_db_field=False)


class HealthStatus(EmbeddedDocument):
    """Contains info about current host health status."""

    STATUS_OK = "ok"
    """All checks passed"""

    STATUS_BROKEN = "broken"  # ok < broken < failure
    """Have some missing or suspected checks (not ok and not failed)"""

    STATUS_FAILURE = "failure"
    """Have any failed checks"""

    STATUSES = [STATUS_OK, STATUS_BROKEN, STATUS_FAILURE]

    event_id = LongField(help_text="Event ID that increases with time (per host)")

    status = StringField(
        choices=STATUSES, required=True, help_text="Aggregated health status. Speeds up host search and exposed by API."
    )
    reasons = ListField(
        StringField(),
        default=None,
        help_text="A list of reasons for failure status. Exposed by API and used for metrics collection.",
    )
    check_statuses = DictField(
        required=True, help_text="Maps checks that are enabled for the host to their statuses. Exposed by API."
    )

    decision = EmbeddedDocumentField(Decision, default=None, help_text="Decision produced for this health status")

    def clone_decision(self):
        # NOTE(rocco66): get decision for future saving (failure_type serialization problem)
        if self.decision:
            decision_dict = self.decision.to_dict()
            if self.decision.failure_type:
                orig_failure_type = decision_dict["failure_type"]
                if isinstance(orig_failure_type, str):
                    new_failure_type = orig_failure_type
                elif isinstance(orig_failure_type, list):
                    new_failure_type = orig_failure_type[0]
                else:
                    raise RuntimeError(f"unsupported failure_type {repr(orig_failure_type)}")
                decision_dict["failure_type"] = new_failure_type
            decision = Decision(**decision_dict)
            return decision

    def __eq__(self, other):
        return isinstance(other, type(self)) and other.to_mongo() == self.to_mongo()

    def __repr__(self):
        return "<{}>({})".format(type(self), repr(self.to_mongo()))


class DnsConfiguration(EmbeddedDocument):
    """Contains information about current DNS configuration for the host."""

    project = StringField(help_text="Project ID for which IP addresses have been generated")
    switch = StringField(help_text="Switch name for which networks IP addresses have been generated")
    mac = StringField(help_text="MAC address for which IP addresses have been generated")
    vlan_scheme = StringField(help_text="VLAN scheme for which IP addresses have been generated")
    vlans = ListField(
        IntField(min_value=walle_constants.VLAN_ID_MIN, max_value=walle_constants.VLAN_ID_MAX),
        help_text="Host VLANs for which IP addresses have been generated",
    )
    ips = ListField(StringField(), help_text="Host's IP addresses reported by agent on host")
    error_time = LongField(help_text="A time when last error has occurred during checking the host")
    check_time = LongField(
        help_text="Time when hosts' DNS records has been checked against the specified switch and MAC. "
        "Every change of switch and MAC fields must be accompanied by changing this field. "
        "If check_time is set but switch and mac fields are unset it means that the host has been "
        "checked but we've failed to determine the right switch and MAC and can't guarantee "
        "consistency of DNS."
    )

    update_time = LongField(help_text="The latest time when the host's DNS records were changed")


class Task(EmbeddedDocument):
    """Represents a long-running host task."""

    task_id = LongField(
        required=True,
        help_text="Task ID. It's a monotonically increasing number, so it determines task creation order",
    )
    iss_banned = BooleanField(default=False, help_text="Permission from CMS already acquired")
    owner = StringField(required=True, help_text="Task owner")
    type = StringField(choices=TaskType.ALL, required=True, help_text="Task type", default=HostType.SERVER.value)
    name = StringField(choices=OPERATION_HOST_STATUSES, required=False, help_text="Task/process name")
    audit_log_id = StringField(required=True, help_text="ID of audit log entry that corresponds to this task")
    cms_task_id = StringField(help_text="CMS task id for this task")

    ignore_cms = BooleanField(default=False, help_text="Don't acquire permission from CMS")
    keep_downtime = BooleanField(help_text="Keep downtime flag after task finished")
    keep_cms_task = BooleanField(help_text="Keep cms task after this task finished")

    disable_admin_requests = BooleanField(
        default=False, help_text="Don't issue any admin requests if something is broken - just fail the task"
    )
    enable_auto_healing = BooleanField(
        help_text="Try to heal host automatically if task will fail (for manual tasks only)"
    )

    stages = ListField(EmbeddedDocumentField(Stage), help_text="A list of stages assigned to this task")
    target_status = StringField(
        choices=HostStatus.ALL, help_text="Status that will be assigned to the host on successful task completion"
    )
    stage_uid = StringField(help_text="Current stage UID")
    stage_name = StringField(help_text="Current stage name")
    status = StringField(required=True, help_text="An aggregated task status that is exposed by API")
    status_message = StringField(help_text="An optional message that stage may set to describe current progress")
    error = StringField(help_text="An error string if task is experiencing an error now")
    next_check = LongField(required=True, help_text="A time the next check should be processed at")
    location = DictField(help_text="Switch and port")

    hardware_error_count = IntField(default=0, help_text="Hardware error count")
    power_error_count = IntField(default=0, help_text="Power on/off timeout error count")
    reopened_admin_request_count = IntField(
        default=0, help_text="A number of reopened admin requests for power or IPMI errors"
    )

    revision = IntField(required=True, help_text="Revision of the document")

    decision = EmbeddedDocumentField(
        Decision,
        default=None,
        help_text=(
            "Decision on this task generation moment. Used by task rescheduling on most priority failure (autohealing only)"
        ),
    )

    @staticmethod
    def next_task_id():
        return monotonic.get_next("task_id")

    def get_cms_task_id(self):
        if self.cms_task_id is None:
            return CmsNamespace.add_namespace(str(self.task_id))
        else:
            return self.cms_task_id

    def is_automated_task(self):
        return self.type == TaskType.AUTOMATED_HEALING or self.enable_auto_healing

    @staticmethod
    def make_status(stage_name, stage_status):
        if not stage_status or stage_name == stage_status:
            return stage_name
        else:
            return "{}:{}".format(stage_name, stage_status)

    def allows_power_off(self):
        return self.status in [
            self.make_status(Stages.CHANGE_DISK, StageStatus.HW_ERRORS_WAITING_DC),
            self.make_status(Stages.HW_REPAIR, StageStatus.HW_ERRORS_WAITING_DC),
        ]


class HostPlatform(EmbeddedDocument):
    """Represents host platform (system/MoBo models)"""

    system = StringField(help_text="System model name (dmidecode -t1)")
    board = StringField(help_text="Baseboard model name (dmidecode -t2)")


def _uuid():
    return uuid4().hex


class InfinibandInfo(EmbeddedDocument):
    """Represents racktables infiniband info"""

    cluster_tag = StringField(help_text="Cluster tag (e.g. YATI1)")
    ports = ListField(StringField(), help_text="Infiniband ports for host")

    def has_changes(self, info: racktables_client.InfinibandInfo):
        return self.cluster_tag != info.cluster_tag or set(self.ports) != info.ports

    @staticmethod
    def from_racktables_info(info: racktables_client.InfinibandInfo):
        return InfinibandInfo(info.cluster_tag, sorted(info.ports))

    def __str__(self):
        return f"<InfinibandInfo: {self.cluster_tag}, {'/'.join(str(p) for p in self.ports)}>"


@register_model
class HostDecision(Document):
    uuid = StringField(primary_key=True, required=True)
    decision = EmbeddedDocumentField(Decision, help_text="Decision produced for this health status")
    counter = LongField(default=0, help_text="Automatically incremented decision counter")
    screening_time = LongField(required=False, help_text="timestamp for the first decision for current status")

    meta = {
        "collection": "host_decision",
    }


@register_model
class Host(Document):
    """Represents a host controlled by the system."""

    uuid = StringField(primary_key=True, required=True, default=_uuid)
    inv = LongField(min_value=0, required=True, help_text="Inventory number")
    name = StringField(min_length=1, help_text="FQDN")

    type = StringField(choices=HostType.get_choices(), required=True, help_text="Type of host")

    project = StringField(required=True, help_text="Project ID")
    restrictions = ListField(StringField(choices=restrictions.ALL), default=None, help_text="Host restrictions")
    extra_vlans = ListField(
        IntField(min_value=walle_constants.VLAN_ID_MIN, max_value=walle_constants.VLAN_ID_MAX),
        default=None,
        help_text="Extra VLANs that should be assigned to the host",
    )
    infiniband_info = EmbeddedDocumentField(InfinibandInfo, help_text="Infiniband info from racktables")

    scenario_id = IntField(help_text="Current scenario id that is currently use this host")

    tier = IntField(default=DEFAULT_TIER_ID, help_text="Which tier should handle this host")

    provisioner = StringField(choices=walle_constants.PROVISIONERS, help_text="Provisioner")
    config = StringField(min_length=1, help_text="Assigned deploy config")
    deploy_tags = ListField(StringField(), default=None, help_text="Deploy tags")
    deploy_network = StringField(choices=NetworkTarget.DEPLOYABLE, help_text="Deploy network (service or project)")
    deploy_config_policy = StringField(
        choices=DeployConfigPolicies.get_all_names(), help_text="Deploy config policy", required=False
    )

    state = StringField(choices=HostState.ALL, required=True, help_text="Current host state")
    state_time = LongField(required=True, help_text="Time when the current state has been set")
    state_expire = EmbeddedDocumentField(StateExpire, help_text="State expiration config")
    state_author = StringField(help_text="Issuer of the command that led to this state")
    state_audit_log_id = StringField(help_text="ID of audit log entry that is responsible for this state change.")
    state_reason = StringField(help_text="Optional reason for state change")

    operation_state = StringField(
        choices=HostOperationState.ALL, default=HostOperationState.OPERATION, help_text="Current host operation state"
    )

    status = StringField(choices=HostStatus.ALL, required=True, help_text="Current host state's status")
    status_time = LongField(required=True, help_text="Time when the current status has been set")
    status_author = StringField(help_text="Issuer of the command that led to this status")
    status_audit_log_id = StringField(
        help_text="ID of audit log entry that is responsible for this status change. "
        "Not required only because of old statuses that was set before "
        "introducing this feature."
    )
    status_reason = StringField(help_text="Optional reason for status change")

    cms_task_id = StringField(help_text="CMS task id for current host status")
    on_downtime = BooleanField(default=False, help_text="True if the host is on downtime in Juggler")
    rename_time = LongField(help_text="The latest time when the host could be renamed considering the worst case")

    ipmi_mac = StringField(help_text="Host's IPMI MAC")
    macs = ListField(StringField(), help_text="Host's MACs")

    ips = ListField(StringField(), help_text="Host's IP addresses")

    platform = EmbeddedDocumentField(
        HostPlatform, required=True, help_text="Host platform info: system and baseboard model names. "
    )
    agent_version = StringField(help_text="Agent version on host")

    # Attention: Host may have `active_mac_source` and `active_mac_time` without `active_mac`: it means that we've
    # learned host's active MACs, but it had a few active MACs.
    active_mac = StringField(help_text="Host's active MAC")
    active_mac_source = StringField(
        choices=walle_constants.MAC_SOURCES, help_text="Source name the active MAC has been learned from"
    )

    location = EmbeddedDocumentField(HostLocation, required=True, help_text="Location of a host (network and physical)")
    dns = EmbeddedDocumentField(DnsConfiguration, help_text="Current DNS configuration")

    health = EmbeddedDocumentField(HealthStatus, help_text="Current health status")

    # TODO: healthy_checks_min_time, active_check_min_time and passive_check_min_time are obsolete.
    # Drop them from hosts which have `checks_min_time` field set.
    healthy_checks_min_time = LongField(
        default=0,
        help_text="If all checks are passed and >= this time, "
        "processed task can be considered as successfully completed",
    )
    active_check_min_time = LongField(default=0, help_text="Time starting from which active checks become valid")
    passive_check_min_time = LongField(default=0, help_text="Time starting from which passive checks become valid")
    checks_min_time = LongField(default=0, help_text="Health reset time, starting from which checks become valid")

    # NB: this timestamp is not when current decision have been produced,
    # it is when decision.action changed from HEALTHY to not-HEALTHY or from not-HEALTHY to HEALTHY
    # It may also coincide with health status going from OK to FAILURE or from FAILURE to OK, but not always.
    decision_status = StringField(choices=DecisionStatus.ALL, default=None)
    decision_status_timestamp = LongField(
        required=False, help_text="timestamp for the first decision for current status"
    )

    task = EmbeddedDocumentField(Task, help_text="Current long-running task status info")
    ticket = StringField(help_text="Startrek ticket key for host error, if any")

    messages = MapField(
        ListField(EmbeddedDocumentField(HostMessage), default=list),
        default=dict,
        help_text="Different subsystems store their messages in one place so that UI could always find them",
    )

    walle_agent_errors_flag = BooleanField(
        help_text="Flag which shows if any logical errors were detected "
        "by walle agent. Has value True if there are some errors."
    )

    release_fields = [
        "config",
        "deploy_tags",
        "deploy_network",
        "deploy_config_policy",
        "dns",
        "extra_vlans",
        "health",
        "checks_min_time",
        "provisioner",
        "restrictions",
        "on_downtime",
        "cms_task_id",
        "state_expire",
        "operation_state",
        "walle_agent_errors_flag",
    ]
    """Fields that should be deleted from object on host releasing."""

    default_api_fields = (
        "uuid",
        "inv",
        "name",
        "state",
        "state_reason",
        "status",
        "status_author",
        "status_reason",
        "config",
        "deploy_tags",
        "task.status",
        "ticket",
    )

    iam_public_fields = (
        # NOTE(rocco66): UI needs
        "health.check_statuses",
        "inv",
        "location.city",
        "location.port",
        "location.switch",
        "messages",
        "name",
        "project",
        "restrictions",
        "scenario_id",
        "state",
        "state_reason",
        "status",
        "status_reason",
        "task.status",
        "task.status_message",
        "ticket",
        "tier",
        "uuid",
        # NOTE(rocco66): additional juggler needs
        "location.short_datacenter_name",
        "location.short_queue_name",
        "status",
    )

    api_fields = default_api_fields + (
        "type",
        "project",
        "messages",
        "provisioner",
        "deploy_network",
        "deploy_config_policy",
        "restrictions",
        "status_audit_log_id",
        "status_time",
        "extra_vlans",
        "state_expire",
        "state_audit_log_id",
        "state_author",
        "state_time",
        # Health
        "health.status",
        "health.reasons",
        "health.check_statuses",
        "health.decision",
        # MACs
        "ipmi_mac",
        "macs",
        "active_mac",
        "active_mac_source",
        # IPs
        "ips",
        # Physical location
        "location.country",
        "location.city",
        "location.datacenter",
        "location.queue",
        "location.rack",
        "location.unit",
        "location.short_datacenter_name",
        "location.short_queue_name",
        "location.physical_timestamp",
        # Network location
        "location.switch",
        "location.port",
        "location.network_source",
        # Infiniband
        "infiniband_info.cluster_tag",
        "infiniband_info.ports",
        # Task info
        "task.owner",
        "task.status_message",
        "task.error",
        "task.audit_log_id",
        "task.cms_task_id",
        # Platform information
        "platform",
        # Wall-e agent info
        "agent_version",
        # Scenario info
        "scenario_id",
        # Operation state
        "operation_state",
        # Tier settings
        "tier",
    )

    meta = {
        "collection": "hosts",
        "indexes": [
            {"name": "inv", "fields": ["inv"], "unique": True},
            {"name": "name", "fields": ["name"], "sparse": True, "unique": True},
            {"name": "state", "fields": ["state"]},
            {"name": "inv_project", "fields": ["inv", "project"]},
            {"name": "type", "fields": ["type"]},
            # This index is also used for metrics creation
            {"name": "task_id", "fields": ["status", "task.task_id"]},
            # This index is used by downtime GC
            {"name": "downtime", "fields": ["on_downtime"], "sparse": True},
            # This index is also used for metrics creation
            {"name": "host_checking", "fields": ["health.status", "status"]},
            # This index is used for netmon data retrieval
            {"name": "switch", "fields": ["location.switch"], "sparse": True},
            # This index is used by scenario
            {"name": "scenario_id", "fields": ["scenario_id"]},
            # This index is used by CAuth api
            {"name": "project", "fields": ["project"]},
            # This index is used for rack_topology_sync
            {"name": "queue_rack", "fields": ["location.short_queue_name", "location.rack"]},
            # This index is used for triage
            {"name": "tier_triage", "fields": ["tier", "status", "decision_status", "inv"]},
            # This index is used for screening
            {"name": "tier_screening", "fields": ["tier", "state", "inv"]},
        ],
    }

    @classmethod
    def get_by_inv(cls, inv, fields=None):
        """Returns a host by its inventory num."""
        return _get_with_query(fields, inv=inv)

    @classmethod
    def get_by_uuid(cls, uuid, fields=None):
        """Returns a host by its UUID."""
        return _get_with_query(fields, uuid=uuid)

    @classmethod
    def get_by_name(cls, name, fields=None):
        """Returns a host by its FQDN."""
        return _get_with_query(fields, name=name)

    @classmethod
    def get_by_host_id_query(cls, host_id_query):
        try:
            return Host.objects.get(**host_id_query.kwargs())
        except mongoengine.DoesNotExist:
            raise HostNotFoundError()

    def deduce_profile_configuration(self, profile=None, profile_tags=None, profile_mode=None):
        """
        Gets a profile configuration obtained from user (possibly partial), checks it, merges with project's default
        profile configuration and returns a result profile configuration that should be used for host profiling.

        Notice: `profile_tags` are expected to be validated against `walle.views.api.common.get_eine_tags_schema()`.

        :raises MissingDefaultProfileConfigurationError
        """

        from walle.clients import inventory

        if self.project is None:
            raise LogicalError()

        project = self.get_project(fields=("profile", "profile_tags"))

        project_profile_tags = project.profile_tags or []
        if profile_tags is None:
            profile_tags = project_profile_tags
        else:
            profile_tags = set(profile_tags) | set(project_profile_tags)

        if profile is None:
            profile = project.profile or FLEXY_EINE_PROFILE

        else:
            if profile not in inventory.get_eine_profiles():
                raise InvalidProfileNameError(profile)

        if profile in EINE_PROFILES_WITH_DC_SUPPORT:
            if profile_mode is None:
                profile_modes = ProfileMode.get_modes(profile_tags)
            else:
                profile_modes, add_tags, exclude_tags = ProfileMode.get_configuration(profile, profile_mode)
                profile_tags = sorted(set(profile_tags) - exclude_tags | add_tags)
        else:
            if profile_mode is not None:
                raise Error(
                    "Logical error: An attempt to get profile configuration for {} with profile mode '{}' for '{}' "
                    "profile that doesn't support it.",
                    self.human_id(),
                    profile_mode,
                    profile,
                )

            profile_modes = None

        return ProfileConfiguration(profile, sorted(set(profile_tags)), profile_modes)

    def get_project_deploy_configuration(self):
        project = self.get_project(
            fields=(
                "provisioner",
                "deploy_config",
                "deploy_tags",
                "deploy_network",
                "certificate_deploy",
                "deploy_config_policy",
            )
        )

        return DeployConfiguration(
            provisioner=project.provisioner,
            config=project.deploy_config,
            tags=project.deploy_tags,
            certificate=project.certificate_deploy,
            ipxe=self.platform_ipxe_supported(),
            network=project.deploy_network or NetworkTarget.DEFAULT,
            deploy_config_policy=project.deploy_config_policy,
        )

    def get_aggregate(self):
        return get_aggregate_name(self.location.short_queue_name, self.location.rack)

    def get_deploy_configuration(self):
        provisioner, deploy_config = self.provisioner, self.config

        if (provisioner is None) ^ (deploy_config is None):
            raise LogicalError()

        # mixing provisioner and config from different sources is prohibited. Deploy config is a unit.
        if provisioner is not None and deploy_config is not None:
            return DeployConfiguration(
                provisioner=provisioner,
                config=deploy_config,
                tags=self.deploy_tags,
                certificate=self._need_deploy_certificate(),
                ipxe=self.platform_ipxe_supported(),
                network=self.deploy_network or NetworkTarget.DEFAULT,
                deploy_config_policy=self.deploy_config_policy,
            )
        elif provisioner is None and deploy_config is None:
            return self.get_project_deploy_configuration()
        else:
            raise LogicalError()

    def deduce_deploy_configuration(
        self,
        requested_provisioner=None,
        requested_config=None,
        requested_tags=None,
        requested_network=None,
        requested_deploy_config_policy=None,
    ):
        """
        Tries to figure out from input parameters whether to update host instance with a new deploy config
        or not, and what to pass to a deploy client.

        Returns a tuple of six parameters:
        (host_provisioner, host_config, host_tags, host_deploy_network, host_deploy_config_policy, DeployConfiguration).

        First five should be used to update the host instance in a database.
        DeployConfiguration should be passed to deploy stage.
        """
        if (self.provisioner is None) ^ (self.config is None):
            raise LogicalError()
        if self.deploy_tags is not None and self.config is None:
            raise LogicalError()

        if requested_provisioner is not None and requested_config is None:
            raise RequestValidationError("Provisioner must be specified with deploy config.")

        if requested_tags is not None and requested_config is None:
            raise RequestValidationError("Deploy config is required if deploy tags are specified.")

        if requested_network is None:
            if self._need_service_network_for_pxe():
                requested_network = NetworkTarget.SERVICE

        # something was requested => use requested deploy info, with fallback to host's and project's deploy info
        if requested_config is not None:
            current_deploy_config = self.get_deploy_configuration()
            deploy_config = DeployConfiguration(
                provisioner=requested_provisioner or current_deploy_config.provisioner,
                config=requested_config,
                tags=requested_tags or None,
                network=requested_network or NetworkTarget.DEFAULT,
                certificate=current_deploy_config.certificate,
                ipxe=self.platform_ipxe_supported(),
                deploy_config_policy=requested_deploy_config_policy,
            )

            host_provisioner = deploy_config.provisioner
            host_config = requested_config
            host_tags = deploy_config.tags
            host_network = requested_network
            host_deploy_config_policy = deploy_config.deploy_config_policy

            # Recursive imports :-(
            from walle.clients import inventory

            inventory.check_deploy_configuration(
                host_provisioner,
                host_config,
                self.get_eine_box(),
                host_tags,
                deploy_config.certificate,
                host_network,
                host_deploy_config_policy,
            )
        elif requested_network is not None:
            deploy_config = self.get_deploy_configuration()
            deploy_config = deploy_config._replace(network=requested_network)

            host_provisioner, host_config, host_tags, _, host_network, _, host_deploy_config_policy = deploy_config
        else:
            deploy_config = self.get_deploy_configuration()
            host_provisioner, host_config, host_tags, host_network, host_deploy_config_policy = (
                None,
                None,
                None,
                None,
                None,
            )

        return (
            # Values that are used to update Host in database
            host_provisioner,
            host_config,
            host_tags,
            host_network,
            host_deploy_config_policy,
            deploy_config,
        )

    def _need_deploy_certificate(self):
        return self.get_project(fields=("certificate_deploy",)).certificate_deploy

    def platform_ipxe_supported(self):
        platform = create_platform_for_host(self)
        return platform.ipxe_supported()

    def get_platform_eine_tags(self) -> list[str]:
        platform = create_platform_for_host(self)
        return platform.get_platform_specific_eine_tags()

    def _need_service_network_for_pxe(self):
        platform = create_platform_for_host(self)
        return platform.need_service_network_for_pxe()

    def human_id(self):
        return get_host_human_id(self.inv, self.name)

    def human_name(self):
        return get_host_human_name(self.inv, self.name)

    def _get_maintenance_author(self):
        return self.state_author if self.state == HostState.MAINTENANCE else self.status_author

    def is_maintenance(self, issuer=None):
        """If issuer is provided, then check if host is under maintenance set by another issuer.
        If issuer is None, then just check host status."""
        if issuer is None:
            return self.state == HostState.MAINTENANCE
        else:
            return self.state == HostState.MAINTENANCE and (self.state_author and self.state_author != issuer)

    def authorize(self, issuer, ignore_maintenance, allowed_host_types=None):
        if self.is_maintenance(issuer) and not ignore_maintenance:
            raise HostUnderMaintenanceError(self._get_maintenance_author())

        if self.type and allowed_host_types and self.type not in allowed_host_types:
            raise OperationRestrictedForHostTypeError(self.type, allowed_host_types)

        if has_iam():
            return

        project = self.get_project(fields=("id",))
        owners_and_users = project.owners or []
        for role in [ProjectRole.OWNER, ProjectRole.USER, ProjectRole.SUPERUSER]:
            owners_and_users.extend(ProjectRole.get_role_manager(role, project).list_members())
        # unlike Project.authorize_user, we don't count wall-e admins as owners
        blackbox.authorize(
            issuer,
            "You must be '{}' project user or owner to perform this request.".format(self.project),
            owners=set(owners_and_users),
        )

    def get_ipmi_client(self):
        if self.ipmi_mac is None:
            raise NonIpmiHostError()
        box_id = projects.get_eine_box(self.project)
        if box_id is None:
            provider = walle.clients.ipmiproxy.get_yandex_internal_provider(self.ipmi_mac)
        else:
            provider = walle.clients.ipmiproxy.get_yandex_box_provider(box_id, self.inv)

        return walle.clients.ipmiproxy.get_client(provider, self.human_id())

    def get_ssh_client(self):
        return walle.clients.ssh.get_client(self.name)

    def get_dns(self) -> dns_clients.interface.DnsClientInterface:
        project = self.get_project(fields=["id", "yc_dns_zone_id"])
        return project.get_dns()

    def get_cms(self, context_logger=None):
        project = self.get_project(fields=("id", "cms", "cms_api_version", "cms_tvm_app_id"))
        return project.get_cms(context_logger=context_logger)

    def get_cms_clients(self, context_logger=None, filter_condition=None):
        project = self.get_project(fields=("id", "cms_settings"))
        return project.get_cms_clients(context_logger=context_logger, filter_condition=filter_condition)

    def get_project(self, fields=None):
        # Protects us from using an object without required fields
        if self.project is None:
            raise LogicalError()

        try:
            return projects.get_by_id(self.project, fields=fields)
        except projects.ProjectNotFoundError:
            raise Error(
                "Internal error occurred: host {} belongs to unknown project {}.", self.human_id(), self.project
            )

    def get_eine_box(self):
        if self.project is None:
            raise LogicalError()

        return projects.get_eine_box(self.project)

    def applied_restrictions(self, *restrictions):
        """Return subset of given restrictions list that apply to this host.
        Empty input set produces empty output set.
        """

        restriction_set = set(restrictions)
        host_restrictions = set(self.restrictions) if self.restrictions else set()
        used_restrictions = host_restrictions & restriction_set

        if not used_restrictions:
            if self.type == HostType.MAC:
                used_restrictions = restriction_set - EXCLUDE_FOR_MACS

        return used_restrictions

    def get_active_mac(self):
        try:
            host_network = HostNetwork.get_by_uuid(self.uuid)
        except HostNetworkNotFoundError:
            return None
        if host_network.active_mac:
            return ActiveMacAddressInfo(
                host_network.active_mac, host_network.active_mac_source, host_network.active_mac_time
            )
        if self.macs and len(self.macs) == 1:
            return ActiveMacAddressInfo(self.macs[0], walle_constants.MAC_SOURCE_CONFIGURATION, 0)

    def get_current_network_location(self):
        try:
            host_network = HostNetwork.objects(
                uuid=self.uuid, network_timestamp__gte=timestamp() - _LOCATION_STALE_TIMEOUT
            ).get()
        except mongoengine.DoesNotExist:
            return None
        if host_network.network_switch is None or host_network.network_port is None:
            return None

        return host_network

    def matches_location(self, country=None, city=None, datacenter=None, queue=None, rack=None, unit=None):
        location = self.location
        if location is None:
            return False

        for field, value in (
            ("country", country),
            ("city", city),
            ("datacenter", datacenter),
            ("queue", queue),
            ("rack", rack),
            ("unit", unit),
        ):
            if value is None:
                continue

            values = (value,) if isinstance(value, str) else value
            if location[field] not in values:
                return False

        return True

    def set_state(self, state, issuer, audit_log_id, expire=None, reason=None):
        self.state = state
        self.state_time = timestamp()
        self.state_author = issuer
        self.state_audit_log_id = audit_log_id
        self.state_expire = expire
        self.state_reason = reason

    @staticmethod
    def set_state_kwargs(state, issuer, audit_log_id, expire=None, reason=None):
        return fix_mongo_set_kwargs(
            set__state=state,
            set__state_time=timestamp(),
            set__state_author=issuer,
            set__state_audit_log_id=audit_log_id,
            set__state_expire=expire,
            set__state_reason=reason,
        )

    def set_status(self, status, issuer, audit_log_id, reason=None, confirmed=True, downtime=None):
        # If healthy host status is not confirmed, we must take it into account to prevent considering the host as dead
        # by Expert System.

        self.status = status
        self.status_time = timestamp()
        self.status_author = issuer
        self.status_audit_log_id = audit_log_id

        if status == HostStatus.INVALID:
            del self.macs
            del self.active_mac
            del self.active_mac_source
            del self.on_downtime
            del self.health
            del self.checks_min_time

        if reason is None:
            del self.status_reason
        else:
            self.status_reason = reason

        if self.state in HostState.ALL_ASSIGNED and status == HostStatus.READY and not confirmed:
            self.checks_min_time = timestamp()

        if downtime is not None:
            if downtime:
                self.on_downtime = True
            else:
                del self.on_downtime

    @staticmethod
    def set_status_kwargs(
        state,
        status,
        issuer,
        audit_log_id,
        reason=None,
        ticket_key=None,
        confirmed=True,
        downtime=None,
        unset_ticket=False,
    ):
        """See set_status().

        :param state: may be None when not known, but only if status != HostStatus.READY.
        """

        if status == HostStatus.READY and state is None:
            raise LogicalError()

        kwargs = dict(
            set__status=status,
            set__status_time=timestamp(),
            set__status_author=issuer,
            set__status_audit_log_id=audit_log_id,
        )

        if status == HostStatus.INVALID:
            kwargs.update(
                unset__macs=True,
                unset__active_mac=True,
                unset__active_mac_source=True,
                unset__on_downtime=True,
                unset__health=True,
                unset__checks_min_time=True,
            )

        if unset_ticket:
            kwargs.update(unset__ticket=True)

        kwargs.update(
            fix_mongo_set_kwargs(
                set__status_reason=reason,
            )
        )

        if state in HostState.ALL_ASSIGNED and status == HostStatus.READY and not confirmed:
            kwargs.update(set__checks_min_time=timestamp()),

        if downtime is not None:
            if downtime:
                kwargs.update(set__on_downtime=True)
            elif state not in HostState.ALL_DOWNTIME:
                kwargs.update(unset__on_downtime=True)

        if ticket_key is not None:
            kwargs.update(set__ticket=ticket_key)

        return kwargs

    def set_messages(self, **kwargs):
        """Set messages for given modules. Arguments are:
        module_name=List[HostMessage]|None
        """
        return self.modify(**self.set_messages_kwargs(**kwargs))

    @staticmethod
    def set_messages_kwargs(**kwargs):
        """Create mongoengine update kwargs for messages for given modules. Arguments are:
        module_name=List[HostMessage]|None
        """
        message_update_kwargs = {}
        for module_name, messages in kwargs.items():
            if messages:
                message_update_kwargs["set__messages__{}".format(module_name)] = messages
            else:
                message_update_kwargs["unset__messages__{}".format(module_name)] = True

        return message_update_kwargs

    def may_cancel_task_for_higher_priority_failure(self):
        # NOTE(rocco66): we should not try to cancel task after an active phase of healing was started
        # TODO(rocco66): separate field for that
        # TODO(rocco66): disk hot swap support
        return self.get_iss_ban_flag()

    def get_iss_ban_flag(self):
        if self.task:
            return self.task.iss_banned
        return False

    def set_iss_ban_flag(self, flag=True):
        if self.task:
            self.task.iss_banned = flag

    def set_scenario(self, scenario_id):
        self.modify(set__scenario_id=scenario_id)

    def unset_scenario(self):
        self.modify(set__scenario_id=None)

    def in_scenario(self):
        return self.scenario_id is not None

    def get_task_id(self) -> tp.Optional[int]:
        try:
            return int(self.cms_task_id.split("-")[-1])
        except Exception as e:
            log.error("Host %s doesn't have cms_task_id: %s", self.uuid, e)
            return None

    def __str__(self):
        return "<Host: {}>".format(self.human_name())

    def __repr__(self):
        return self.__str__()


def get_host_human_id(inv, name=None, bot_format=False):
    if name is None:
        human_id = "#{inv}".format(inv=inv)
    else:
        if bot_format:
            human_id_format = "#{inv} ({name})"
        else:
            human_id_format = "{name} (#{inv})"

        human_id = human_id_format.format(inv=inv, name=name)

    return human_id


def get_host_human_name(inv, name=None):
    return "#{}".format(inv) if name is None else name


def sync_macs(host, host_network, macs):
    if host.macs == macs or (host.task and host.task.name == Operation.PROFILE):
        return False

    audit_params = {
        "new_macs": macs,
        "old_macs": host.macs,
        "old_active_mac": host_network.active_mac,
        "old_active_mac_time": host_network.active_mac_time,
        "old_active_mac_source": host_network.active_mac_source,
    }

    host_update_params = fix_mongo_set_kwargs(set__macs=macs)
    host_network_update_params = fix_mongo_set_kwargs()

    if macs and (host_network.active_mac is None or host_network.active_mac in macs):
        audit_params.update(
            new_active_mac=host_network.active_mac,
            new_active_mac_time=host_network.active_mac_time,
            new_active_mac_source=host_network.active_mac_source,
        )
        host_update_params.update(
            **fix_mongo_set_kwargs(set__active_mac=host.active_mac, set__active_mac_source=host.active_mac_source)
        )
        host_network_update_params.update(
            **fix_mongo_set_kwargs(
                set__active_mac=host_network.active_mac,
                set__active_mac_time=host_network.active_mac_time,
                set__active_mac_source=host_network.active_mac_source,
            )
        )
    else:
        audit_params.update(new_active_mac=None, new_active_mac_time=None, new_active_mac_source=None)
        host_update_params.update(unset__active_mac=True, unset__active_mac_source=True)
        host_network_update_params.update(
            unset__active_mac=True, unset__active_mac_time=True, unset__active_mac_source=True
        )

    if all(
        (
            HostNetwork.objects(uuid=host_network.uuid).update(multi=False, **host_network_update_params),
            Host.objects(inv=host.inv, state__in=HostState.MAC_SYNC_STATES, status__ne=HostStatus.INVALID).update(
                multi=False, **host_update_params
            ),
        )
    ):

        audit_log.on_host_macs_changed(host.project, host.inv, host.name, host.uuid, **audit_params).complete()

        if audit_params["old_active_mac"] != audit_params["new_active_mac"]:
            audit_log.on_active_mac_changed(
                host.project,
                host.inv,
                host.name,
                host.uuid,
                audit_params["new_active_mac"],
                audit_params["new_active_mac_time"],
                audit_params["new_active_mac_source"],
                audit_params["old_active_mac"],
                audit_params["old_active_mac_time"],
                audit_params["old_active_mac_source"],
            ).complete()

        return True

    return False


def update_snmp_active_mac(host, host_network, active_mac, actualization_time) -> bool:
    return _update_active_mac(host, host_network, active_mac, actualization_time, walle_constants.MAC_SOURCE_SNMP)


def update_agent_active_mac(host, host_network, active_mac, actualization_time) -> bool:
    return _update_active_mac(host, host_network, active_mac, actualization_time, walle_constants.MAC_SOURCE_AGENT)


def _get_eine_actual_border():
    return timestamp() - 12 * constants.HOUR_SECONDS


def update_eine_active_mac(host, host_network, active_mac, actualization_time) -> bool:
    # NOTE(rocco66): https://st.yandex-team.ru/WALLESUPPORT-1418 for details
    if actualization_time < _get_eine_actual_border():
        return False

    return _update_active_mac(host, host_network, active_mac, actualization_time, walle_constants.MAC_SOURCE_EINE)


def update_racktables_active_mac(host, host_network, active_mac, actualization_time) -> bool:
    if not config.get_value("racktables.netmap_sync.enabled"):
        return False
    # Pessimize RackTables as unreliable data source to always prefer more recent data from Eine/LLDP if available.
    actualization_time = min(actualization_time, timestamp() - constants.HOUR_SECONDS)

    # NOTE(rocco66): https://st.yandex-team.ru/WALLESUPPORT-1418 for details
    replace_eine_filter = Q(active_mac_source=walle_constants.MAC_SOURCE_EINE) & Q(
        active_mac_time__lt=_get_eine_actual_border()
    )
    replace_non_eine_filter = Q(active_mac_source__ne=walle_constants.MAC_SOURCE_EINE) & Q(
        active_mac_time__lt=actualization_time
    )
    update_filter = Q(active_mac_time__exists=False) | replace_eine_filter | replace_non_eine_filter
    return _update_active_mac(
        host, host_network, active_mac, actualization_time, walle_constants.MAC_SOURCE_RACKTABLES, update_filter
    )


def _update_active_mac(
    host, host_network, active_mac, actualization_time, source, host_network_update_filter=None
) -> bool:
    if host.status == HostStatus.INVALID:
        return False

    if host_network.active_mac_time is not None and host_network.active_mac_time >= actualization_time:
        return False

    if host_network_update_filter is None:
        host_network_update_filter = Q(active_mac_time__exists=False) | Q(active_mac_time__lt=actualization_time)

    # this returns the OLD version of host
    if not HostNetwork.objects(host_network_update_filter, uuid=host.uuid).modify(
        **fix_mongo_set_kwargs(
            set__active_mac=active_mac,
            set__active_mac_time=actualization_time,
            set__active_mac_source=source,
        )
    ):
        return False

    Host.objects(inv=host.inv, macs=active_mac).modify(
        **fix_mongo_set_kwargs(
            set__active_mac=active_mac,
            set__active_mac_source=source,
        )
    )

    # Notice: Prefer cheap host updating here to reliability of audit log since these audit log entries not so critical
    # and mostly intended for analytics instead of blaming someone.
    if host_network.active_mac == active_mac:
        return False

    audit_log.on_active_mac_changed(
        host.project,
        host.inv,
        host.name,
        host.uuid,
        active_mac,
        actualization_time,
        source,
        host_network.active_mac,
        host_network.active_mac_time,
        host_network.active_mac_source,
    ).complete()

    if active_mac:
        default_credit_timeout = constants.MINUTE_SECONDS * 20
        walle.expert.automation.PROJECT_DNS_AUTOMATION.increase_credit(
            host.project,
            projects.MAX_DNS_FIXES_LIMIT,
            credit_time=default_credit_timeout,
            reason="Host {} active mac changed".format(host.human_name()),
        )

    return True


def update_network_location(host, host_network, switch, port, actualization_time, source):
    # Pessimize RackTables as unreliable data source to always prefer data from Eine/LLDP

    if host.status == HostStatus.INVALID:
        return False

    if source == NETWORK_SOURCE_RACKTABLES:
        actualization_time = min(actualization_time, timestamp() - constants.HOUR_SECONDS)

    if host_network.network_timestamp is not None and host_network.network_timestamp >= actualization_time:
        return False

    switch_port_changed = (switch, port) != (host_network.network_switch, host_network.network_port)

    # WALLE-3855 Do not set switch and port to 'unknown' when LLDP on host fails to get them.
    if source == NETWORK_SOURCE_LLDP and switch_port_changed and switch == "unknown" and port == "unknown":
        return False

    if source == NETWORK_SOURCE_RACKTABLES and not config.get_value("racktables.netmap_sync.enabled"):
        _DELTA_ACTUALIZATION_TIME = constants.DAY_SECONDS + constants.HOUR_SECONDS
        if (
            host_network.network_source != NETWORK_SOURCE_RACKTABLES
            and switch_port_changed
            and (
                host_network.network_timestamp
                >= actualization_time
                >= (host_network.network_timestamp - _DELTA_ACTUALIZATION_TIME)
                or actualization_time
                >= host_network.network_timestamp
                >= actualization_time - _DELTA_ACTUALIZATION_TIME
            )
        ):
            log.error(
                "Network location for #%s from %s (%s) doesn't match location from %s (%s).",
                host.inv,
                NETWORK_SOURCE_RACKTABLES,
                "{}/{} at {}".format(switch, port, format_time(actualization_time)),
                host_network.network_source,
                "{}/{} at {}".format(
                    host_network.switch, host_network.network_port, format_time(host_network.network_timestamp)
                ),
            )
        return False

    if not HostNetwork.objects(
        Q(network_timestamp__exists=False) | Q(network_timestamp__lt=actualization_time), uuid=host.uuid
    ).modify(
        set__network_switch=switch,
        set__network_port=port,
        set__network_source=source,
        set__network_timestamp=actualization_time,
    ):
        return False

    Host.objects(inv=host.inv, status__ne=HostStatus.INVALID).modify(
        set__location__switch=switch, set__location__port=port, set__location__network_source=source
    )

    # Notice: Prefer cheap host updating here to reliability of audit log since these audit log entries not so critical
    # and mostly intended for analytics instead of blaming someone.
    if (host_network.network_switch, host_network.network_port) != (switch, port):
        audit_log.on_switch_port_changed(
            host.project,
            host.inv,
            host.name,
            host.uuid,
            switch,
            port,
            actualization_time,
            source,
            host_network.network_switch,
            host_network.network_port,
            host_network.network_timestamp,
            host_network.network_source,
        ).complete()
        return True
    else:
        return False


def set_health(inv, old_event_id, health):
    """This function uses pymongo to set host's health because mongoengine is very, very slow."""
    inv_field = Host.inv.db_field
    health_field = Host.health.db_field

    query = {inv_field: inv}
    if old_event_id is None:
        query[health_field] = {"$exists": False}
    else:
        field = ".".join([health_field, HealthStatus.event_id.db_field])
        query[field] = old_event_id

    if health is None:
        update = {"$unset": {health_field: True}}
    else:
        update = {"$set": {health_field: health}}

    response = Host.get_collection().find_and_modify(query, update, new=True, fields=(inv_field, health_field))
    return bool(response)


def get_host_query(
    issuer, ignore_maintenance, allowed_states, allowed_statuses=None, forbidden_statuses=None, **kwargs
):
    query = dict()
    if isinstance(allowed_states, (tuple, list)):
        query.update(state__in=allowed_states)
    else:
        query.update(state=allowed_states)

    if allowed_statuses is not None:
        query.update(status__in=allowed_statuses)

    if forbidden_statuses is not None:
        query.update(status__nin=forbidden_statuses)

    query = Q(**query)
    if not ignore_maintenance:
        query &= Q(state__ne=HostState.MAINTENANCE) | Q(state_author=issuer) | Q(state_author__exists=False)

    # this query should be passed to QuerySet as **query or to .modify as an object.
    return dict(kwargs, q_obj=query)


def deploy_configuration(
    provisioner,
    config,
    tags=None,
    certificate=False,
    network=NetworkTarget.DEFAULT,
    ipxe=True,
    deploy_config_policy=None,
):
    """Constructor for DeployConfiguration with default (backward compatible) values for new parameters.

    Use this function to construct DeployConfiguration object from existing data that comes from sources
    that can lag behind the most recent changes (like stage data).
    """
    # KOSTYLI: some hosts still have an old format of DeployConfiguration serialized in db, so we have to convert them
    if deploy_config_policy == [None]:
        deploy_config_policy = None

    return DeployConfiguration(provisioner, config, tags, certificate, network, ipxe, deploy_config_policy)


def _get_with_query(fields, **query):
    objects = Host.objects(**query)
    if fields is not None:
        objects = objects.only(*fields)

    try:
        return objects.get()
    except mongoengine.DoesNotExist:
        raise HostNotFoundError()


def find_hosts_with_stage(stage_name):
    def has_stage(stages, stage_name):
        for stage in stages:
            if stage.name == stage_name:
                return True
            if stage.stages:
                if has_stage(stage.stages, stage_name):
                    return True
        return False

    hosts = []
    for host in Host.objects(task__exists=True):
        if has_stage(host.task.stages, stage_name):
            hosts.append(host)
    return hosts


def generate_external_inventory_number():
    inv = MAX_BOT_INVENTORY_NUMBER + 1
    host = Host.get_collection().find_one({}, {Host.inv.db_field: True}, sort=[(Host.inv.db_field, DESCENDING)])
    if host is None or host["inv"] < inv:
        return inv
    return host["inv"] + 1


@register_model
class DMCRule(Document):
    """Represents a rule for dmc."""

    id = LongField(primary_key=True, required=True, help_text="Rule ID (increments automatically)")
    rule_query = DictField(help_text="Dictionary with query")

    api_fields = (
        "id",
        "rule_query",
    )

    meta = {
        "collection": "dmc_rules",
    }

    def __str__(self):
        return "<DMC rule #{}>".format(self.id)

    @classmethod
    def with_fields(cls, fields: tp.Optional[list[str]]) -> sepelib.mongo.document.QuerySet:
        return cls.objects.only(*cls.api_query_fields(fields or None))

    @staticmethod
    def next_id() -> int:
        return monotonic.get_next("dmc_rule_id")


def get_raw_query_for_dmc_rules():
    if DMCRule.objects.count():
        dmc_raw_rules = {"$or": [transform_to_raw_query(rule.rule_query) for rule in DMCRule.objects()]}
    else:
        dmc_raw_rules = {}
    return dmc_raw_rules


def transform_to_raw_query(rule_query):
    raw_query = {
        "$and": [
            parse_physical_location_query(rule_query.get("physical_locations_included")),
            parse_physical_location_query(rule_query.get("physical_locations_excluded"), exp="$nor"),
            parse_tier_id_query(rule_query.get("tiers_included")),
            parse_tier_id_query(rule_query.get("tiers_excluded"), exp="$nor"),
            parse_project_id_query(rule_query.get("project_ids_included")),
            parse_project_id_query(rule_query.get("project_ids_excluded"), exp="$nor"),
        ]
    }
    return raw_query


def parse_physical_location_query(physical_locations, exp="$or"):
    def full_field_name(name):
        return Host.location.db_field + "." + getattr(HostLocation, name).db_field

    if not physical_locations:
        return {}

    match_locations_expressions = []

    for physical_location in physical_locations:
        try:
            conditions = [
                {full_field_name(field_name): field_value}
                for field_name, field_value in HostLocation.split_physical_location_string_to_fields(
                    physical_location
                ).items()
            ]
        except ValueError as exc:
            raise BadRequestError(exc)

        if conditions:
            match_locations_expressions.append({"$and": conditions})

    if match_locations_expressions:
        return {exp: match_locations_expressions}
    else:
        return {}


def parse_tier_id_query(tiers, exp="$or"):
    if not tiers:
        return {}

    match_locations_expressions = []

    for tier in tiers:
        match_locations_expressions.append({Host.tier.db_field: tier})

    if match_locations_expressions:
        return {exp: match_locations_expressions}
    else:
        return {}


def parse_project_id_query(project_ids, exp="$or"):
    if not project_ids:
        return {}

    match_locations_expressions = []

    for project_id in project_ids:
        match_locations_expressions.append({Host.project.db_field: project_id})

    if match_locations_expressions:
        return {exp: match_locations_expressions}
    else:
        return {}
