import collections
from copy import deepcopy

from sepelib.core.exceptions import Error


class MissingField(Error):
    def __init__(self, field):
        super().__init__("Required field {} is missing".format(field))


class WrongNodeType(Error):
    def __init__(self, node):
        super().__init__("Node '{}' has wrong type".format(node))


class IdmNodeView(collections.UserDict):  # UserDict is an old-style class
    required_fields = ()

    def to_dict(self):
        raise NotImplementedError()

    def validate(self, node_path=None):
        for field in self.required_fields:
            if field not in self.data:
                raise MissingField(field)


class RoleNodeView(IdmNodeView):
    required_fields = ("slug", "name", "values")

    def to_dict(self):
        d = deepcopy(self.data)
        for val_name, val_node in self.get_values():
            d["values"][val_name] = val_node.to_dict()
        return d

    def add_value(self, value_name, value_node):
        if not isinstance(value_node, ValueNodeView):
            raise WrongNodeType(value_node)

        # Role can have multiple values
        self.data.setdefault("values", {})
        self.data["values"][value_name] = value_node

    def get_values(self):
        return self.data["values"].items()

    def validate(self, node_path=None):
        super().validate()
        for value_name, value_node in self.get_values():
            value_node.validate()


class ValueNodeView(IdmNodeView):
    required_fields = ("name",)

    def to_dict(self):
        d = deepcopy(self.data)
        if "roles" in self.data:
            d["roles"] = self.get_role().to_dict()
        return d

    def set_role(self, role_node):
        if not isinstance(role_node, RoleNodeView):
            raise WrongNodeType(role_node)

        # Value can have only one role
        self.data["roles"] = role_node

    def get_role(self):
        return self.data.get("roles")

    def validate(self, node_path=None):
        super().validate()
        role_node = self.get_role()
        if role_node is not None:
            role_node.validate()

    def set_name(self, name):
        self.data["name"] = name

    def set_help(self, help):
        self.data["help"] = help

    def set_visibility(self, visible):
        self.data["visibility"] = visible

    def set_set_attribute(self, value):
        self.data["set"] = value


def push_api_node_view(node):
    """Push API doesn't make distinction between role and value nodes, all of them must have a slug"""
    data = dict(node.get_idm_properties())
    if "slug" not in data:  # value nodes don't export their name as a slug
        data["slug"] = node.slug
    return data
