import logging

from sepelib.core.exceptions import Error
from walle import projects
from walle.clients import idm, staff
from walle.idm import traversal
from walle.idm.node_views import push_api_node_view
from walle.idm.project_role_managers import get_project_idm_role_prefix
from walle.constants import HostType

log = logging.getLogger(__name__)


class IDMPushException(Error):
    def __init__(self, message, *args):
        super().__init__("Error during IDM push: " + message, *args)


def add_project_role_tree_nodes(batch, project):
    log.info("Pushing nodes of project %s into IDM role tree", project.id)
    for path, node in traversal.iter_subtree_nodes(get_project_idm_role_prefix(project.id)):
        parent_path = path[:-1]
        node_dict = push_api_node_view(node)
        log.info("Pushing node %s (%s)", path, node_dict)
        batch.add_role_node(parent_path, node_dict)


def delete_project_role_tree_nodes(batch, project):
    log.info("Deleting nodes of project %s from IDM role tree", project.id)
    batch.remove_role_node(get_project_idm_role_prefix(project.id))


def revoke_project_roles(batch, project):
    for role_type in ["active", "requested"]:
        roles = get_project_roles(project, type=role_type)
        log.info("Revoking %s roles of project %s: %s", role_type, project.id, roles)
        for role in roles:
            batch.revoke_role(role.id)


def request_project_role_clones(batch, project, orig_project_id):
    log.info("Requesting roles for project %s (clone of project %s)", project.id, orig_project_id)
    orig_project = projects.get_by_id(orig_project_id)
    for orig_role in get_project_roles(orig_project, type="active"):
        _request_role_clone_for_project(batch, orig_role, orig_project_id, project)


def get_project_roles(project, **query_args):
    if project.type == HostType.SHADOW_SERVER:
        return []
    else:
        return idm.iter_roles(path_prefix=get_project_idm_role_prefix(project.id), **query_args)


def _request_role_clone_for_project(batch, orig_role, orig_project_id, project):
    path = list(orig_role.path)
    path[path.index(orig_project_id)] = project.id

    if orig_role.is_group:
        return batch.request_role(path, group=staff.group(orig_role.member))
    else:
        return batch.request_role(path, user=orig_role.member)
