import abc
import re
from collections import defaultdict, namedtuple

from walle import audit_log
from walle.clients import idm, staff
from walle.constants import ROBOT_WALLE_OWNER
from walle.errors import BadRequestError
from walle.idm import role_storage


class ProjectRoleManagerBase(metaclass=abc.ABCMeta):
    @abc.abstractproperty
    def role_name(self):
        """name of the role (leaf node)"""
        raise NotImplementedError()

    @abc.abstractproperty
    def storage_strategy(self):
        """Class that implements the way info about membership is stored
        Old roles store members in Project fields, new ones will store in common storage
        :type: type[IMembershipStorage]
        """
        raise NotImplementedError()

    @abc.abstractproperty
    def member_processing_strategy(self):
        """Class that checks or sets member
        Roles that have some fixed set of users (reboot_via_ssh/noc/exp) will set their virtual user here,
        roles dealing with real users will not do anything
        :type: type[IMemberProcessor]
        """
        raise NotImplementedError()

    @abc.abstractproperty
    def audit_log_strategy(self):
        """Class that writes info to audit log
        :type: type[IAuditLogWriter]
        """
        raise NotImplementedError()

    def __init__(self, project):
        self._project = project
        self._role_path = get_project_idm_role_prefix(project.id) + ["role", self.role_name]
        self._storage = self.storage_strategy(self._project, self._role_path)
        self._member_processor = self.member_processing_strategy()
        self.audit_log_writer = self.audit_log_strategy(self._project, self.role_name)

    def request_add_member(self, batch, issuer_login, member=None, with_autoapprove=False):
        """Request role membership in IDM
        Member can be None for role types which can have only one predefined member
        """
        member = self._member_processor.process_member(member)
        requester = self._member_processor.process_requester(issuer_login)

        if member in self.list_members():
            return

        request_args = {
            "path": self._role_path,
        }
        if not with_autoapprove:
            request_args["_requester"] = requester
        if staff.is_group(member):
            request_args["group"] = member
        else:
            request_args["user"] = member

        batch.request_role(**request_args)

    def request_remove_member(self, batch, member=None):
        """Request role membership revocation in IDM
        Member can be None for role types which can have only one predefined member
        """
        member = self._member_processor.process_member(member)

        if member not in self.list_members():
            return

        if staff.is_group(member):
            group_id = staff.get_group_id(member)
            role = idm.get_role(self._role_path, group_id=group_id, type="active")
        else:
            role = idm.get_role(self._role_path, user=member, type="active")

        if role is not None:
            batch.revoke_role(role.id)

    def list_members(self):
        """List members that already have this role"""
        return self._storage.list_members()

    def add_member(self, member):
        """Used by IDM role leaf nodes to actually add member"""
        if member not in self.list_members():
            self._storage.add_member(member)

    def remove_member(self, member):
        """Used by IDM role leaf nodes to actually remove member"""
        if member in self.list_members():
            self._storage.remove_member(member)

    def list_requested_members(self):
        """List members whose membership was requested but it still not approved"""
        return list(idm.iter_roles(path_prefix=self._role_path, type="requested"))

    def list_revoking_members(self):
        """List members whose membership is in process of revocation"""
        return list(idm.iter_roles(path_prefix=self._role_path, state=["depriving", "depriving_validation"]))


class IMembershipStorage:
    def __init__(self, project, role_path):
        self.project = project
        self.role_path = role_path

    def list_members(self):
        raise NotImplementedError()

    def add_member(self, member):
        raise NotImplementedError()

    def remove_member(self, member):
        raise NotImplementedError()


class CommonStorage(IMembershipStorage):
    """Use common role membership storage"""

    def list_members(self):
        return role_storage.get_role_members(self.role_path)

    def add_member(self, member):
        role_storage.add_role_member(self.role_path, member)

    def remove_member(self, member):
        role_storage.remove_role_member(self.role_path, member)


class IMemberProcessor:
    def process_member(self, member):
        """Can validate or modify member"""
        raise NotImplementedError()

    def process_requester(self, issuer_login):
        """Can set role requester"""
        raise NotImplementedError()


class RealUserProcessor(IMemberProcessor):
    def process_member(self, member):
        if member is None:
            raise BadRequestError("Member name (group or user) must be provided")
        return member

    def process_requester(self, issuer_login):
        return issuer_login


class FixedMemberProcessor(IMemberProcessor):
    """Some roles are used as feature flags: they can be granted only to one fixed user,
    and their approval means that some feature is enabled for the project
    If the initial requester was successfully authorized, such requests must be auto-approved, that's why requester
    is set to robot-walle (auto-approval is handled in IDM workflow)
    """

    @abc.abstractproperty
    def _fixed_member(self):
        raise NotImplementedError()

    def process_member(self, member):
        if member is not None and member != self._fixed_member:
            raise BadRequestError("This role can be only given to/revoked from {}".format(self._fixed_member))
        return self._fixed_member

    def process_requester(self, issuer_login):
        return ROBOT_WALLE_OWNER


def gen_fixed_member_processor_strategy(fixed_member):
    class ConcreteFixedMemberStrategy(FixedMemberProcessor):
        _fixed_member = fixed_member

    return ConcreteFixedMemberStrategy


class IAuditlogWriter:
    def __init__(self, project, role):
        self.project = project
        self.role = role

    def on_request_add_member(self, member, issuer, reason):
        raise NotImplementedError()

    def on_request_remove_member(self, member, issuer, reason):
        raise NotImplementedError()

    def on_add_member(self, member):
        raise NotImplementedError()

    def on_remove_member(self, member):
        raise NotImplementedError()


class CommonProjectRoleChange(IAuditlogWriter):
    """Uses common audit log types"""

    def on_request_add_member(self, member, issuer, reason):
        return audit_log.on_project_role_update_request(issuer, self.project.id, self.role, {"add": member}, reason)

    def on_request_remove_member(self, member, issuer, reason):
        return audit_log.on_project_role_update_request(issuer, self.project.id, self.role, {"remove": member}, reason)

    def on_add_member(self, member):
        return audit_log.on_project_role_add_member(self.project.id, self.role, member)

    def on_remove_member(self, member):
        return audit_log.on_project_role_remove_member(self.project.id, self.role, member)


class SshRebooterRoleChange(IAuditlogWriter):
    """Uses reboot_via_ssh audit log types"""

    def on_request_add_member(self, member, issuer, reason):
        return audit_log.on_project_request_toggle_reboot_via_ssh(
            issuer, self.project.id, is_enabled=True, reason=reason
        )

    def on_request_remove_member(self, member, issuer, reason):
        return audit_log.on_project_request_toggle_reboot_via_ssh(
            issuer, self.project.id, is_enabled=False, reason=reason
        )

    def on_add_member(self, member):
        return audit_log.on_project_toggle_reboot_via_ssh(self.project.id, is_enabled=True)

    def on_remove_member(self, member):
        return audit_log.on_project_toggle_reboot_via_ssh(self.project.id, is_enabled=False)


class OwnerRoleChange(IAuditlogWriter):
    def on_request_add_member(self, member, issuer, reason):
        return audit_log.on_project_owners_update_request(issuer, self.project.id, {"owners": {"add": member}}, reason)

    def on_request_remove_member(self, member, issuer, reason):
        return audit_log.on_project_owners_update_request(
            issuer, self.project.id, {"owners": {"remove": member}}, reason
        )

    def on_add_member(self, member):
        return audit_log.on_project_add_owner(self.project.id, member)

    def on_remove_member(self, member):
        return audit_log.on_project_remove_owner(self.project.id, member)


class OwnerManager(ProjectRoleManagerBase):
    role_name = "owner"
    storage_strategy = CommonStorage
    member_processing_strategy = RealUserProcessor
    audit_log_strategy = OwnerRoleChange


class UserManager(ProjectRoleManagerBase):
    role_name = "user"
    storage_strategy = CommonStorage
    member_processing_strategy = RealUserProcessor
    audit_log_strategy = CommonProjectRoleChange


class NocAccessManager(ProjectRoleManagerBase):
    role_name = "noc_access"
    storage_strategy = CommonStorage
    member_processing_strategy = RealUserProcessor
    audit_log_strategy = CommonProjectRoleChange


class SshRebooterManager(ProjectRoleManagerBase):
    role_name = "ssh_rebooter"
    storage_strategy = CommonStorage
    member_processing_strategy = gen_fixed_member_processor_strategy(ROBOT_WALLE_OWNER)
    audit_log_strategy = SshRebooterRoleChange


class SuperuserManager(ProjectRoleManagerBase):
    """Just like user, but with nopasswd sudo"""

    role_name = "superuser"
    storage_strategy = CommonStorage
    member_processing_strategy = RealUserProcessor
    audit_log_strategy = CommonProjectRoleChange


class ProjectRole:
    OWNER = "owner"
    USER = "user"
    SUPERUSER = "superuser"
    NOC_ACCESS = "noc_access"
    SSH_REBOOTER = "ssh_rebooter"

    _alias_to_manager_class = {
        OWNER: OwnerManager,
        USER: UserManager,
        SUPERUSER: SuperuserManager,
        NOC_ACCESS: NocAccessManager,
        SSH_REBOOTER: SshRebooterManager,
    }

    ALL = list(_alias_to_manager_class.keys())

    @classmethod
    def get_role_manager(cls, alias, project):
        """
        :rtype: ProjectRoleManagerBase
        """
        return cls._alias_to_manager_class[alias](project)


def get_project_idm_role_prefix(project_id):
    return ["scopes", "project", "project", project_id]


def get_projects_roles_members(project_ids=None):
    """
    Extracts members of all roles of passed projects
    :param project_ids: list of project ids or None (means all projects)
    :return: aggregated projects with their roles [{role: [members]}]
    """
    project_id_to_roles = defaultdict(lambda: {role: [] for role in ProjectRole.ALL})
    for project_id, role, member in iter_project_roles_memberships(project_ids):
        project_id_to_roles[project_id][role].append(member)
    return project_id_to_roles


_ProjectRolePath = namedtuple("ProjectRolePath", "scope scope_name project project_id role role_name")


def iter_project_roles_memberships(project_ids=None, **query_args):
    for membership in role_storage.IDMRoleMembership.objects(
        path=get_project_role_path_prefix_re(project_ids), **query_args
    ):
        path = _ProjectRolePath(*membership.path_components)
        yield path.project_id, path.role_name, membership.member


def get_project_role_path_prefix_re(project_ids=None):
    path_prefix_re = r"^/scopes/project/project/"  # mongo uses index to find matches with regex prefix
    if project_ids is not None:  # narrow the search with specific projects
        path_prefix_re += r"({})/".format("|".join(project_ids))
    return re.compile(path_prefix_re)
