import logging
from collections import defaultdict

from sepelib.core.exceptions import Error
from walle.clients import bot, staff, juggler
from walle.clients.juggler import JUGGLER_MSG_MAX_SIZE
from walle.util.text import shrink_string

log = logging.getLogger(__name__)


class _ConversionItem:
    """Conversion works as pipeline, here we store the results of item conversion steps"""

    bot_project_id = None
    planner_id = None
    staff_id = None

    def __init__(self, bot_project_id):
        self.bot_project_id = bot_project_id

    def __repr__(self):
        return "<CI bot_project_id={} planner_id={} staff_id={}>".format(
            self.bot_project_id, self.planner_id, self.staff_id
        )


class MissingItem(Error):
    pass


class ProjectStaffIdConverter:
    """Bulk converts projects' bot project ids to staff ids
    Staff id is needed to get project's department's heads in IDM workflow
    Conversion is done from project.bot_project_id to planner_id to staff id
    Bulk conversion is used to speed up IDM role tree generation
    """

    def __init__(self, projects):
        # multiple projects can have same bot project id
        self._bpi_to_projects = defaultdict(list)
        for project in projects:
            self._bpi_to_projects[project.bot_project_id].append(project.id)

        # bot project id -> _ConversionItem
        self._bpi_to_conv_item = {bpi: _ConversionItem(bpi) for bpi in self._bpi_to_projects.keys()}
        # bot project id -> error msg
        self._bpi_to_error = {}

        self._convert_bpis_to_staff_ids()

        if self._bpi_to_error:
            self._send_errors_to_juggler(self._bpi_to_error.values())

    def get_staff_id(self, project):
        if project.bot_project_id in self.get_bot_project_ids_with_errors():
            raise MissingItem("Project {}: {}", project.id, self._bpi_to_error[project.bot_project_id])

        return self._bpi_to_conv_item[project.bot_project_id].staff_id

    def get_bot_project_ids_with_errors(self):
        return self._bpi_to_error

    def _convert_bpis_to_staff_ids(self):
        self._fill_planner_ids()
        self._fill_staff_ids()

    def _fill_planner_ids(self):
        oebs_projects = bot.get_oebs_projects()
        for item in list(self._bpi_to_conv_item.values()):
            try:
                item.planner_id = oebs_projects[item.bot_project_id]["planner_id"]
            except KeyError:
                self._handle_item_missing_in_bot(item.bot_project_id)

    def _fill_staff_ids(self):
        planner_ids = {item.planner_id: item for item in self._bpi_to_conv_item.values()}
        planner_ids_to_staff_ids = {
            g["service"]["id"]: g["id"] for g in staff.groups_by_planner_ids(list(planner_ids.keys()))
        }

        if len(planner_ids_to_staff_ids) != len(planner_ids):
            for missing_planner_id in set(planner_ids) - set(planner_ids_to_staff_ids):
                self._handle_item_missing_in_staff(planner_ids[missing_planner_id])

        for item in self._bpi_to_conv_item.values():
            item.staff_id = planner_ids_to_staff_ids[item.planner_id]

    def _handle_item_missing_in_bot(self, bpi):
        self._bpi_to_conv_item.pop(bpi)

        projs_repr = self._proj_repr_by_bpi(bpi)

        self._bpi_to_error[bpi] = "{}: Bot project id {} was not found in BOT".format(projs_repr, bpi)
        log.error("%s: Bot project id %s was not found in BOT", projs_repr, bpi)

    def _handle_item_missing_in_staff(self, item):
        self._bpi_to_conv_item.pop(item.bot_project_id)

        projs_repr = self._proj_repr_by_bpi(item.bot_project_id)

        error_msg = "{}: Planner id {} (bot project id {}) was not found in Staff".format(
            projs_repr, item.planner_id, item.bot_project_id
        )
        self._bpi_to_error[item.bot_project_id] = error_msg

        log.error(
            "%s: Planner id %s (bot project id %s) was not found in Staff",
            projs_repr,
            item.planner_id,
            item.bot_project_id,
        )

    def _proj_repr_by_bpi(self, bpi):
        projects_by_bpi = self._bpi_to_projects[bpi]
        if len(projects_by_bpi) == 1:
            return "Project {}".format(projects_by_bpi[0])
        else:
            return "Projects {}".format(", ".join(projects_by_bpi))

    @staticmethod
    def _send_errors_to_juggler(errors):
        msg = shrink_string('\n'.join(errors), JUGGLER_MSG_MAX_SIZE)
        juggler.send_event("wall-e-project-staff-id-conversion", juggler.JugglerCheckStatus.CRIT, msg)
