from mongoengine import StringField, ListField

from sepelib.core.exceptions import LogicalError
from sepelib.mongo.util import register_model
from walle.idm.common import idm_list_path_to_string, idm_string_path_to_list
from walle.models import Document


@register_model
class IDMRoleMembership(Document):
    path = StringField(required=True, help_text="Role path")
    path_components = ListField(StringField(), required=False, help_text="Role path components")
    member = StringField(required=True, help_text="Role member (user or @group)")

    meta = {
        "indexes": [
            {"name": "path", "fields": ["path"]},
            {"name": "member", "fields": ["member"]},
            {"name": "role_membership", "fields": ["path", "member"], "unique": True},
        ]
    }

    api_fields = ["path", "member"]


def get_role_members(path):
    path = _path_as_string(path)
    return IDMRoleMembership.objects(path=path).distinct("member")


def is_role_member(path, member):
    path = _path_as_string(path)
    return bool(IDMRoleMembership.objects(path=path, member=member).count())


def add_role_member(path, member):
    path, path_components = _convert_path(path)
    role_member = IDMRoleMembership(path=path, path_components=path_components, member=member)
    role_member.save()


def remove_all_roles_for_project(project):
    IDMRoleMembership.objects(path_components__2="project", path_components__3=project.id).delete()


def remove_role_member(path, member):
    path = _path_as_string(path)
    IDMRoleMembership.objects(path=path, member=member).delete()


def _path_as_string(path):
    if isinstance(path, (list, tuple)):
        return idm_list_path_to_string(path)
    return path


def _convert_path(path):
    """
    Convert path to both file-like path and list of its components
    :rtype: tuple(str, list[str])
    """
    if isinstance(path, (list, tuple)):
        path, path_components = idm_list_path_to_string(path), path
    elif isinstance(path, str):
        path, path_components = path, idm_string_path_to_list(path)
    else:
        raise LogicalError()

    return path, path_components
