import abc
import logging

from sepelib.core.exceptions import Error
from walle import projects
from walle.constants import ROBOT_WALLE_OWNER
from walle.idm.project_role_managers import (
    UserManager,
    NocAccessManager,
    SshRebooterManager,
    SuperuserManager,
    OwnerManager,
)
from walle.idm.project_staff_id_converter import ProjectStaffIdConverter, MissingItem
from walle.idm.role_storage import get_role_members, remove_role_member, add_role_member
from walle.scenario.script import ScriptRegistry
from walle.util.gevent_tools import gevent_idle_iter

logger = logging.getLogger(__name__)


class ChildNotFound(Error):
    def __init__(self, child_name):
        self.child_name = child_name
        super().__init__("Child node {} not found", child_name)


class WrongMember(Error):
    def __init__(self, member):
        super().__init__("Wrong member {} passed (only {} is allowed)", member, ROBOT_WALLE_OWNER)


class ITreeNode:
    slug = None

    def list_children(self):
        raise NotImplementedError()

    def get_idm_properties(self):
        raise NotImplementedError()


class IInnerNode(ITreeNode):
    def get_child(self, name):
        raise NotImplementedError()


class ILeafNode(ITreeNode):
    def list_children(self):
        return []

    def list_role_members(self):
        raise NotImplementedError()

    def add_role_member(self, member):
        raise NotImplementedError()

    def remove_role_member(self, member):
        raise NotImplementedError()


class BaseInnerNode(IInnerNode):
    # defines basic methods for accessing children and providing idm properties
    _children = []
    _idm_props = {}

    def __init__(self):
        self._slug_to_child = {ch.slug: ch for ch in self.list_children()}

    def list_children(self):
        return self._children

    def get_child(self, name):
        try:
            return self._slug_to_child[name]
        except KeyError:
            raise ChildNotFound(name)

    def get_idm_properties(self):
        return self._idm_props


class RootNode(BaseInnerNode):
    slug = "scopes"

    def __init__(self):
        self._children = [ProjectValueNode(), ScenarioValueNode()]
        self._idm_props = {"slug": self.slug, "name": "Scopes"}
        super().__init__()


class ProjectValueNode(BaseInnerNode):
    slug = "project"

    def __init__(self):
        self._children = [ProjectRoleNode()]
        self._idm_props = {"name": self.slug}
        super().__init__()


class ProjectRoleNode(BaseInnerNode):
    slug = "project"
    _idm_props = {"slug": slug, "name": "Project"}

    def __init__(self):
        projs = self._list_projects()
        self._project_staff_id_converter = ProjectStaffIdConverter(projs)
        self._children = [SpecificProjectNode(project, self._get_project_staff_id(project)) for project in projs]
        super().__init__()

    @staticmethod
    def _list_projects():
        fields = ("id", "name", "bot_project_id")
        return list(gevent_idle_iter(projects.Project.objects.only(*fields)))

    def _get_project_staff_id(self, project):
        try:
            staff_id = self._project_staff_id_converter.get_staff_id(project)
        except MissingItem:
            staff_id = None
        return staff_id


class ScenarioValueNode(BaseInnerNode):
    slug = "scenario"

    def __init__(self):
        self._children = [ScenarioRoleNode()]
        self._idm_props = {"name": self.slug}
        super().__init__()


class ScenarioRoleNode(BaseInnerNode):
    slug = "scenario"
    _idm_props = {"slug": slug, "name": "Scenario"}

    def __init__(self):
        self._children = [SpecificScenarioNode(scenario_type) for scenario_type in ScriptRegistry.get_keys()]
        super().__init__()


class SpecificScenarioNode(ILeafNode):
    def __init__(self, scenario_type):
        self.slug = scenario_type
        self._children = []
        self._scenario_type = scenario_type
        self._idm_props = {"name": "Scenario '{}'".format(self._scenario_type)}
        super().__init__()

    def get_idm_properties(self):
        return {
            "name": "Scenario '{}' user".format(self._scenario_type),
            "help": "Can create and control scenarios with type {}".format(self._scenario_type),
            "set": "scenario_user",
        }

    @property
    def _role_path(self):
        return [RootNode.slug, ScenarioValueNode.slug, ScenarioRoleNode.slug, self.slug]

    def list_role_members(self):
        return get_role_members(self._role_path)

    def add_role_member(self, member):
        return add_role_member(self._role_path, member)

    def remove_role_member(self, member):
        return remove_role_member(self._role_path, member)


class SpecificProjectNode(BaseInnerNode):
    def __init__(self, project, project_staff_id):
        self.slug = project.id
        self._children = [RoleNode(project, project_staff_id)]
        self._idm_props = {"name": "Project '{}'".format(project.name)}
        super().__init__()


class RoleNode(BaseInnerNode):
    slug = "role"

    def __init__(self, project, project_staff_id):
        self._children = [
            ProjectOwnerNode(project, project_staff_id),
            ProjectUserNode(project),
            ProjectSuperuserNode(project),
            SshRebooterNode(project),
            NocAccessNode(project),
        ]

        self._idm_props = {"slug": self.slug, "name": "Role in project"}
        super().__init__()


class ProjectRoleBase(ILeafNode, metaclass=abc.ABCMeta):
    @abc.abstractproperty
    def manager_class(self):
        raise NotImplementedError()

    def __init__(self, project):
        self._role_manager = self.manager_class(project)

    def list_role_members(self):
        return self._role_manager.list_members()

    def add_role_member(self, member):
        with self._role_manager.audit_log_writer.on_add_member(member):
            self._role_manager.add_member(member)

    def remove_role_member(self, member):
        with self._role_manager.audit_log_writer.on_remove_member(member):
            self._role_manager.remove_member(member)


class ProjectOwnerNode(ProjectRoleBase):
    slug = "owner"
    manager_class = OwnerManager

    def __init__(self, project, project_staff_id):
        super().__init__(project)
        self._project = project
        self._project_staff_id = project_staff_id

    def get_idm_properties(self):
        props = {
            "name": "Project owner",
            "help": "Owner of a project, can ssh to project's hosts and has sudo on them",
            "set": "project_owner",
        }

        if self._project_staff_id is not None:
            props = self._fill_staff_id_alias(props)

        return props

    def _fill_staff_id_alias(self, idm_properties):
        aliases = [{"type": "project-staff-id", "name": str(self._project_staff_id)}]
        idm_properties["aliases"] = aliases
        return idm_properties


class NocAccessNode(ProjectRoleBase):
    slug = "noc_access"
    manager_class = NocAccessManager

    def get_idm_properties(self):
        props = {
            "name": "NOC access",
            "help": "Grants limited access to project hosts to NOC team",
            "set": "noc_access",
        }
        return props


class ProjectUserNode(ProjectRoleBase):
    slug = "user"
    manager_class = UserManager

    def get_idm_properties(self):
        props = {"name": "Project user", "help": "Can execute operations on host", "set": "project_user"}
        return props


class ProjectSuperuserNode(ProjectRoleBase):
    slug = "superuser"
    manager_class = SuperuserManager

    def get_idm_properties(self):
        props = {
            "name": "Project superuser",
            "help": "Can execute operations on host with NOPASSWD sudo",
            "set": "project_superuser",
        }
        return props


class SshRebooterNode(ProjectRoleBase):
    """Only one user can have this role -- it is robot-walle (workflow will not allow other users)
    Adding it as member means "enable rebooting via ssh", removing means the opposite
    """

    slug = "ssh_rebooter"
    manager_class = SshRebooterManager

    def get_idm_properties(self):
        props = {"name": "SSH rebooter", "help": "User that is able to reboot hosts via SSH", "set": "ssh_rebooter"}
        return props
