from collections import namedtuple

from walle.idm.role_tree import RootNode, ChildNotFound

ValueNodeWalkInfo = namedtuple("ValueNodeWalkInfo", "role_node, value_path, value_node")


def get_root_node():
    return RootNode()


def iter_subtree_nodes(path=None):
    def iter_nodes_aux(path, node):
        yield path, node
        for child_node in node.list_children():
            child_path = path + [child_node.slug]
            yield from iter_nodes_aux(child_path, child_node)

    path = path or [get_root_node().slug]
    node = get_node(path)
    for path_, node in iter_nodes_aux(path, node):
        yield path_, node


def iter_value_nodes(role_node=None, path=()):
    """Iterates all pairs (role node, its value node)"""
    role_node = role_node or get_root_node()
    role_node_name = role_node.get_idm_properties()["slug"]
    for value_node in role_node.list_children():
        value_name = value_node.slug
        cur_pair_path = path + (role_node_name, value_name)
        yield ValueNodeWalkInfo(role_node, cur_pair_path, value_node)

        # just an easy way to iter 0 or 1 child of value
        for child_node in value_node.list_children():
            yield from iter_value_nodes(child_node, cur_pair_path)


def iter_leaves(role_node=None):
    role_node = role_node or get_root_node()
    for walk_info in iter_value_nodes(role_node):
        if not walk_info.value_node.list_children():
            yield walk_info.value_path, walk_info.value_node


def get_node(path, root_node=None):
    def _get_node_aux(cur_path, cur_node):
        if not cur_path:
            return cur_node
        else:
            next_name = cur_path[0]
            next_child = cur_node.get_child(next_name)
            return _get_node_aux(cur_path[1:], next_child)

    root_node = root_node or get_root_node()
    root_slug, path = path[0], path[1:]
    # there is only one variant of first path component -- RootNode slug
    if root_slug != root_node.get_idm_properties()["slug"]:
        raise ChildNotFound(root_slug)

    return _get_node_aux(path, root_node)
