import logging
from collections import defaultdict

from walle.clients import staff
from walle.idm.node_views import ValueNodeView, RoleNodeView
from walle.idm.role_tree import RootNode
from walle.idm.traversal import iter_leaves
from walle.util.misc import iter_chunks

log = logging.getLogger(__name__)


class RoleTreeView:
    def to_dict(self):
        tree_view = self.build_role_node_view(RootNode())
        return {"roles": tree_view.to_dict()}

    def build_role_node_view(self, node):
        node_view = RoleNodeView(**node.get_idm_properties())
        for child_node in node.list_children():
            child_view = self.build_value_node_view(child_node)
            node_view.add_value(child_node.slug, child_view)
        return node_view

    def build_value_node_view(self, node):
        node_view = ValueNodeView(**node.get_idm_properties())
        # value node can have no children (leaf node) or one child (which is role node)
        children = node.list_children()
        if children:
            child_node = children[0]
            child_view = self.build_role_node_view(child_node)
            node_view.set_role(child_view)
        return node_view


class RoleMembersView:
    def to_dict(self):
        users, groups = self._get_all_role_members()
        return {"users": self._render_users(users), "groups": self._render_groups(groups)}

    @staticmethod
    def _get_all_role_members():
        users = defaultdict(set)
        groups = defaultdict(set)

        for leaf_path, leaf_node in iter_leaves():
            for member in leaf_node.list_role_members():
                dst = groups if staff.is_group(member) else users
                dst[member].add(leaf_path)

        return users, groups

    def _render_users(self, users):
        rendered = []
        for user, role_paths in users.items():
            rendered.append({"login": user, "roles": [self._render_tree_path(role_path) for role_path in role_paths]})
        return rendered

    def _render_groups(self, groups):
        group_name_to_id = staff.groups_to_ids(groups.keys())
        rendered = []
        for group, role_paths in groups.items():
            group_id = group_name_to_id.get(group)
            if group_id is not None:
                rendered.append(
                    {
                        "group": group_name_to_id[group],
                        "roles": [self._render_tree_path(role_path) for role_path in role_paths],
                    }
                )
            else:
                log.error("Could not get staff id of group %s", group)
        return rendered

    @staticmethod
    def _render_tree_path(path):
        """Make dict from pairs of path parts"""
        return dict(iter_chunks(path, 2))
