"""Provides distributed locks used by Wall-E."""
import logging
import typing as tp
from datetime import timedelta

from walle.util.mongo import lock as mongo_lock

_TIMEOUT = 30
_LONG_TIMEOUT = 120

log = logging.getLogger(__name__)


def lost_mongo_lock_retry(func):
    """Wraps a callable object to retry calls that raise LockIsLostError."""

    def wrapper(*args, **kwargs):
        error = None

        while True:
            try:
                result = func(*args, **kwargs)
            except mongo_lock.LockIsLostError as e:
                error = e
                log.error("Retry %s call due to error: %s", func, error)
            else:
                if error is None:
                    return result
                else:
                    raise error

    return wrapper


class ProjectInterruptableLock(mongo_lock.InterruptableLock):
    """
    Locks project for the following actions:

    1:
        * Add host to the project.
        * Delete the project.

        Guarantees that there will be no hosts that belong to unexisting projects.

    2:
        VLAN scheme changing.

        Guarantees that no hosts will be moved between incompatible projects.

    It's allowed to acquire the lock inside of HostInterruptableLock but not vice versa.
    """

    def __init__(self, project_id):
        super().__init__(project_id, prefix="projects", timeout=_TIMEOUT)


class HostInterruptableLock(mongo_lock.InterruptableLock):
    """Locks host for modification that is not atomic."""

    def __init__(self, uuid, tier=None):
        super().__init__(str(uuid), prefix="hosts", timeout=_TIMEOUT)


class ScenarioInterruptableLock(mongo_lock.InterruptableLock):
    """Locks scenario for modification that is not atomic."""

    def __init__(self, scenario_id):
        super().__init__(str(scenario_id), prefix="scenarios", timeout=_TIMEOUT)


class HostHealingInterruptableGlobalLock(mongo_lock.InterruptableLock):
    """Acquires a right to heal hosts."""

    def __init__(self):
        super().__init__("host-healing", timeout=_TIMEOUT)


class SettingsUpdateInterruptableGlobalLock(mongo_lock.InterruptableLock):
    """Acquires a right to update settings."""

    def __init__(self):
        super().__init__("settings-update", timeout=_TIMEOUT)


class ScenarioAcquiringInterruptableGlobalLock(mongo_lock.InterruptableLock):
    """Acquires a right to acquire scenario."""

    def __init__(self):
        super().__init__("", prefix="scenario-permission", timeout=_LONG_TIMEOUT)


class AutomationPlotInterruptableLock(mongo_lock.InterruptableLock):
    """Locks automation plot for operation that are potentially racy, like
    deleting automation plot and assigning one to a project."""

    def __init__(self, inv):
        super().__init__(
            str(inv),
            prefix="automation-plots",
            timeout=_TIMEOUT,
        )


class MaintenancePlotInterruptableLock(mongo_lock.InterruptableLock):
    """Locks maintenance plot for operation that are potentially racy, like
    deleting maintenance plot and assigning one to a project."""

    def __init__(self, plot_id):
        super().__init__(
            str(plot_id),
            prefix="maintenance-plots",
            timeout=_TIMEOUT,
        )


class CronJobInterruptableLock(mongo_lock.InterruptableLock):
    """Locks cron job for execution"""

    def __init__(self, job_id: str, locking_time: tp.Union[int, timedelta]) -> None:
        super().__init__(
            job_id,
            locking_time=locking_time,
            prefix="cron",
            blocking=False,
            timeout=_TIMEOUT,
        )


class VlanTogglerSwitchInterruptableLock(mongo_lock.InterruptableLock):
    """Locks cron job for execution"""

    def __init__(self, switch_id: str) -> None:
        super().__init__(
            switch_id,
            prefix="vlan_toggler_switch",
            blocking=False,
            timeout=_TIMEOUT,
        )


class PartitionerPartyInterruptableLock(mongo_lock.InterruptableLock):
    def __init__(self, partitioner_type, worker_id: str):
        self._prefix = f"partitioner-{partitioner_type}-party"
        super().__init__(worker_id, locking_time=timedelta(seconds=30), prefix=self._prefix, timeout=_LONG_TIMEOUT)

    def get_whole_party(self) -> list[str]:
        return list(
            sorted(mongo_lock.MongoLock.objects(locked_object_id__startswith=self._prefix).distinct("instance"))
        )


class PartitionerShardInterruptableLock(mongo_lock.InterruptableLock):
    def __init__(self, partitioner_type, shard_id):
        self.shard_id = str(shard_id)
        super().__init__(
            self.shard_id,
            locking_time=timedelta(minutes=1),
            prefix=f"partitioner-{partitioner_type}-shard",
            blocking=False,
            timeout=_TIMEOUT,
        )
