"""Wall-E's main module."""

from os.path import abspath, join, dirname

import gevent.monkey

gevent.monkey.patch_all(subprocess=True)

import grpc.experimental.gevent as grpc_gevent

grpc_gevent.init_gevent()

import logging
import gevent.hub

log = logging.getLogger(__name__)


# Customize gevent hub to log all exceptions instead of printing them to stderr
class Hub(gevent.hub.Hub):
    def print_exception(self, context, type, value, tb):
        exc_info = (type, value, tb)

        if context is None:
            log.error("Gevent internal error:", exc_info=exc_info)
        elif "<bound method GeventedHTTPTransport.send of <raven.transport.gevent.GeventedHTTPTransport object" in repr(
            context
        ):
            # Raven constantly gets timeout errors on sending logs to Sentry. These errors are logged by this hub
            # which makes Raven to send it's own errors to Sentry, which may fail and the error will be catched by this
            # hub and sent to Sentry and so on, and so on.
            #
            # Don't know how to make it better.
            logging.getLogger("raven").error("%s has crashed:", context, exc_info=exc_info)
        else:
            log.error("%s has crashed:", context, exc_info=exc_info)


# gevent.hub._threadlocal.Hub = Hub

import argparse
import time

import yaml

from sepelib.core import config
from sepelib.core.exceptions import Error

import walle.statbox.loggers

from walle import constants
from walle.application import app
from walle.util.cloud_tools import get_instance_role, get_config_path

from walle.util.gevent_tools import monkeypatch_json

monkeypatch_json()

# Override the default string handling function to always return unicode objects
yaml.Loader.add_constructor("tag:yaml.org,2002:str", lambda self, node: self.construct_scalar(node))
yaml.SafeLoader.add_constructor("tag:yaml.org,2002:str", lambda self, node: self.construct_scalar(node))


def main():
    args = _parse_args()
    load_config(args)
    app.setup_logging()

    try:
        if args.port is not None:
            config.set_value("web.http.port", args.port)

        if config.get_value("automation.enabled") and not config.get_value("expert_system.enabled"):
            raise Error("Got conflicting configuration parameters: automation is enabled with disabled Expert System.")

        if config.get_value("roles.enabled"):
            role = get_instance_role() if args.role is None else args.role
        else:
            if args.role is not None:
                raise Error("Instance role is specified but role mode is disabled.")

            role = None

        app.init_role(role)
        walle.statbox.loggers.init()

        log.info("Starting the daemon...")
        app.start()
    except Exception:
        log.exception("The daemon has crashed:")

        sentry_flush_time = 5
        log.info("Sleeping %s seconds to give a time to Sentry to flush the error messages.", sentry_flush_time)
        time.sleep(sentry_flush_time)

    log.info("Exiting...")


def start_shell():
    from IPython import start_ipython

    args = _parse_args()
    args.config_context = args.config_context or [("log_dir", "logs")]
    load_config(args, strict=False)
    app.setup_logging()

    with app.as_context(connect_database=False):
        # having common ipython dir in repo allows us to share code and settings
        ipython_dir = abspath(join(dirname(__file__), "..", ".ipython"))
        start_ipython(
            [
                "--ipython-dir={}".format(ipython_dir),
                "--profile=walle_shell",
                "--no-banner",
            ]
        )


def _parse_args():
    parser = argparse.ArgumentParser(description="Wall-E server")
    parser.add_argument("-c", "--config", default=get_config_path(), help="configuration file path")
    parser.add_argument("-p", "--port", type=int, help="port to listen to on")
    parser.add_argument("-r", "--role", help="instance role")
    parser.add_argument("-d", "--debug", default=False, action="store_true", help="debug mode")
    parser.add_argument("--chdir", default="/", help="change working directory to")
    config.augment_args_parser(parser)
    return parser.parse_args()


def load_config(args, strict=True):
    config_context = config.get_context_from_env(prefix="WALLE")
    config_context.update(args.config_context)
    config.load(path=args.config, defaults=constants.DEFAULT_CONFIG_PATH, config_context=config_context, strict=strict)

    if args.debug:
        config.set_value("run.debug", args.debug)
