import dataclasses
import typing

from walle.clients import abc
from walle.maintenance_plot.jsonschema import (
    ABC_ROLE_SCOPE_SLUGS_SCHEMA,
    ABC_ROLES_CODES_SCHEMA,
    STAFF_LOGINS_SCHEMA,
    DONT_CREATE_APPROVEMENT_IF_TOTAL_COUNT_OF_TAKEN_HOSTS_FROM_PLOT_LESS_THAN,
    DONT_ALLOW_START_SCENARIO_IF_TOTAL_NUMBER_OF_ACTIVE_HOSTS_MORE_THAN,
    ABC_DUTY_SCHEDULE_SLUGS_SCHEMA,
)


@dataclasses.dataclass(eq=True, frozen=True)
class CommonScenarioSettings:
    total_number_of_active_hosts: typing.Optional[int] = None
    dont_allow_start_scenario_if_total_number_of_active_hosts_more_than: typing.Optional[int] = None

    jsonschema: typing.ClassVar = {
        "type": "object",
        "properties": {
            "total_number_of_active_hosts": DONT_CREATE_APPROVEMENT_IF_TOTAL_COUNT_OF_TAKEN_HOSTS_FROM_PLOT_LESS_THAN,
            "dont_allow_start_scenario_if_total_number_of_active_hosts_more_than": DONT_ALLOW_START_SCENARIO_IF_TOTAL_NUMBER_OF_ACTIVE_HOSTS_MORE_THAN,
        },
        "additionalProperties": False,
        "description": "Common scenarios settings for all types of scenarios",
        "required": [],
    }

    def __str__(self):
        return (
            f"Common scenarios settings: "
            f"total number of active hosts: {self.total_number_of_active_hosts}, "
            f"don't allow start scenarios if total number of active hosts more than: "
            f"{self.dont_allow_start_scenario_if_total_number_of_active_hosts_more_than}"
        )

    @classmethod
    def from_dict(cls, values_dict):
        return CommonScenarioSettings(
            total_number_of_active_hosts=values_dict.get("total_number_of_active_hosts", None),
            dont_allow_start_scenario_if_total_number_of_active_hosts_more_than=values_dict.get(
                "dont_allow_start_scenario_if_total_number_of_active_hosts_more_than", None
            ),  # noqa
        )

    def to_dict(self):
        return {
            "total_number_of_active_hosts": self.total_number_of_active_hosts,
            "dont_allow_start_scenario_if_total_number_of_active_hosts_more_than": self.dont_allow_start_scenario_if_total_number_of_active_hosts_more_than,
        }


@dataclasses.dataclass(eq=True, frozen=True)
class MaintenanceApprovers:
    logins: typing.List[str] = dataclasses.field(default_factory=list)
    abc_roles_codes: typing.List[str] = dataclasses.field(default_factory=list)
    abc_role_scope_slugs: typing.List[str] = dataclasses.field(default_factory=list)
    abc_duty_schedule_slugs: typing.List[str] = dataclasses.field(default_factory=list)

    jsonschema: typing.ClassVar = {
        "type": "object",
        "properties": {
            "logins": STAFF_LOGINS_SCHEMA,
            "abc_roles_codes": ABC_ROLES_CODES_SCHEMA,
            "abc_role_scope_slugs": ABC_ROLE_SCOPE_SLUGS_SCHEMA,
            "abc_duty_schedule_slugs": ABC_DUTY_SCHEDULE_SLUGS_SCHEMA,
        },
        "additionalProperties": False,
        "description": "Maintenance approvers",
        "required": ["logins", "abc_roles_codes", "abc_role_scope_slugs"],
    }

    def __str__(self):
        return (
            f"Maintenance approvers: "
            f"logins: {self.logins}, "
            f"role codes: {self.abc_roles_codes}, "
            f"role scope slugs: {self.abc_role_scope_slugs}, "
            f"duty schedule slugs: {self.abc_duty_schedule_slugs}"
        )

    @classmethod
    def from_dict(cls, values_dict):
        return MaintenanceApprovers(
            logins=values_dict.get("logins", []),
            abc_roles_codes=values_dict.get("abc_roles_codes", []),
            abc_role_scope_slugs=values_dict.get("abc_role_scope_slugs", []),
            abc_duty_schedule_slugs=values_dict.get("abc_duty_schedule_slugs", []),
        )

    def get_approvers(self, abc_service_slug: str) -> typing.List[str]:
        abc_logins_with_role_codes = []
        if self.abc_roles_codes:
            abc_logins_with_role_codes = self._get_abc_logins_with_role_codes(abc_service_slug, self.abc_roles_codes)
        abc_logins_with_role_scope_slugs = []
        if self.abc_role_scope_slugs:
            abc_logins_with_role_scope_slugs = self._get_abc_logins_with_role_scope_slugs(
                abc_service_slug, self.abc_role_scope_slugs
            )
        abc_logins_with_duty_schedule_slugs = []
        if self.abc_duty_schedule_slugs:
            abc_logins_with_duty_schedule_slugs = self._get_abc_logins_with_duty_schedule_slugs(
                abc_service_slug, self.abc_duty_schedule_slugs
            )

        return list(
            set(abc_logins_with_role_codes)
            | set(abc_logins_with_role_scope_slugs)
            | set(abc_logins_with_duty_schedule_slugs)
            | set(self.logins)
        )

    def to_dict(self):
        return {
            "logins": self.logins or [],
            "abc_roles_codes": self.abc_roles_codes or [],
            "abc_role_scope_slugs": self.abc_role_scope_slugs or [],
            "abc_duty_schedule_slugs": self.abc_duty_schedule_slugs or [],
        }

    @staticmethod
    def _get_abc_logins_with_role_codes(abc_service_slug: str, role_codes: [str]) -> [str]:
        return [
            member["person"]["login"]
            for member in abc.get_service_members(service_slug=abc_service_slug, role_codes=role_codes)
            if not member["person"]["is_robot"]
        ]

    @staticmethod
    def _get_abc_logins_with_role_scope_slugs(abc_service_slug: str, role_scope_slugs: [str]) -> [str]:
        return [
            member["person"]["login"]
            for member in abc.get_service_members(service_slug=abc_service_slug, role_scope_slugs=role_scope_slugs)
            if not member["person"]["is_robot"]
        ]

    @staticmethod
    def _get_abc_logins_with_duty_schedule_slugs(abc_service_slug: str, duty_schedule_slugs: [str]) -> [str]:
        return abc.get_service_on_duty_logins(abc_service_slug, duty_schedule_slugs)
