"""Maintenance plot - hosts' settings for scenarios.
https://wiki.yandex-team.ru/wall-e/devel/drafts/maintenance-plot-proektov/"""

import dataclasses
import logging
import typing

from mongoengine import DictField, ListField, StringField, BooleanField

import sepelib
from sepelib.mongo.util import register_model
from walle.clients import abc
from walle.maintenance_plot import constants
from walle.maintenance_plot.common_settings import MaintenanceApprovers, CommonScenarioSettings
from walle.maintenance_plot.exceptions import (
    MaintenancePlotError,
    MaintenancePlotScenarioSettingsNotFoundError,
    MaintenancePlotScenarioSettingsAlreadyExistError,
    MaintenancePlotScenarioSettingsTypeDoesntExists,
)
from walle.maintenance_plot.scenarios_settings.base import BaseScenarioMaintenancePlotSettings
from walle.models import Document
from walle.scenario.constants import ScriptName

log = logging.getLogger(__name__)


MAINTENANCE_PLOT_ID_REGEX = r"(^[a-z][a-z0-9]*(?:-[a-z0-9]+)*$)"
MAINTENANCE_PLOT_ID_TYPE = str


@dataclasses.dataclass
class MaintenancePlotMetaInfo:
    abc_service_slug: str
    name: typing.Optional[str]

    @classmethod
    def get_json_schema(cls):
        return {
            "type": "object",
            "properties": {
                "abc_service_slug": {
                    "type": "string",
                    "minLength": 1,
                    "description": "ABC service slug",
                },
                "name": {
                    "type": "string",
                    "description": "Name of a plot",
                },
            },
            "additionalProperties": False,
            "description": "Maintenance plot meta info",
            "required": ["abc_service_slug", "name"],
        }

    @classmethod
    def from_dict(cls, values_dict: dict):
        return MaintenancePlotMetaInfo(
            abc_service_slug=values_dict.get("abc_service_slug"), name=values_dict.get("name")
        )

    def to_dict(self):
        return {"abc_service_slug": self.abc_service_slug, "name": self.name}


@dataclasses.dataclass(eq=True)
class MaintenancePlotCommonSettings:
    maintenance_approvers: MaintenanceApprovers
    common_scenarios_settings: CommonScenarioSettings

    @classmethod
    def get_json_schema(cls):
        return {
            "type": "object",
            "properties": {
                "maintenance_approvers": MaintenanceApprovers.jsonschema,
                "common_scenario_settings": CommonScenarioSettings.jsonschema,
            },
            "additionalProperties": False,
            "description": "Common settings",
            "required": ["maintenance_approvers"],
        }

    @classmethod
    def from_dict(cls, values_dict):
        return MaintenancePlotCommonSettings(
            maintenance_approvers=MaintenanceApprovers.from_dict(values_dict.get("maintenance_approvers", {})),
            common_scenarios_settings=CommonScenarioSettings.from_dict(values_dict.get("common_scenario_settings", {})),
        )

    def to_dict(self):
        return {
            "maintenance_approvers": self.maintenance_approvers.to_dict(),
            "common_scenario_settings": self.common_scenarios_settings.to_dict(),
        }


@dataclasses.dataclass(eq=True)
class MaintenancePlotLimitView:
    active_hosts_number: int
    affected_scenarios: typing.List[int]

    @classmethod
    def get_json_schema(cls):
        return {
            "type": "object",
            "properties": {
                "active_hosts_number": {"type": "integer"},
                "affected_scenarios": {"type": "array", "items": {"type": "integer"}},
            },
        }

    @classmethod
    def from_dict(cls, values_dict):
        return MaintenancePlotLimitView(
            active_hosts_number=values_dict.get("active_hosts_number", 0),
            affected_scenarios=values_dict.get("affected_scenarios", []),
        )

    def to_dict(self):
        return {
            "active_hosts_number": self.active_hosts_number,
            "affected_scenarios": self.affected_scenarios,
        }


@dataclasses.dataclass
class MaintenancePlotScenarioSettings:
    scenario_type: typing.Literal[tuple(ScriptName.ALL_EXISTING)]
    settings: typing.Type[BaseScenarioMaintenancePlotSettings]

    @classmethod
    def get_json_schema(cls):
        return {
            "anyOf": [
                {
                    "type": "object",
                    "properties": {
                        "scenario_type": {"const": scenario_type},
                        "settings": settings.jsonschema,
                    },
                    "additionalProperties": False,
                    "required": ["scenario_type", "settings"],
                    "description": "Scenario settings",
                }
                for scenario_type, settings in constants.SCENARIO_TYPES_SETTINGS_MAP.items()
            ],
        }

    @classmethod
    def from_dict(cls, values_dict: dict):
        scenario_type = values_dict.get("scenario_type")
        settings_dict = values_dict.get("settings", {})

        settings = constants.SCENARIO_TYPES_SETTINGS_MAP.get(scenario_type)
        if not settings:
            raise MaintenancePlotScenarioSettingsTypeDoesntExists

        return MaintenancePlotScenarioSettings(scenario_type=scenario_type, settings=settings.from_dict(settings_dict))

    def to_dict(self):
        return {"scenario_type": self.scenario_type, "settings": self.settings.to_dict()}


#  TODO: need to create a separate class for plot_document.scenarios_settings
#   and encapsulate all data transformation there (and scheme method!)
def get_scenarios_settings_json_schema():
    return {
        "type": "object",
        "properties": {
            "scenarios_settings": {
                "type": "array",
                "items": MaintenancePlotScenarioSettings.get_json_schema(),
            },
        },
        "additionalProperties": False,
        "required": ["scenarios_settings"],
    }


@dataclasses.dataclass
class MaintenancePlot:
    id: str
    meta_info: MaintenancePlotMetaInfo
    common_settings: MaintenancePlotCommonSettings
    limit_view: MaintenancePlotLimitView = dataclasses.field(
        default_factory=lambda: MaintenancePlotLimitView(active_hosts_number=0, affected_scenarios=[])
    )
    scenarios_settings: typing.List[MaintenancePlotScenarioSettings] = dataclasses.field(
        default_factory=lambda: [
            MaintenancePlotScenarioSettings(scenario_type=scenario_type, settings=scenario_maintenance_plot_settings())
            for scenario_type, scenario_maintenance_plot_settings in constants.SCENARIO_TYPES_SETTINGS_MAP.items()
        ]
    )

    @classmethod
    def get_json_schema(cls):
        return {
            "type": "object",
            "properties": {
                "id": {"type": "string", "minLength": 2, "maxLength": 32, "description": "ID"},
                "meta_info": MaintenancePlotMetaInfo.get_json_schema(),
                "common_settings": MaintenancePlotCommonSettings.get_json_schema(),
                "limit_view": MaintenancePlotLimitView.get_json_schema(),
                "scenarios_settings": {
                    "type": "array",
                    "items": MaintenancePlotScenarioSettings.get_json_schema(),
                },
            },
            "additionalProperties": False,
            "required": ["id", "meta_info"],
        }

    @staticmethod
    def _load_scenario_settings_from_dicts(scenario_settings_dicts_list: typing.List[dict]):
        # Use values from dictionary or create an object with default values.
        result = []
        for scenario_type, scenario_maintenance_plot_settings in constants.SCENARIO_TYPES_SETTINGS_MAP.items():
            scenario_settings = None

            for scenario_settings_dict in scenario_settings_dicts_list:
                if scenario_settings_dict.get("scenario_type") == scenario_type:
                    scenario_settings = MaintenancePlotScenarioSettings.from_dict(scenario_settings_dict)
                    break

            result.append(
                scenario_settings
                if scenario_settings is not None
                else MaintenancePlotScenarioSettings(
                    scenario_type=scenario_type, settings=scenario_maintenance_plot_settings()
                )
            )
        return sorted(result, key=lambda x: x.scenario_type)

    @classmethod
    def from_dict(cls, values_dict: dict):
        return MaintenancePlot(
            id=values_dict["id"],
            meta_info=MaintenancePlotMetaInfo.from_dict(values_dict.get("meta_info", {})),
            common_settings=MaintenancePlotCommonSettings.from_dict(values_dict.get("common_settings", {})),
            limit_view=MaintenancePlotLimitView.from_dict(values_dict.get("limit_view", {})),
            scenarios_settings=cls._load_scenario_settings_from_dicts(values_dict.get("scenarios_settings", [])),
        )

    def get_approvers(self) -> typing.List[str]:
        logins = (
            self.common_settings.maintenance_approvers.get_approvers(self.meta_info.abc_service_slug)
            or self._get_default_approvers_for_service_slug(self.meta_info.abc_service_slug)
            or self._get_approvers_from_parent_service()
        )
        if not logins:
            raise MaintenancePlotError(
                "Can not find any approvers for ABC service '%s'" % self.meta_info.abc_service_slug
            )
        return sorted(logins)

    def _get_default_approvers_for_service_slug(self, service_slug) -> list[str]:
        for default_maintenance_approvers in constants.DEFAULT_MAINTENANCE_APPROVERS:
            logins = default_maintenance_approvers.get_approvers(service_slug)
            if logins:
                log.info("Using %s as default maintenance approvers" % default_maintenance_approvers)
                return logins
        return []

    def _get_approvers_from_parent_service(self) -> list[str]:
        parent_slug = abc.get_service_parent_slug(self.meta_info.abc_service_slug)
        return self._get_default_approvers_for_service_slug(parent_slug)

    def get_scenario_settings(
        self, scenario_type: typing.Literal[tuple(ScriptName.ALL_EXISTING)]
    ) -> typing.Type[BaseScenarioMaintenancePlotSettings]:
        for item in self.scenarios_settings or []:
            if item.scenario_type == scenario_type:
                return item.settings
        # Return defaults for a given scenario type.
        return MaintenancePlotScenarioSettings.from_dict(dict(scenario_type=scenario_type)).settings

    def to_dict(self):
        return {
            "id": self.id,
            "meta_info": self.meta_info.to_dict(),
            "common_settings": self.common_settings.to_dict(),
            "limit_view": self.limit_view.to_dict(),
            "scenarios_settings": [item.to_dict() for item in self.scenarios_settings],
        }


@register_model
class MaintenancePlotModel(Document):
    id = StringField(
        primary_key=True, required=True, regex=MAINTENANCE_PLOT_ID_REGEX, min_length=2, max_length=32, help_text="ID"
    )
    meta_info = DictField(help_text="Dictionary with meta information about maintenance plot")
    common_settings = DictField(help_text="Dictionary with settings used in all scenario types")
    limit_view = DictField(default=dict, help_text="Dictionary with meta information about active hosts and scenarios")
    scenarios_settings = ListField(default=list, help_text="Settings for different scenarios types")

    gc_enabled = BooleanField(default=False, help_text="If plot don't have any links with projects, it will be gc")

    meta = {
        "collection": "maintenance_plot",
    }

    default_api_fields = (
        "id",
        "meta_info",
        "common_settings",
        "limit_view",
        "scenarios_settings",
        "gc_enabled",
    )
    api_fields = default_api_fields

    def __str__(self) -> str:
        return "<Maintenance plot \"{}\">".format(self.id)

    @classmethod
    def with_fields(cls, fields: typing.Optional[list[str]]) -> sepelib.mongo.document.QuerySet:
        return cls.objects.only(*cls.api_query_fields(fields or None))

    def as_dataclass(self):
        return MaintenancePlot(
            id=self.id,
            meta_info=MaintenancePlotMetaInfo.from_dict(self.meta_info),
            common_settings=MaintenancePlotCommonSettings.from_dict(self.common_settings),
            limit_view=MaintenancePlotLimitView.from_dict(self.limit_view or {}),
            scenarios_settings=[
                MaintenancePlotScenarioSettings.from_dict(item) for item in self.scenarios_settings or []
            ],
        )

    @classmethod
    def from_dataclass(cls, plot_obj: MaintenancePlot):
        return cls(**plot_obj.to_dict())

    def set_meta_info(self, meta_info: MaintenancePlotMetaInfo):
        return self.modify(set__meta_info=meta_info.to_dict())

    def set_common_settings(self, common_settings: MaintenancePlotCommonSettings):
        return self.modify(set__common_settings=common_settings.to_dict())

    def set_limit_view(self, limit_view: MaintenancePlotLimitView):
        return self.modify(set__limit_view=limit_view.to_dict())

    def set_scenario_settings(self, scenario_settings: list[MaintenancePlotScenarioSettings]):
        return self.modify(set__scenarios_settings=[settings.to_dict() for settings in scenario_settings])

    def get_scenario_settings(
        self, scenario_type: typing.Literal[tuple(ScriptName.ALL_EXISTING)]
    ) -> MaintenancePlotScenarioSettings:
        for item in self.scenarios_settings or []:
            scenario_settings = MaintenancePlotScenarioSettings.from_dict(item)
            if scenario_settings.scenario_type == scenario_type:
                return scenario_settings
        raise MaintenancePlotScenarioSettingsNotFoundError()

    def modify_scenario_settings(self, new_scenario_settings: MaintenancePlotScenarioSettings):
        if not self._have_scenario_settings(new_scenario_settings.scenario_type):
            raise MaintenancePlotScenarioSettingsNotFoundError()

        new_scenarios_settings = []
        for item in self.scenarios_settings or []:
            existing_scenario_settings = MaintenancePlotScenarioSettings.from_dict(item)
            if existing_scenario_settings.scenario_type == new_scenario_settings.scenario_type:
                new_scenarios_settings.append(new_scenario_settings.to_dict())
            else:
                new_scenarios_settings.append(existing_scenario_settings.to_dict())

        self.scenarios_settings = new_scenarios_settings
        self.save()

    def add_scenario_settings(self, scenario_settings: MaintenancePlotScenarioSettings):
        if self._have_scenario_settings(scenario_settings.scenario_type):
            raise MaintenancePlotScenarioSettingsAlreadyExistError()

        self.scenarios_settings.append(scenario_settings.to_dict())
        self.save()

    def delete_scenario_settings(self, scenario_type_to_delete: typing.Literal[tuple(ScriptName.ALL_EXISTING)]):
        if not self._have_scenario_settings(scenario_type_to_delete):
            raise MaintenancePlotScenarioSettingsNotFoundError()

        new_scenarios_settings = []
        for item in self.scenarios_settings or []:
            existing_scenario_settings = MaintenancePlotScenarioSettings.from_dict(item)
            if existing_scenario_settings.scenario_type != scenario_type_to_delete:
                new_scenarios_settings.append(existing_scenario_settings.to_dict())
            else:
                continue

        self.scenarios_settings = new_scenarios_settings
        self.save()

    def _have_scenario_settings(self, scenario_type: typing.Literal[tuple(ScriptName.ALL_EXISTING)]) -> bool:
        return any(
            MaintenancePlotScenarioSettings.from_dict(item).scenario_type == scenario_type
            for item in self.scenarios_settings
        )
