import dataclasses
import typing

from walle.maintenance_plot.jsonschema import (
    REQUEST_CMS_X_SECONDS_BEFORE_MAINTENANCE_START_TIME,
    ENABLE_REDEPLOY_AFTER_CHANGE_OF_MAC_ADDRESS,
    GET_APPROVERS_TO_TICKET_IF_HOSTS_NOT_IN_MAINTENANCE_BY_START_TIME,
    GET_APPROVERS_TO_TICKET_IF_HOSTS_NOT_IN_MAINTENANCE_BY_X_SECONDS,
    IGNORE_CMS_ON_HOST_OPERATIONS,
    USE_YP_SLA,
    APPROVAL_SLA,
    ENABLE_MANUAL_APPROVAL_AFTER_HOSTS_POWER_OFF,
)
from walle.maintenance_plot.scenarios_settings.base import BaseScenarioMaintenancePlotSettings, HostTaskOffsetTimes
from walle import restrictions


@dataclasses.dataclass(eq=True, frozen=True)
class NocHardMaintenancePlotSettings(BaseScenarioMaintenancePlotSettings):
    enable_redeploy_after_change_of_mac_address: bool = False
    get_approvers_to_ticket_if_hosts_not_in_maintenance_by_start_time: bool = False
    ignore_cms_on_host_operations: bool = False
    request_cms_x_seconds_before_maintenance_start_time: typing.Optional[int] = None
    get_approvers_to_ticket_if_hosts_not_in_maintenance_by_x_seconds: typing.Optional[int] = None
    approval_sla: typing.Optional[int] = None
    enable_manual_approval_after_hosts_power_off: bool = False

    use_yp_sla: bool = False

    jsonschema: typing.ClassVar = {
        "type": "object",
        "properties": {
            "enable_redeploy_after_change_of_mac_address": ENABLE_REDEPLOY_AFTER_CHANGE_OF_MAC_ADDRESS,
            "get_approvers_to_ticket_if_hosts_not_in_maintenance_by_start_time": GET_APPROVERS_TO_TICKET_IF_HOSTS_NOT_IN_MAINTENANCE_BY_START_TIME,
            "ignore_cms_on_host_operations": IGNORE_CMS_ON_HOST_OPERATIONS,
            "request_cms_x_seconds_before_maintenance_start_time": REQUEST_CMS_X_SECONDS_BEFORE_MAINTENANCE_START_TIME,
            "get_approvers_to_ticket_if_hosts_not_in_maintenance_by_x_seconds": GET_APPROVERS_TO_TICKET_IF_HOSTS_NOT_IN_MAINTENANCE_BY_X_SECONDS,
            "use_yp_sla": USE_YP_SLA,
            "approval_sla": APPROVAL_SLA,
            "enable_manual_approval_after_hosts_power_off": ENABLE_MANUAL_APPROVAL_AFTER_HOSTS_POWER_OFF,
        },
        "additionalProperties": False,
        "description": "'noc-maintenance' scenario settings",
        "required": [],
    }

    @classmethod
    def from_dict(cls, values_dict: dict):
        return NocHardMaintenancePlotSettings(
            enable_redeploy_after_change_of_mac_address=values_dict.get(
                "enable_redeploy_after_change_of_mac_address", False
            ),
            get_approvers_to_ticket_if_hosts_not_in_maintenance_by_start_time=values_dict.get(
                "get_approvers_to_ticket_if_hosts_not_in_maintenance_by_start_time", False
            ),
            ignore_cms_on_host_operations=values_dict.get("ignore_cms_on_host_operations", False),
            request_cms_x_seconds_before_maintenance_start_time=values_dict.get(
                "request_cms_x_seconds_before_maintenance_start_time" or None
            ),
            get_approvers_to_ticket_if_hosts_not_in_maintenance_by_x_seconds=values_dict.get(
                "get_approvers_to_ticket_if_hosts_not_in_maintenance_by_x_seconds" or None
            ),
            use_yp_sla=values_dict.get("use_yp_sla", False),
            approval_sla=values_dict.get("approval_sla" or None),
            enable_manual_approval_after_hosts_power_off=values_dict.get(
                "enable_manual_approval_after_hosts_power_off", False
            ),
        )

    def to_dict(self):
        return {
            "enable_redeploy_after_change_of_mac_address": self.enable_redeploy_after_change_of_mac_address,
            "get_approvers_to_ticket_if_hosts_not_in_maintenance_by_start_time": self.get_approvers_to_ticket_if_hosts_not_in_maintenance_by_start_time,
            "ignore_cms_on_host_operations": self.ignore_cms_on_host_operations,
            "request_cms_x_seconds_before_maintenance_start_time": self.request_cms_x_seconds_before_maintenance_start_time,
            "get_approvers_to_ticket_if_hosts_not_in_maintenance_by_x_seconds": self.get_approvers_to_ticket_if_hosts_not_in_maintenance_by_x_seconds,
            "use_yp_sla": self.use_yp_sla,
            "approval_sla": self.approval_sla,
            "enable_manual_approval_after_hosts_power_off": self.enable_manual_approval_after_hosts_power_off,
        }

    def get_offset_times_of_host_tasks(self) -> HostTaskOffsetTimes:
        return HostTaskOffsetTimes(
            request_cms_x_seconds_before_maintenance_start_time=self.request_cms_x_seconds_before_maintenance_start_time
        )

    def get_restrictions_disabled_by_settings(self) -> list[str]:
        result = []
        if self.enable_manual_approval_after_hosts_power_off:
            result.append(restrictions.AUTOMATION)
        elif not self.enable_redeploy_after_change_of_mac_address:
            result.append(restrictions.AUTOMATED_REDEPLOY)
        return result
