from mongoengine import StringField, LongField, ListField, DictField

from walle.errors import ResourceNotFoundError
from walle.models import Document, register_model
from walle.util.mongo import MongoDocument, SECONDARY_LOCAL_DC_PREFERRED
from walle.util.mongo.document import DocumentDoesNotExist

_formats = {
    "yasm": "yasm_data",
    "solomon": "solomon_data",
}


@register_model
class Metrics(Document):
    id = StringField(primary_key=True, required=True, help_text="Metrics ID")
    time = LongField(help_text="Time when the metrics has been collected")
    yasm_data = ListField(required=False, help_text="Collected metrics in yasm format")
    solomon_data = DictField(required=False, help_text="Collected metrics in solomon format")

    meta = {"collection": "metrics"}


def get(metrics_id: str, format: str):
    metrics_document = MongoDocument.for_model(Metrics)
    format_field = _formats.get(format, "yasm_data")
    fields = ("time", format_field)
    try:
        metrics = metrics_document.find_one(
            {"_id": metrics_id}, fields=fields, read_preference=SECONDARY_LOCAL_DC_PREFERRED
        )
        if format_field == "yasm_data":
            data = metrics.yasm_data
        if format_field == "solomon_data":
            data = metrics.solomon_data
        return data, metrics.time
    except DocumentDoesNotExist:
        raise ResourceNotFoundError("The metrics haven't been collected yet.")


def get_yasm_metrics(metrics, prefix=None):
    """Converts Wall-E metrics to yasm metrics format.

    See https://wiki.yandex-team.ru/golovan/userdocs/stat-handle/ for details.
    """

    return [
        [(metric if prefix is None else prefix + "-" + metric).replace("_", "-") + "_attt", _get_yasm_value(value)]
        for metric, value in metrics.items()
    ]


def _get_yasm_value(value):
    if isinstance(value, bool):
        return int(value)
    else:
        return value
