"""Host network management.

See definitive guide to L3 networks - https://clubs.at.yandex-team.ru/sysadmin/10327
"""

import datetime
import logging
import re
import time
import typing as tp
from collections import namedtuple
from string import Formatter

import mongoengine
from mongoengine import StringField, DateTimeField, FloatField

from sepelib.core.exceptions import Error, LogicalError
from sepelib.mongo.util import register_model
from sepelib.util.misc import doesnt_override
from walle import constants as walle_constants
from walle.clients import bot, eine, racktables, yasubr
from walle.clients.network.racktables_client import RacktablesClient
from walle.constants import NetworkTarget, YCNetworkState
from walle.errors import (
    InvalidHostNameError,
    InvalidHostConfiguration,
    NoInformationError,
    YaSubrRecordNotFoundError,
    HostNameTemplateError,
)
from walle.host_network import HostNetwork
from walle.hosts import HostStatus, HostNetworkLocationInfo
from walle.models import Document
from walle.physical_location_tree import LocationNamesMap
from walle.util import net
from walle.util.text import enumeration_join

DNS_REQUIRED_PROJECT_FIELDS = (
    "id",
    "vlan_scheme",
    "native_vlan",
    "extra_vlans",
    "dns_domain",
    "hbf_project_id",
    "yc_dns_zone_id",
)
FB_PREFIX = "fb-"

log = logging.getLogger(__name__)

DnsRecord = namedtuple("DnsRecord", ["type", "name", "value"])


class NoNetworkOnSwitch(Error):
    def __init__(self, switch, vlan):
        super().__init__("{} switch have no L3 networks for {} VLAN.", switch, vlan)


@register_model
class BlockedHostName(Document):
    """Hold hostnames that shouldn't be given to new hosts for some time"""

    fqdn = StringField(
        primary_key=True, min_length=1, required=True, help_text="FQDN that is free but shouldn't be given to any host"
    )
    host_released_at = DateTimeField(required=True)
    name_blocked_until = DateTimeField(required=True)
    release_timestamp = FloatField(help_text="Time when host has been released")

    meta = {
        "collection": "blocked_host_names",
        "indexes": [
            {"fields": ["name_blocked_until"], "expireAfterSeconds": 0},  # expire at the very moment
        ],
    }

    api_fields = ("fqdn", "release_timestamp")

    DEFAULT_FQDN_RELEASE_TIMEOUT_DAYS = 365 * 20  # no leap years :)

    @classmethod
    @doesnt_override(Document)
    def store(cls, fqdn, timeout_days=None):
        if timeout_days is None:
            timeout_days = cls.DEFAULT_FQDN_RELEASE_TIMEOUT_DAYS

        timeout_days = datetime.timedelta(days=timeout_days)
        utc_now = datetime.datetime.utcnow()
        cls(
            fqdn=fqdn.lower(),
            host_released_at=utc_now,
            release_timestamp=time.time(),
            name_blocked_until=utc_now + timeout_days,
        ).save()

    @classmethod
    @doesnt_override(Document)
    def exists(cls, fqdn):
        """Check if hostname is blocked"""
        try:
            # keep __iexact until no more mixed-case fqdn left in collection.
            cls.objects.get(fqdn__iexact=fqdn.lower())
        except cls.DoesNotExist:
            return False
        else:
            return True


class HostNameTemplate:
    def __init__(self, template, location, regex_fqdn, regex_short, upper_bound=None, with_buckets=False):
        self._format = template
        self._location = location
        self._regex_fqdn = regex_fqdn
        self._regex_short = regex_short

        if with_buckets and not upper_bound:
            raise LogicalError
        self._upper_bound = upper_bound
        self._with_buckets = with_buckets

    def fqdn_matches(self, fqdn):
        return self._regex_fqdn.match(fqdn) is not None

    def shortname_matches(self, fqdn):
        return self._regex_short.match(fqdn) is not None

    def fill(self, index):
        if self._with_buckets:
            bucket = index // self._upper_bound
            index -= bucket * self._upper_bound
        else:
            bucket = None

        if self._with_buckets and self._upper_bound and index >= self._upper_bound:
            raise LogicalError()
        if self._with_buckets and bucket > 9:
            raise HostNameTemplateError("All suitable host names are taken")

        return self._format.format(location=self._location, bucket=bucket, index=index)

    def get_index(self, fqdn):
        match = self._regex_short.match(fqdn)
        if not match:
            # don't run this on host names that don't match
            raise LogicalError

        numbers = match.groupdict()
        if self._with_buckets:
            return int(numbers["bucket"] + numbers["index"])
        else:
            return int(numbers["index"])


def get_default_host_name_template(location, domain):
    size = 4

    template = "{{location}}{{bucket}}-{{index:0{size}d}}".format(size=size)
    template = _make_full_fqdn_template_from_short(template, domain)

    regex_short = r"^{location}(?P<bucket>\d)-(?P<index>\d{{{size}}})\.".format(location=location, size=size)
    regex_full = re.compile(_make_full_fqdn_regex_from_short(regex_short, domain))
    regex_short = re.compile(regex_short)

    return HostNameTemplate(template, location, regex_full, regex_short, upper_bound=10**size, with_buckets=True)


def get_free_host_name_template():
    """Returns template for generation host names for free hosts."""

    domain = walle_constants.WALLE_HOST_FQDN_SUFFIX

    template = _make_full_fqdn_template_from_short("free-{index}", domain)
    regex_short = r"^free-(?P<index>\d+)\."

    return HostNameTemplate(
        template,
        location="free",
        regex_fqdn=re.compile(_make_full_fqdn_regex_from_short(regex_short, domain)),
        regex_short=re.compile(regex_short),
    )


def get_custom_host_name_template(template, location, domain):
    with_buckets, upper_bound, regex_short = _analyze_template(template, location)

    regex_full = re.compile(_make_full_fqdn_regex_from_short(regex_short, domain))
    regex_short = re.compile(regex_short)

    template = _make_full_fqdn_template_from_short(template, domain)
    return HostNameTemplate(
        template, location, regex_full, regex_short, upper_bound=upper_bound, with_buckets=with_buckets
    )


def _analyze_template(template, location):
    with_buckets = False
    upper_bound = None

    short_name_regex_parts = [r"^"]

    for literal_text, field_name, format_spec, conversion in Formatter().parse(template):
        short_name_regex_parts.append(re.escape(literal_text))
        if field_name == "location":
            short_name_regex_parts.append(location)

        if field_name == "bucket":
            with_buckets = True
            if format_spec.startswith("0"):
                # custom fixed size bucket
                # this actually is not supported yet :)
                bucket_size = int(re.sub(r"\D+", "", format_spec))
                short_name_regex_parts.append(r"(?P<bucket>\d{{{size}}})".format(size=bucket_size))
            else:
                # default bucket size is 1
                short_name_regex_parts.append(r"(?P<bucket>\d)")

        if field_name == "index":
            if format_spec.startswith("0"):
                size = int(re.sub(r"\D+", "", format_spec))
                upper_bound = 10**size
                short_name_regex_parts.append(r"(?P<index>\d{{{size}}})".format(size=size))
            else:
                # default bucket size is not limited actually
                short_name_regex_parts.append(r"(?P<index>\d+)")

    short_name_regex_parts.append(r"\.")
    short_name_regex = "".join(short_name_regex_parts)

    return with_buckets, upper_bound, short_name_regex


def _make_full_fqdn_template_from_short(template, domain):
    return template + "." + domain


def _make_full_fqdn_regex_from_short(regex_short, domain):
    return regex_short + re.escape(domain) + r"$"


def get_host_name_template(host):
    """Returns template for host name generation for the specified host.

    :raises InvalidHostConfiguration
    """

    if (
        host.status != HostStatus.INVALID
        and host.location is not None  # Invalid hosts may have an outdated location info
        and host.location.physical_timestamp is not None
    ):
        location = bot.HardwareLocation(
            host.location.country,
            host.location.city,
            host.location.datacenter,
            host.location.queue,
            host.location.rack,
            host.location.unit,
        )
        location_name = host.location.short_datacenter_name
    else:
        bot_info = bot.get_host_info(host.inv)
        if bot_info is None:
            raise InvalidHostConfiguration("The host is not registered in Bot.")

        location = bot_info["location"]
        location_name = LocationNamesMap.get_map().get_name(location, "datacenter", raise_on_missing=False)

    project = host.get_project(fields=["vlan_scheme", "dns_domain", "host_shortname_template"])
    if project.vlan_scheme not in walle_constants.DNS_VLAN_SCHEMES:
        raise InvalidHostConfiguration(
            "Host name auto-assigning is supported only for {} VLAN schemes.",
            enumeration_join(walle_constants.DNS_VLAN_SCHEMES),
        )

    if project.dns_domain is None:
        raise InvalidHostConfiguration("DNS auto-configuration is not enabled for project '{}'", project.id)

    if not location_name:
        try:
            location_name = bot.get_rt_location(location)
        except NoInformationError:
            # To avoid this error: add rt location name into `walle.clients.bot.get_rt_location`
            # and add shortname for dc and queue via scripts/dc_names (see st://WALLE-1523 for more information)
            raise InvalidHostConfiguration(
                "Host name auto-assigning is supported only for some hardcoded locations. "
                "{} location is not supported yet.",
                location.get_path(),
            )

    location = location_name.lower()
    if project.host_shortname_template:
        return get_custom_host_name_template(project.host_shortname_template, location, project.dns_domain)
    else:
        return get_default_host_name_template(location, project.dns_domain)


def get_host_fqdns(host):
    """Returns a list of FQDNs that should be registered in DNS for the host."""

    vlan_scheme = host.get_project(fields=("vlan_scheme",)).vlan_scheme

    if vlan_scheme in walle_constants.FASTBONE_VLAN_SCHEMES:
        return _get_host_fqdns_for_fastbone_vlan_scheme(host.name)
    else:
        return [host.name]


def assert_dns_assigning_supported(project, base_name):
    if project.vlan_scheme not in walle_constants.DNS_VLAN_SCHEMES:
        raise InvalidHostConfiguration(
            "Host name auto-assigning is supported only for {} VLAN schemes at this moment.",
            enumeration_join(walle_constants.DNS_VLAN_SCHEMES),
        )

    if project.dns_domain is None:
        raise InvalidHostConfiguration("DNS auto-configuration is not enabled for project '{}'.", project.id)

    if not base_name.endswith(project.dns_domain):
        raise InvalidHostConfiguration(
            "Host name auto-assigning is only allowed for hosts in '{}' domain for project '{}'.",
            project.dns_domain,
            project.id,
        )


def get_host_dns_records(host, project, base_name, switch, mac) -> list[DnsRecord]:
    """Returns a list of DNS records that should be registered in DNS for the host.

    :raises InvalidHostConfiguration
    :raises NoNetworkOnSwitch
    """

    assert_dns_assigning_supported(project, base_name)
    if project.vlan_scheme == walle_constants.VLAN_SCHEME_SEARCH:
        return _get_host_dns_records_for_search_scheme(
            host, base_name, switch, project.native_vlan, mac, project.vlan_scheme
        )

    if project.vlan_scheme in walle_constants.VLAN_SCHEMES_WITHOUT_FASTBONE:
        return _get_host_dns_records_for_scheme_without_fastbone(
            host, base_name, switch, project.native_vlan, mac, project.vlan_scheme
        )

    if project.vlan_scheme == walle_constants.VLAN_SCHEME_MTN:
        return _get_host_dns_records_for_mtn_scheme(
            host,
            base_name,
            project.native_vlan,
            ipv6_producer=_get_hbf_ipv6_producer(
                switch, net.get_hbf_ipv6_address_with_mac_hostid, project.hbf_project_id, mac
            ),
        )

    if project.vlan_scheme == walle_constants.VLAN_SCHEME_MTN_HOSTID:
        return _get_host_dns_records_for_mtn_scheme(
            host,
            base_name,
            project.native_vlan,
            ipv6_producer=_get_hbf_ipv6_producer(
                switch, net.get_hbf_ipv6_address_with_hostname_hostid, project.hbf_project_id, base_name
            ),
        )

    if project.vlan_scheme == walle_constants.VLAN_SCHEME_STATIC:
        return _get_host_dns_records_for_static_scheme(host, base_name, switch, project.native_vlan, mac)

    raise LogicalError()  # we should get here. Force assert_dns_assigning_supported to check conditions.


# Represents what wall-e thinks about host's VLAN configuration.
_VlanConfig = namedtuple("VlanConfig", ["vlans", "native_vlan", "mtn_project_id"])


def get_host_expected_vlans(host, project=None):
    """Returns VLANs that should be assigned to the host.

    :returns _VlanConfig with expected vlans and auto-configuration flag.
    :raises InvalidHostConfiguration if host is configured for VLAN auto-configuration but we can't apply the specified
            VLAN assigning scheme to it.
    """

    if project is None:
        project = host.get_project(fields=("vlan_scheme", "native_vlan", "extra_vlans", "hbf_project_id"))

    vlans, native_vlan, hbf_project_id = None, None, None

    if project.vlan_scheme:
        vlan_scheme, native_vlan, extra_vlans = project.vlan_scheme, project.native_vlan, project.extra_vlans

        if vlan_scheme == walle_constants.VLAN_SCHEME_STATIC:
            vlans = sorted({native_vlan} | set(extra_vlans or []) | set(host.extra_vlans or []))

        elif vlan_scheme == walle_constants.VLAN_SCHEME_SEARCH:
            _, fb_vlan = _get_fb_vlan_for_search_scheme(host, native_vlan)
            vlans = sorted({native_vlan} | {fb_vlan} | set(extra_vlans or []) | set(host.extra_vlans or []))

        elif vlan_scheme in walle_constants.MTN_VLAN_SCHEMES:
            if not project.hbf_project_id:
                raise InvalidHostConfiguration("Project have MTN vlan scheme enabled, but hbf_project_id is not set.")

            if vlan_scheme == walle_constants.VLAN_SCHEME_MTN_WITHOUT_FASTBONE:
                vlans = sorted([walle_constants.MTN_NATIVE_VLAN] + walle_constants.MTN_EXTRA_VLANS)
            else:
                vlans = sorted(
                    [walle_constants.MTN_NATIVE_VLAN, walle_constants.MTN_FASTBONE_VLAN]
                    + walle_constants.MTN_EXTRA_VLANS
                )

            hbf_project_id = hex(project.hbf_project_id).lower()
        elif vlan_scheme == walle_constants.VLAN_SCHEME_CLOUD:
            vlans = sorted({native_vlan} | set(extra_vlans or []) | set(host.extra_vlans or []))
        else:
            raise LogicalError()

    return _VlanConfig(vlans, native_vlan, hbf_project_id)


def get_current_host_switch_port(host):
    """Returns host switch/port with best effort up-to-date status."""

    switch, port, update_time, source, switch_info = None, None, None, None, None
    query = dict(uuid=host.uuid, network_switch__exists=True, network_port__exists=True, network_source__exists=True)
    try:
        host_network = HostNetwork.objects.only(
            "network_switch", "network_port", "network_source", "network_timestamp"
        ).get(**query)
    except mongoengine.DoesNotExist:
        pass
    else:
        switch = host_network.network_switch
        port = host_network.network_port
        source = host_network.network_source
        update_time = host_network.network_timestamp

    try:
        client = eine.get_client(eine.get_eine_provider(host.get_eine_box()))
        switch_info = client.get_host_status(host.inv, location=True).switch()
    except eine.EineHostDoesNotExistError as e:
        log.error("Failed to get host #%s information from Eine: %s", str(e))

    if switch_info is not None and (update_time is None or switch_info.timestamp > update_time):
        switch, port, update_time, source = (
            switch_info.switch,
            switch_info.port,
            switch_info.timestamp,
            walle_constants.NETWORK_SOURCE_EINE,
        )

    if switch is None or port is None or source is None:
        raise NoInformationError(
            "Unable to determine switch/port of the host. Please, add Wall-E.agent into your setup configuration."
        )

    return HostNetworkLocationInfo(switch=switch, port=port, source=source, timestamp=update_time)


def _get_fb_vlan_for_search_scheme(host, project_native_vlan):
    """Returns VLAN that should be assigned to the host according to Yandex.Search VLAN assigning scheme.

    :raises InvalidHostConfiguration if we can't apply the VLAN assigning scheme to it.
    """

    bb_v4_ip = _get_host_ipv4(host.name)

    if bb_v4_ip is None:
        fb_vlan = _get_validated_fb_vlan(project_native_vlan, host.location)
    else:
        fb_vlan = _get_fb_vlan_from_ya_subr(bb_v4_ip, project_native_vlan)

    if not fb_vlan:
        fb_vlan = _get_validated_fb_vlan(project_native_vlan, host.location)

    return bb_v4_ip, fb_vlan


def _get_fb_vlan_from_ya_subr(bb_v4_ip, project_native_vlan):
    try:
        bb_vlan, fb_vlan = yasubr.get_host_vlans(bb_v4_ip)
        if bb_vlan != project_native_vlan:
            raise InvalidHostConfiguration(
                "Host's backbone VLAN according to ya.subr is {}, but the project's native VLAN is {}.",
                bb_vlan,
                project_native_vlan,
            )
    except YaSubrRecordNotFoundError:
        fb_vlan = None
    return fb_vlan


def _get_host_ipv4(hostname):
    try:
        v4_ips, _v6_ips = net.get_host_ips(hostname)
    except InvalidHostNameError:
        ipv4 = None
    else:
        if not v4_ips:
            ipv4 = None
        elif len(v4_ips) != 1:
            raise InvalidHostConfiguration("Host have more than one IPv4 address.")
        else:
            ipv4 = list(v4_ips)[0]
    return ipv4


def _get_host_dns_records_for_static_scheme(host, fqdn, switch, vlan, mac):
    """Returns a list of DNS records that should be registered in DNS for the host.

    :param fqdn:
    :raises InvalidHostConfiguration
    :raises NoNetworkOnSwitch
    """

    ipv4 = _get_host_ipv4(host.name)
    return _get_host_dns_records_with_ipv4_support(ipv4, fqdn, [(vlan, fqdn)], vlan, mac, switch)


def _get_host_dns_records_for_search_scheme(host, base_name, switch, bb_vlan, mac, vlan_scheme):
    """Returns a list of DNS records that should be registered in DNS for the host.

    :raises InvalidHostConfiguration
    :raises NoNetworkOnSwitch
    """
    bb_v4_ip, fb_vlan = _get_fb_vlan_for_search_scheme(host, bb_vlan)
    fqdns = _get_host_fqdns_for_fastbone_vlan_scheme(base_name)
    fqdns_for_vlans = zip((bb_vlan, fb_vlan), fqdns)
    return _get_host_dns_records_with_ipv4_support(
        bb_v4_ip, base_name, fqdns_for_vlans, bb_vlan, mac, switch, vlan_scheme
    )


def _get_host_dns_records_for_scheme_without_fastbone(host, base_name, switch, bb_vlan, mac, vlan_scheme):
    """Returns a list of DNS records that should be registered in DNS for the host.

    :raises InvalidHostConfiguration
    :raises NoNetworkOnSwitch
    """
    bb_v4_ip = _get_host_ipv4(host.name)
    fqdns_for_vlans = [(bb_vlan, base_name)]
    return _get_host_dns_records_with_ipv4_support(
        bb_v4_ip, base_name, fqdns_for_vlans, bb_vlan, mac, switch, vlan_scheme
    )


def _get_host_dns_records_with_ipv4_support(
    bb_v4_ip, base_name, fqdns_for_vlans, bb_vlan, mac, switch, vlan_scheme=None
):
    dns_records = []
    if bb_v4_ip is not None:
        dns_records.append(DnsRecord("A", base_name, [bb_v4_ip]))

    for vlan, fqdn in fqdns_for_vlans:
        ip_list = _get_ipv6_list_with_eui64_and_ipv4_support(bb_v4_ip, bb_vlan, mac, switch, vlan, vlan_scheme)

        dns_records.append(DnsRecord("AAAA", fqdn, ip_list))

    return dns_records


def _get_ipv6_list_with_eui64_and_ipv4_support(bb_v4_ip, bb_vlan, mac, switch, vlan, vlan_scheme):

    # There is some tricky logic here.
    # If the records not found, we create only one record for the host.
    # For backbone we create a record with ipv4-based IPv6 address if IPv4 is present, EUI-64 otherwise.
    # For fastbone we create a record with EUI-64 IPv6 address.
    # But to allow old hosts keeping their existing IP-addresses and DNS records
    # we return all IP-address types here, with our preferred IP-address first.
    # Then DNS-fixer should use the first IP-address from the list to create a new DNS record.

    ip_v4_based_list = []
    if bb_v4_ip is not None:
        ip_v4_based_address = _get_ipv4_based_ipv6_address(bb_v4_ip, vlan)
        if ip_v4_based_address is not None:
            ip_v4_based_list.append(ip_v4_based_address)

    ip_mac_based_list = _get_eui64_addresses(mac, switch, vlan, vlan_scheme)

    if vlan == bb_vlan:
        # Generating IP-address for backbone interface, prefer IPv4-based IP-address.
        ip_list = ip_v4_based_list + ip_mac_based_list
    else:
        # Generating IP-address for fastbone interface, prefer EUI-64 scheme.
        ip_list = ip_mac_based_list + ip_v4_based_list

    return ip_list


def _get_eui64_addresses(mac, switch, vlan, vlan_scheme):
    networks = racktables.get_vlan_networks(switch, vlan, vlan_scheme)
    if networks is None:
        raise NoNetworkOnSwitch(switch, vlan)

    ip_list = []
    for network in networks:
        try:
            ip_list.append(net.get_eui_64_address(network, mac))
        except net.InvalidNetworkError as e:
            raise InvalidHostConfiguration("Can't generate an IPv6 address for {} network: {}", network, e)

    return ip_list


def _get_ipv4_based_ipv6_address(bb_v4_ip, vlan):
    """Get corresponding network from ya.subr script and return IPv4-embedded IPv6 address for the network.
    Return None if network for the requested VLAN should be configured via RA.
    """
    network = yasubr.get_host_network(bb_v4_ip, vlan)
    if network is None:
        return None

    try:
        ip = net.get_ipv4_embedded_ipv6_address(network, bb_v4_ip)
    except net.InvalidNetworkError as e:
        raise InvalidHostConfiguration("Can't generate an IPv6 address for {} network: {}", network, e)
    return ip


def _get_host_dns_records_for_mtn_scheme(host, base_name, bb_vlan, ipv6_producer):
    """Returns a list of DNS records that should be registered in DNS for the host.

    :raises InvalidHostConfiguration
    :raises NoNetworkOnSwitch
    """
    bb_v4_ip = _get_host_ipv4(host.name)
    if bb_v4_ip is None or racktables.is_nat64_network(bb_v4_ip):
        fb_vlan = _get_validated_fb_vlan(bb_vlan, host.location)
    else:
        raise InvalidHostConfiguration("MTN hosts with non-tunnels v4 addresses are not supported.")

    dns_records = []

    if bb_v4_ip is not None:
        dns_records.append(DnsRecord("A", base_name, [bb_v4_ip]))

    fqdns = _get_host_fqdns_for_fastbone_vlan_scheme(base_name)
    fqdns_for_vlans = zip((bb_vlan, fb_vlan), fqdns)
    dns_records.extend(DnsRecord("AAAA", fqdn, ipv6_producer(vlan)) for vlan, fqdn in fqdns_for_vlans)

    return dns_records


def _get_hbf_ipv6_producer(switch, hbf_ipaddress_producer, project_id, *hostid_args):
    def _get_hbf_ipv6_addresses(vlan):
        networks = racktables.get_vlan_networks(switch, vlan)
        if networks is None:
            raise NoNetworkOnSwitch(switch, vlan)

        ip_list = []
        for network in networks:
            try:
                ip_list.append(hbf_ipaddress_producer(network, project_id, *hostid_args))
            except net.InvalidNetworkError as e:
                raise InvalidHostConfiguration("Can't generate an IPv6 address for {} network: {}", network, e)

        return ip_list

    return _get_hbf_ipv6_addresses


def _get_validated_fb_vlan(bb_vlan, host_location):
    fb_vlans = RacktablesClient.get_fb_vlans(bb_vlan, host_location)

    if not fb_vlans:
        raise InvalidHostConfiguration("The host doesn't have fastbone VLAN.")

    if len(fb_vlans) > 1:
        raise InvalidHostConfiguration(
            "The host has more than one fastbone VLAN: {}.", ",".join(str(vlan) for vlan in fb_vlans)
        )

    fb_vlan = fb_vlans[0]
    if fb_vlan == bb_vlan:
        raise InvalidHostConfiguration("The host has the same VLAN for backbone and fastbone - {}.", bb_vlan)

    return fb_vlan


def _get_host_fqdns_for_fastbone_vlan_scheme(base_name):
    """Returns a list of FQDNs that should be registered in DNS for the host."""

    return [base_name, FB_PREFIX + base_name]


def try_to_remove_fb_prefix(host) -> tp.Optional[str]:
    if host.startswith(FB_PREFIX):
        return host[len(FB_PREFIX) :]


def get_yc_state_from_target(target):
    if target in [NetworkTarget.PROJECT, NetworkTarget.DEPLOY]:
        return YCNetworkState.PROD

    if target in [NetworkTarget.SERVICE, NetworkTarget.PARKING]:
        return YCNetworkState.SETUP

    return YCNetworkState.UNKNOWN
