from collections import namedtuple
from enum import Enum

from walle import restrictions

_OperationDef = namedtuple("_OperationDef", ("type", "restrictions", "host_status", "operation_name"))


class Operation(_OperationDef, Enum):
    def __new__(cls, *args, **kwargs):
        elem = _OperationDef.__new__(cls, *args, **kwargs)
        elem._value_ = args[0]
        return elem

    ADD = _OperationDef(
        type="add",
        restrictions=[],
        host_status="adding",
        operation_name="add host",
    )
    DELETE = _OperationDef(
        type="delete",
        restrictions=[],
        host_status="deleting",
        operation_name="delete host",
    )

    REBOOT = _OperationDef(
        type="reboot",
        restrictions=[restrictions.AUTOMATED_REBOOT],
        host_status="rebooting",
        operation_name="reboot the host",
    )
    PROFILE = _OperationDef(
        type="profile",
        restrictions=[restrictions.AUTOMATED_PROFILE],
        host_status="profiling",
        operation_name="profile the host",
    )
    REDEPLOY = _OperationDef(
        type="redeploy",
        restrictions=[restrictions.AUTOMATED_REDEPLOY],
        host_status="deploying",
        operation_name="redeploy the host",
    )

    REPAIR_HARDWARE = _OperationDef(
        type="hardware-repair",
        restrictions=[restrictions.AUTOMATED_REPAIRING],
        host_status="repairing",
        operation_name="repair host hardware",
    )

    REPORT_FAILURE = _OperationDef(
        type="report-failure",
        restrictions=[restrictions.AUTOMATED_HEALING],
        host_status="reporting-failure",
        operation_name="report host failure",
    )
    REPAIR_RACK_FAILURE = _OperationDef(
        type="repair-rack-failure",
        restrictions=[restrictions.AUTOMATED_RACK_REPAIR],
        host_status="repairing-rack",
        operation_name="report rack failure",
    )

    REPAIR_RACK_OVERHEAT = _OperationDef(
        type="repair-rack-overheat",
        restrictions=[restrictions.AUTOMATED_RACK_REPAIR],
        host_status="repairing-rack-overheat",
        operation_name="report rack overheat",
    )

    REPAIR_LINK = _OperationDef(
        type="repair-link",
        restrictions=[restrictions.AUTOMATED_LINK_REPAIR],
        host_status="repairing-link",
        operation_name="repair link",
    )
    REPAIR_BMC = _OperationDef(
        type="repair-bmc",
        restrictions=[restrictions.AUTOMATED_BMC_REPAIR],
        host_status="repairing-bmc",
        operation_name="repair BMC",
    )
    RESET_BMC = _OperationDef(
        type="reset-bmc",
        restrictions=[restrictions.AUTOMATED_BMC_REPAIR],
        host_status="resetting-bmc",
        operation_name="reset BMC",
    )
    CHANGE_MEMORY = _OperationDef(
        type="change-memory",
        restrictions=[restrictions.AUTOMATED_MEMORY_CHANGE],
        host_status="changing-memory",
        operation_name="change memory",
    )
    CHANGE_DISK = _OperationDef(
        type="change-disk",
        restrictions=[restrictions.AUTOMATED_DISK_CHANGE, restrictions.AUTOMATED_PROFILE],
        host_status="changing-disk",
        operation_name="change disk",
    )
    REPAIR_DISK_CABLE = _OperationDef(
        type="replace-disk-cable",
        restrictions=[restrictions.AUTOMATED_DISK_CABLE_REPAIR],
        host_status="replacing-disk-cable",
        operation_name="replace disk cable",
    )
    REPAIR_CPU = _OperationDef(
        type="repair-cpu",
        restrictions=[restrictions.AUTOMATED_CPU_REPAIR],
        host_status="repairing-cpu",
        operation_name="repair CPU",
    )
    REPAIR_GPU = _OperationDef(
        type="repair-gpu",
        restrictions=[restrictions.AUTOMATED_GPU_REPAIR],
        host_status="repairing-gpu",
        operation_name="repair GPU",
    )
    REPAIR_IB_LINK = _OperationDef(
        type="repair-ib-link",
        restrictions=[restrictions.AUTOMATED_LINK_REPAIR, restrictions.AUTOMATED_INFINIBAND_REPAIR],
        host_status="repairing-ib-link",
        operation_name="repair ib_link",
    )
    REPAIR_INFINIBAND = _OperationDef(
        type="repair-infiniband",
        restrictions=[restrictions.AUTOMATED_INFINIBAND_REPAIR],
        host_status="repairing-infiniband",
        operation_name="repair infiniband",
    )
    REPAIR_MEMORY = _OperationDef(
        type="repair-memory",
        restrictions=[restrictions.AUTOMATED_MEMORY_REPAIR],
        host_status="repairing-memory",
        operation_name="repair memory",
    )
    REPAIR_OVERHEAT = _OperationDef(
        type="repair-cpu-overheat",
        restrictions=[restrictions.AUTOMATED_OVERHEAT_REPAIR],
        host_status="repairing-overheat",
        operation_name="repair CPU overheat",
    )
    REPAIR_CAPPING = _OperationDef(
        type="repair-cpu-capping",
        restrictions=[restrictions.AUTOMATED_CAPPING_REPAIR],
        host_status="repairing-capping",
        operation_name="repair CPU capping",
    )
    REPAIR_REBOOTS = _OperationDef(
        type="repair-reboots",
        restrictions=[restrictions.AUTOMATED_REDEPLOY, restrictions.AUTOMATED_PROFILE],
        host_status="repairing-reboots",
        operation_name="repair intermittent reboots",
    )

    SWITCH_TO_MAINTENANCE = _OperationDef(
        type="switch-to-maintenance",
        restrictions=[],
        host_status="switching-to-maintenance",
        operation_name="maintenance",
    )
    SWITCH_TO_ASSIGNED = _OperationDef(
        type="switch-to-assigned",
        restrictions=[],
        host_status="switching-to-assigned",
        operation_name="set-assigned",
    )

    DEACTIVATE = _OperationDef(
        type="deactivate",
        restrictions=[],
        host_status="deactivating",
        operation_name="deactivate",
    )
    PREPARE = _OperationDef(
        type="prepare",
        restrictions=[],
        host_status="preparing",
        operation_name="prepare",
    )
    POWER_ON = _OperationDef(
        type="power-on",
        restrictions=[],
        host_status="powering-on",
        operation_name="power on",
    )
    POWER_OFF = _OperationDef(
        type="power-off",
        restrictions=[],
        host_status="powering-off",
        operation_name="power off",
    )
    CHECK_DNS = _OperationDef(
        type="check-dns",
        restrictions=[restrictions.AUTOMATED_DNS],
        host_status="checking-dns",
        operation_name="check dns",
    )
    SWITCH_VLANS = _OperationDef(
        type="switch-vlans",
        restrictions=[],
        host_status="switching-vlans",
        operation_name="switch VLAN",
    )
    SWITCH_PROJECT = _OperationDef(
        type="switch-project",
        restrictions=[],
        host_status="switching-project",
        operation_name="switch project",
    )
    RELEASE_HOST = _OperationDef(
        type="release-host",
        restrictions=[],
        host_status="releasing-host",
        operation_name="release host",
    )
    FQDN_DEINVALIDATION = _OperationDef(
        type="fqdn-deinvalidation",
        restrictions=[],
        host_status="fqdn-deinvalidating",
        operation_name="fqdn deinvalidation",
    )

    CANCEL_HEALING = _OperationDef(
        type="cancel-healing", restrictions=[], host_status=None, operation_name="cancel healing"
    )

    BOT_PROJECT_SYNC = _OperationDef(
        type="bot-project-sync",
        restrictions=[],
        host_status="syncing-bot-project",
        operation_name="sync bot project",
    )

    BOT_WAIT_FOR_HOST_ACQUIREMENT = _OperationDef(
        type="bot-wait-for-host-acquirement",
        restrictions=[],
        host_status="waiting-host-acquirement",
        operation_name="bot wait for host acquirement",
    )

    FOOBAR = _OperationDef(type="foobar", restrictions=[], host_status="foobaring", operation_name="foobar")

    REPORT_SECOND_TIME_NODE = _OperationDef(
        type="report-2nd-time-node",
        restrictions=[restrictions.AUTOMATED_REDEPLOY, restrictions.AUTOMATED_PROFILE],
        host_status="report-2nd-time-node",
        operation_name="report 2nd time node to EXP",
    )

    PUSH_CAUTH_SETTING = _OperationDef(
        type="push-cauth-settings",
        restrictions=[],
        host_status="pushing-cauth-settings",
        operation_name="Push host CAuth settings to CAuth",
    )


OPERATION_CHOICES = [op.type for op in Operation]
"""List of operation ids for using in validators"""

OPERATION_HOST_STATUSES = [op.host_status for op in Operation if op.host_status is not None]
"""List of host statuses defined in operations."""
