"""Host operations log.

Used in DMC to decide whether an action should be processed on host or whether it's useless because it has been
processed some time ago and retry shouldn't help.

For now stores only fully processed automated host healing tasks. Be very careful if you decide to store also manual
tasks here: all code that uses this module doesn't expect this.
"""
import logging

from mongoengine import StringField, IntField, DictField, LongField
from pymongo import DESCENDING

from sepelib.core.exceptions import LogicalError
from sepelib.mongo.util import register_model
from walle.hosts import TaskType
from walle.models import Document, timestamp
from walle.operations_log.constants import OPERATION_CHOICES
from walle.util.limits import check_timed_limits, TimedLimit
from walle.util.misc import drop_none

log = logging.getLogger(__name__)


@register_model
class OperationLog(Document):
    id = StringField(primary_key=True, required=True, help_text="ID")
    audit_log_id = StringField(
        required=True, help_text="Audit log ID of the task within which the operation has been processed"
    )

    host_inv = IntField(required=True, help_text="Host inventory number")
    host_name = StringField(help_text="Host name")
    host_uuid = StringField(help_text="Host UUID")

    project = StringField(help_text="Host's project ID")

    scenario_id = IntField(help_text="Target scenario ID (when applicable)")
    aggregate = StringField(help_text="Host aggregate")
    type = StringField(required=True, choices=OPERATION_CHOICES, help_text="Operation type")
    params = DictField(default=None, help_text="Operation specific parameters")
    task_type = StringField(required=False, choices=TaskType.ALL, help_text="Task type")

    time = LongField(required=True, help_text="Operation completion time")

    default_api_fields = (
        "id",
        "audit_log_id",
        "host_inv",
        "host_name",
        "type",
        "time",
        "scenario_id",
        "host_uuid",
        "project",
    )
    api_fields = default_api_fields + ("params", "aggregate")

    meta = {
        "collection": "operations_log",
        "indexes": [
            {"name": "last_operations_by_host_inv", "fields": ["host_inv", "type", "time"]},
            {"name": "last_operations_by_host_name", "fields": ["host_name", "type", "time"]},
            {"name": "last_operations_by_project", "fields": ["project", "type", "time"]},
            {"name": "reversed_last_operations_by_host_name", "fields": ["host_name", "type", "-time"]},
            {"name": "reversed_last_operations_by_host_inv", "fields": ["host_inv", "type", "-time"]},
            {"name": "reversed_last_operations_by_host_uuid", "fields": ["host_uuid", "type", "-time"]},
        ],
    }


def on_completed_operation(host, operation_type, params=None):
    """Logs the specified operation.

    Attention: Operation log works with assumption that within one task there is only one `operation` and all operation
    retries are done through creation of a new task.
    """

    if host.task is None:
        raise LogicalError()

    try:
        _timestamp = timestamp()
        operation_log_id = "{audit_log_id}.{operation}.{timestamp}".format(
            audit_log_id=host.task.audit_log_id, operation=operation_type, timestamp=_timestamp
        )

        OperationLog(
            id=operation_log_id,
            audit_log_id=host.task.audit_log_id,
            scenario_id=host.scenario_id,
            host_name=host.name,
            host_inv=host.inv,
            host_uuid=host.uuid,
            project=host.project,
            task_type=host.task.type,
            type=operation_type,
            params=params,
            time=_timestamp,
        ).save(force_insert=True)
    except Exception as e:
        log.error("Unexpected error in on_complete_operation for host %s: %s", host.human_id(), e)


def check_limits(host, operation, limits: list[TimedLimit], params=None):
    """Checks limits for the specified operation.

    Attention: If `params` is specified and some log record doesn't have the specified parameter, the record will be
    counted to make limits work for old records without new parameters and also to not always allow an action in case
    of typo in parameter name. So be very careful when checking limits with `params`.
    """

    query = drop_none(
        {
            # Use hostname + inventory number as a host hardware + software configuration identifier.
            #
            # TODO: Consider to use project here instead of / in addition to hostname. For now hostname is preferred because
            # host may drift between projects without changing its actual role and configuration (example is our "search",
            # "search-all" and "search-vlan1464" projects) and at this time host name is a more reliable signal of changed
            # configuration or role.
            OperationLog.host_inv.db_field: host.inv,
            OperationLog.host_name.db_field: host.name,
            OperationLog.type.db_field: operation.type,
            OperationLog.task_type.db_field: TaskType.AUTOMATED_HEALING,
        }
    )

    if params:
        for key, value in params.items():
            if "." in key:
                raise LogicalError()

            query[OperationLog.params.db_field + "." + key] = {"$in": [None, value]}

    return check_timed_limits(OperationLog, query, OperationLog.time, limits, inclusive=False)


def get_last_operation(host):
    query = drop_none(
        {
            OperationLog.host_inv.db_field: host.inv,
            OperationLog.host_name.db_field: host.name,
        }
    )

    last_operation = OperationLog._get_collection().find_one(
        query,
        {"_id": False, OperationLog.type.db_field: True, OperationLog.scenario_id.db_field: True},
        sort=[(OperationLog.time.db_field, DESCENDING)],
    )
    return last_operation


def compare_last_operation(host, operation):
    last_operation = get_last_operation(host)
    return operation.type == last_operation["type"] if last_operation else False


def get_last_n_operations(host, operation, n=10, params=None):
    query = drop_none(
        {
            OperationLog.host_inv.db_field: host.inv,
            OperationLog.host_name.db_field: host.name,
            OperationLog.type.db_field: operation.type,
        }
    )

    if params:
        for key, value in params.items():
            if "." in key:
                raise LogicalError()

            query[OperationLog.params.db_field + "." + key] = {"$in": [value]}

    operations = list(
        OperationLog._get_collection().find(query, sort=[(OperationLog.time.db_field, DESCENDING)], limit=n)
    )
    return operations
