import re

import natsort
from mongoengine import (
    EmbeddedDocument,
    StringField,
    LongField,
    EmbeddedDocumentListField,
    ValidationError,
)

from sepelib.core import config
from walle.clients.bot import HardwareLocation as BotHardwareLocation
from walle.errors import ResourceConflictError
from walle.hosts import Host
from walle.models import Document
from walle.util.misc import get_location_path, drop_none, gevent_idle_iter, LocationSegment
from walle.util.mongo import MongoDocument, SECONDARY_LOCAL_DC_PREFERRED

LOCATION_TREE_ID = "the_only_tree"


class NoShortNameError(ResourceConflictError):
    def __init__(self, path):
        self.path = path
        super().__init__(
            "Datacenter or queue {} does not have a short name. "
            "Please, consult the https://st.yandex-team.ru/WALLESUPPORT.",
            path,
        )


class NoAutogeneratedShortNameError(ResourceConflictError):
    def __init__(self):
        super().__init__(
            "Datacenter or queue have an empty short name. "
            "Please, consult the https://st.yandex-team.ru/WALLESUPPORT."
        )


class LocationRack(EmbeddedDocument):
    name = StringField()

    def __repr__(self):
        return "<{}: name={}>".format(self.__class__.__name__, self.name)


class LocationQueue(EmbeddedDocument):
    name = StringField()
    racks = EmbeddedDocumentListField(LocationRack)

    def __repr__(self):
        return "<{}: name={}>".format(self.__class__.__name__, self.name)


class LocationDatacenter(EmbeddedDocument):
    name = StringField()
    queues = EmbeddedDocumentListField(LocationQueue)

    def __repr__(self):
        return "<{}: name={}>".format(self.__class__.__name__, self.name)


class LocationCity(EmbeddedDocument):
    name = StringField()
    datacenters = EmbeddedDocumentListField(LocationDatacenter)

    def __repr__(self):
        return "<{}: name={}>".format(self.__class__.__name__, self.name)


class LocationCountry(EmbeddedDocument):
    name = StringField()
    cities = EmbeddedDocumentListField(LocationCity)

    def __repr__(self):
        return "<{}: name={}>".format(self.__class__.__name__, self.name)


class LocationTree(Document):
    id = StringField(primary_key=True, required=True, help_text="ID")
    countries = EmbeddedDocumentListField(LocationCountry)
    timestamp = LongField(help_text="Locations tree updated at")

    meta = {"collection": "location_tree"}


def _match_name_regex(val):
    if not re.match("^[a-zA-Z0-9-]+$", val):
        raise ValidationError('value of name have to match pattern "^[a-zA-Z0-9-]+$": {}'.format(val))
    return True


class LocationNamesMap(Document):
    path = StringField(primary_key=True, required=True, help_text="BOT location path")
    name = StringField(required=True, help_text="Location name", validation=_match_name_regex)

    meta = {"collection": "location_names_map"}

    class _Lookup(dict):
        def get_name(self, location, level, raise_on_missing=True):
            try:
                return self[get_location_path(location, level)]
            except KeyError as e:
                # Need to add a short name for the datacenter or queue.
                # See st://WALLE-1523 and scripts/dc_names --help
                if raise_on_missing:
                    raise NoShortNameError(str(e))

            return None

    @classmethod
    def get_map(cls):
        try:
            path_field = cls.path.db_field
            name_field = cls.name.db_field
            return cls._Lookup({i[path_field]: i[name_field] for i in cls.objects.as_pymongo()})
        except cls.DoesNotExist:
            return cls._Lookup()

    def clean(self):
        """
        Check that 'path' is path to datacenter and validate it's name (can be only "^[a-zA-Z]+$")
        """
        locations = self.path.split("|")

        #  if it is a path like "FI|MANTSALA|B"
        if len(locations) == 3:
            #  check that the name consists only of lowercase letters
            if not re.match("^[a-z0-9]+$", self.name):
                raise ValidationError('name of DC have to match pattern "^[a-z0-9]+$": {}'.format(self.name))


def build_tree_for_project(projects):
    return build_matching_tree({"project": {"$in": projects}})


def sort_tree(nodes):
    """
    Sort tree using natural sorting recursively.
    *WARNING* Modifies passed tree, but you need to use returned value.
    """
    nodes = natsort.natsorted(nodes, key=lambda node: node["name"].lower())
    for node in nodes:
        if "nodes" not in node:
            continue
        node["nodes"] = sort_tree(node["nodes"])
    return nodes


def build_matching_tree(query):
    location_names_map = {}
    location_names_map_collection = MongoDocument.for_model(LocationNamesMap)
    location_names_map_entries = location_names_map_collection.find()
    for entry in gevent_idle_iter(location_names_map_entries):
        location_names_map[entry.path] = entry.name

    location_tree = []
    path_to_node_map = {}

    hosts_model = MongoDocument.for_model(Host)
    hosts_entries = hosts_model.find(query, ("location",), read_preference=SECONDARY_LOCAL_DC_PREFERRED)
    for host in gevent_idle_iter(hosts_entries):
        path_parts = _get_path_parts(host.location)
        add_node_to_location_tree(location_tree, location_names_map, path_to_node_map, path_parts)

    return sort_tree(location_tree)


def add_node_to_location_tree(location_tree, location_names_map, path_to_node_map, path_parts, prev_path=""):
    if not path_parts:
        return

    name = path_parts.pop()
    if not prev_path:
        cur_path = name
    else:
        cur_path = "{}|{}".format(prev_path, name)

    with_nodes = len(path_parts) != 0
    if cur_path not in path_to_node_map:
        node = _create_location_node(location_names_map, cur_path, name, with_nodes=with_nodes)

        if not prev_path:
            location_tree.append(node)
        else:
            path_to_node_map[prev_path]['nodes'].append(node)

        path_to_node_map[cur_path] = node

    return add_node_to_location_tree(location_tree, location_names_map, path_to_node_map, path_parts, cur_path)


def _create_location_node(location_names_map, path, name, with_nodes=True):
    location_name = location_names_map.get(path)

    return drop_none({"path": path, "name": name, "short_name": location_name, "nodes": [] if with_nodes else None})


def _get_path_parts(location, default="Unknown"):
    return [
        location.rack or default,
        location.queue or default,
        location.datacenter or default,
        location.city or default,
        location.country or default,
    ]


def build_tree_from_bot():
    """A physical location tree built from cached BOT database."""
    location_tree_model = LocationTree.objects.get(id=LOCATION_TREE_ID)
    location_tree = build_cached_tree(location_tree_model)
    return location_tree


def build_cached_tree(tree_model):
    location_names = LocationNamesMap.get_map()

    def append(where, key, what):
        try:
            where[key].append(what)
        except KeyError:
            where[key] = [what]

    def build_nodes(model, path, attributes):
        current_attribute = attributes[0]
        new_attributes = attributes[1:]
        current_path = path + "|" + model.name if path else model.name

        current_node = {"path": current_path, "name": model.name}
        short_name = location_names.get(current_path, None)
        if short_name:
            current_node["short_name"] = short_name

        if current_attribute:
            for child_model in getattr(model, current_attribute):

                child_node = build_nodes(child_model, current_path, new_attributes)
                if child_node:
                    append(current_node, "nodes", child_node)
        return current_node

    attributes = ["cities", "datacenters", "queues", "racks", ""]

    # Build tree recursively
    output_tree = []
    for country in tree_model.countries:
        output_tree.append(build_nodes(country, "", attributes))

    return output_tree


def get_shortname(location_lookup, bot_location, level, logical_datacenter=None):
    location = LogicalHardwareLocation(bot_location, logical_datacenter)

    shortname = location_lookup.get_name(location, level, raise_on_missing=False)

    if not shortname:
        cities_with_disabled_name_autogeneration = config.get_value(
            "shortnames.cities_with_disabled_name_autogeneration"
        )
        if location.city in cities_with_disabled_name_autogeneration:
            raise NoShortNameError(get_location_path(location, level))

        generate_shortnames_for_location(location_lookup, location)

        shortname = LocationNamesMap.get_map().get_name(location, level)

    return shortname


def generate_shortnames_for_location(location_lookup, location):  # NOTE(alexsmirnov) WALLE-4270
    NON_APLHABETIC_SYMBOLS = ".-_"
    filtered_datacenter_name = "".join(char for char in location.datacenter.lower() if char.isalpha())
    filtered_queue_name = "".join(
        char for char in location.queue.lower() if char.isalpha() or char.isdigit() or char in NON_APLHABETIC_SYMBOLS
    )

    datacenter_path = get_location_path(location, LocationSegment.DATACENTER)
    queue_path = get_location_path(location, LocationSegment.QUEUE)

    short_datacenter_name = filtered_datacenter_name[:6]
    short_queue_name = "{}-{}".format(short_datacenter_name, filtered_queue_name[:5].rstrip(NON_APLHABETIC_SYMBOLS))

    if not short_datacenter_name or not short_queue_name:
        raise NoAutogeneratedShortNameError

    if not location_lookup.get_name(location, LocationSegment.DATACENTER, raise_on_missing=False):
        LocationNamesMap(path=datacenter_path, name=short_datacenter_name).save()

    if not location_lookup.get_name(location, LocationSegment.QUEUE, raise_on_missing=False):
        LocationNamesMap(path=queue_path, name=short_queue_name).save()


class LogicalHardwareLocation:
    def __init__(self, bot_location: BotHardwareLocation, logical_datacenter: str):
        self.country = bot_location.country
        self.city = bot_location.city
        self.physical_datacenter = bot_location.datacenter
        self.logical_datacenter = logical_datacenter
        self.queue = bot_location.queue
        self.rack = bot_location.rack
        self.unit = bot_location.unit

    @property
    def datacenter(self):
        return self.logical_datacenter or self.physical_datacenter
