"""Server preorder management."""

import logging
from io import StringIO

import mongoengine
from mongoengine import BooleanField, StringField, ListField, IntField

import walle.expert.automation
import walle.tasks
from sepelib.core.exceptions import LogicalError
from sepelib.mongo.util import register_model
from walle import audit_log, authorization, constants as walle_constants, host_operations, restrictions
from walle.admin_requests.severity import EineTag
from walle.authorization import blackbox
from walle.clients import abc, bot
from walle.errors import (
    ResourceAlreadyExistsError,
    NoInformationError,
    InvalidHostConfiguration,
    InvalidProfileNameError,
    InvalidDeployConfiguration,
    InvalidHostStateError,
    ResourceConflictError,
    UnauthorizedError,
)
from walle.errors import ResourceNotFoundError
from walle.hosts import HostState, HostStatus, TaskType
from walle.models import Document
from walle.util import notifications
from walle.util.misc import parallelize_processing, fix_mongo_set_kwargs
from walle.util.tasks import RequestRejectedByCmsError

log = logging.getLogger(__name__)

PREORDER_DISPENSER_BOTS = ["robot-di-bot-prod"]


class PreorderNotFoundError(ResourceNotFoundError):
    def __init__(self):
        super().__init__("The specified preorder ID doesn't exist.")


class _PreorderStateChangedError(ResourceConflictError):
    def __init__(self, preorder_id):
        super().__init__("Preorder {} has changed it's state.", preorder_id)


@register_model
class Preorder(Document):
    """Represents a preorder for servers."""

    id = IntField(min_value=1, primary_key=True, required=True, help_text="ID")
    owner = StringField(regex=walle_constants.LOGIN_RE, required=True, help_text="Owner")
    issuer = StringField(required=True, help_text="User that added the preorder to Wall-E")

    project = StringField(required=True, help_text="Project ID to add the hosts to")
    bot_project = IntField(required=True, help_text="Project ID of preorder in bot service")
    prepare = BooleanField(required=True, help_text="Prepare taken hosts")
    provisioner = StringField(choices=walle_constants.PROVISIONERS, help_text="Provisioner")
    deploy_config = StringField(min_length=1, help_text="Config to deploy the hosts with")
    restrictions = ListField(
        StringField(choices=restrictions.ALL), default=None, help_text="Restrictions to set for the hosts"
    )

    audit_log_id = StringField(help_text="ID of audit log entry that corresponds to processing of the preorder")
    acquired_hosts = ListField(IntField(), help_text="A list of hosts that have been acquired from the preorder")
    failed_hosts = ListField(IntField(), help_text="A list of hosts that have failed to acquire")

    processed = BooleanField(required=True, help_text="True if the preorder has been processed by Wall-E")
    errors = ListField(StringField(), default=None, help_text="Errors that has occurred during preorder processing")
    messages = ListField(
        StringField(),
        default=None,
        help_text="Messages and notifications that has popped up during preorder processing",
    )

    default_api_fields = ("id", "owner", "issuer", "project", "prepare", "processed", "bot_project")
    api_fields = default_api_fields + (
        "provisioner",
        "deploy_config",
        "restrictions",
        "acquired_hosts",
        "failed_hosts",
        "errors",
        "messages",
    )

    meta = {"collection": "preorders"}

    def authorize(self, issuer, sudo):
        # Protects us from using object without required fields
        if self.id is None or self.owner is None:
            raise LogicalError()

        try:
            blackbox.authorize(
                issuer,
                "You must be owner of #{} preorder to perform this request.".format(self.id),
                owners=[self.owner],
                authorize_admins_with_sudo=sudo,
                authorize_by_group=False,
            )
        except UnauthorizedError:
            if self.owner not in PREORDER_DISPENSER_BOTS:
                raise
            abc_service_id = bot.get_planner_id_by_bot_project_id(self.bot_project)
            abc_service_members = abc.get_service_members(abc_service_id)
            abc_service_hardware_owners = {
                member["person"]["login"]
                for member in abc_service_members
                if member["role"]["code"]
                in {abc.Role.PRODUCT_HEAD, abc.Role.HARDWARE_RESOURCES_OWNER, abc.Role.HARDWARE_RESOURCES_MANAGER}
            }
            issuer_login = authorization.get_issuer_login(issuer)
            if issuer_login not in abc_service_hardware_owners:
                raise UnauthorizedError(
                    "User '%s' must have hardware_resources_owner or hardware_resources_manager or product_head role "
                    "in ABC service '%s' to perform this request." % (issuer_login, abc_service_id)
                )


def check_id(preorder_id):
    """Checks a preorder ID."""

    get_by_id(preorder_id, fields=("id",))


def get_by_id(preorder_id, fields=None):
    """Returns a preorder by its ID."""

    objects = Preorder.objects(id=preorder_id)
    if fields is not None:
        objects = objects.only(*fields)

    try:
        return objects.get()
    except mongoengine.DoesNotExist:
        raise PreorderNotFoundError()


def _process_preorders():
    if not walle.expert.automation.GLOBAL_HEALING_AUTOMATION.is_enabled():
        return

    def process(preorder):
        messages = []
        try:
            _process_preorder(preorder, messages)
        except _PreorderStateChangedError as e:
            # don't write messages to preorder, keep messages that another process saved
            log.error("Got an error during processing #%s preorder: %s", preorder.id, e)

        except Exception as e:
            log.error("Got an error during processing #%s preorder: %s", preorder.id, e)
            messages.append(str(e))
            _store_preorder_messages(preorder, messages)
        else:
            _store_preorder_messages(preorder, messages)

    parallelize_processing(process, Preorder.objects(processed=False), threads=10)


def _process_preorder(preorder, messages):
    """Processes all required actions for acquiring hosts from preorder.

    The function is designed to be reentrant.
    """

    log_prefix = "#{} preorder".format(preorder.id)
    log.info("%s: Processing...", log_prefix)

    if preorder.audit_log_id is None:
        if not _start_processing(preorder):
            return

    try:
        info = bot.get_preorder_info(preorder.id)
    except bot.InvalidPreorderIdError:
        error = "The preorder has been deleted from BOT."
        log.error("%s: %s", log_prefix, error)

        audit_log_id = preorder.audit_log_id
        _modify_in_process(preorder, add_to_set__errors=error, unset__audit_log_id=True, set__processed=True)
        audit_log.fail_request(audit_log_id, error)

        return

    for inv, status in info["servers"].items():
        if inv in preorder.acquired_hosts or inv in preorder.failed_hosts:
            continue

        try:
            log.info("%s: Processing #%s host...", log_prefix, inv)
            _process_host(preorder, inv, status)
        except _PreorderStateChangedError:
            raise
        except Exception as e:
            error_message = "Got an error during processing #{} host: {}.".format(inv, e)
            log.error("%s: %s", log_prefix, error_message)
            messages.append(error_message)

    if info["status"] != bot.PreorderStatus.CLOSED:
        log.info("%s: The preorder is not closed yet. Come back to it later.", log_prefix)
        return

    preorder_invs = set(info["servers"])
    processed_invs = set(preorder.acquired_hosts) | set(preorder.failed_hosts)
    in_process_invs = preorder_invs - processed_invs

    if in_process_invs:
        message = (
            "The preorder is closed, but there are {} hosts that haven't been processed yet. "
            "Come back to them later.".format(len(in_process_invs))
        )
        log.info("%s: %s", log_prefix, message)
        messages.append(message)
        return

    log.info(
        "%s: The preorder is closed and we've processed all %s of its hosts. Consider the preorder as processed.",
        log_prefix,
        len(preorder_invs),
    )

    audit_log_id = preorder.audit_log_id
    _modify_in_process(preorder, unset__audit_log_id=True, set__processed=True, unset__messages=True)

    audit_log.complete_request(audit_log_id)
    _send_preorder_processed_notification(preorder)


def _start_processing(preorder):
    audit_entry = audit_log.on_process_preorder(
        authorization.ISSUER_WALLE, preorder.project, preorder.to_api_obj(Preorder.api_fields)
    )

    with audit_entry:
        started = preorder.modify(dict(processed=False, audit_log_id__exists=False), set__audit_log_id=audit_entry.id)
        if not started:
            audit_entry.cancel("The preorder has changed its state.")

    return started


def restart_processing(issuer, preorder, reason=None):
    if preorder.audit_log_id:
        audit_log.cancel_request(preorder.audit_log_id, reason)

    audit_entry = audit_log.on_restart_preorder(
        issuer, preorder.project, preorder.to_api_obj(Preorder.api_fields), reason=reason
    )

    with audit_entry:
        _restart_preorder(preorder, audit_entry.id)


# noinspection PyTypeChecker
def _send_preorder_processed_notification(preorder):
    subject = "[Wall-E] #{} preorder has been processed".format(preorder.id)
    body = StringIO()

    if preorder.acquired_hosts:
        print("Acquired hosts: {}.".format(" ".join(str(inv) for inv in preorder.acquired_hosts)), file=body)

        print("\nAll acquired hosts have been added to '{}' project".format(preorder.project), end="", file=body)
        if preorder.prepare:
            print(" and sent to prepare stage", end="", file=body)
        print(".", file=body)
    else:
        print("No hosts have been acquired.", file=body)

    if preorder.failed_hosts:
        print("\nFailed hosts: {}.".format(" ".join(str(inv) for inv in preorder.failed_hosts)), file=body)

    if preorder.errors:
        print("\nThe following errors have occurred during processing the preorder:", file=body)
        for error in preorder.errors:
            print("* " + error, file=body)

    recipients = {authorization.get_login_email(preorder.owner)}
    issuer_email = authorization.get_user_email(preorder.issuer)
    if issuer_email is not None:
        recipients.add(issuer_email)

    recipients_config = notifications.get_recipients_config(preorder.project, suppress_errors=True)
    recipients.update(recipients_config.get(notifications.SEVERITY_WARNING, []))

    notifications.send_email(recipients, subject, body.getvalue().strip())


def _process_host(preorder, inv, status):
    """Processes all required actions for configuring the host."""

    if status == bot.ServerStatus.TAKEN:
        return _fail_host(preorder, inv, error="Host #{} has been already taken.".format(inv))

    log_prefix = "#{} preorder: #{}".format(preorder.id, inv)
    name = "new-{preorder}-{inv}.{suffix}".format(
        preorder=preorder.id, inv=inv, suffix=walle_constants.WALLE_HOST_FQDN_SUFFIX
    )
    log.info("%s: Adding the host...", log_prefix)

    try:
        host = host_operations.add_host(
            authorization.ISSUER_WALLE,
            inv=inv,
            default_name=name,
            project=preorder.project,
            preorder_id=preorder.id,
            state=HostState.FREE,
            status=HostStatus.READY,
            reason="Acquire the host from #{} preorder added by {}.".format(
                preorder.id, authorization.get_issuer_name(preorder.issuer)
            ),
        )
    except ResourceAlreadyExistsError:
        return _fail_host(preorder, inv, "Host #{} already exists in Wall-e.".format(inv))
    except (NoInformationError, InvalidHostConfiguration, bot.BotProjectIdNotFound) as e:
        return _fail_host(preorder, inv, e)

    log.info("%s: Acquiring #%s...", log_prefix, inv)
    try:
        bot.acquire_preordered_host(preorder.id, inv, host.name)
    except bot.InvalidInventoryNumber as e:
        return _fail_host(preorder, inv, e)

    if preorder.prepare:
        log.info("%s: Schedule preparing of the host...", log_prefix)

        expected_exceptions = (
            InvalidProfileNameError,
            InvalidDeployConfiguration,
            InvalidHostStateError,
            RequestRejectedByCmsError,
        )
        try:
            walle.tasks.schedule_prepare(
                authorization.ISSUER_WALLE,
                TaskType.AUTOMATED_ACTION,
                host,
                skip_profile=True,
                provisioner=preorder.provisioner,
                config=preorder.deploy_config,
                host_restrictions=preorder.restrictions,
                with_auto_healing=True,
                update_firmware=True,
                reason="Host preparing has been requested by #{} preorder added by {}.".format(
                    preorder.id, authorization.get_issuer_name(preorder.issuer)
                ),
                repair_request_severity=EineTag.LOW,
            )
        except expected_exceptions as e:
            return _fail_host(preorder, inv, "Failed to schedule host preparing: {}.".format(e))
    else:
        walle.tasks.schedule_wait_for_bot_acquirement(authorization.ISSUER_WALLE, TaskType.AUTOMATED_ACTION, host)

    _modify_in_process(preorder, add_to_set__acquired_hosts=inv)


def _modify(preorder, query, **kwargs):
    if preorder.modify(query, **kwargs):
        return

    raise _PreorderStateChangedError(preorder.id)


def _modify_in_process(preorder, **kwargs):
    _modify(preorder, dict(processed=False, audit_log_id=preorder.audit_log_id), **kwargs)


def _restart_preorder(preorder, audit_entry_id):
    q_obj = mongoengine.Q(processed=preorder.processed, audit_log_id=preorder.audit_log_id) | mongoengine.Q(
        processed=True, audit_log_id__exists=False
    )

    _modify(
        preorder, {"q_obj": q_obj}, set__audit_log_id=audit_entry_id, set__processed=False, unset__failed_hosts=True
    )


def _store_preorder_messages(preorder, messages):
    """Store or clear preorder messages after is was processed.
    Don't throw exceptions out of here as this function itself is used in an exception handler."""

    if not messages:
        messages = None  # prevent empty lists here, just drop the whole field.

    try:
        # don't save messages for processed preorder
        if not preorder.processed:
            _modify_in_process(preorder, **fix_mongo_set_kwargs(set__messages=messages))
    except Exception as e:
        log.error("Failed to save preoder #%s messages: %s", preorder.id, e)


def _fail_host(preorder, inv, error):
    error = "Failed to acquire #{}: {}".format(inv, error)
    log.error("#%s preorder: %s", preorder.id, error)
    _modify_in_process(preorder, add_to_set__failed_hosts=inv, add_to_set__errors=error)
