"""Collects information about last profile time for each host."""

import logging

from mongoengine import IntField, LongField, NotUniqueError, DoesNotExist

from walle.clients import eine
from walle.models import Document, register_model
from walle.util.gevent_tools import gevent_idle_iter

log = logging.getLogger(__name__)


@register_model
class ProfileStat(Document):
    inv = IntField(min_value=0, primary_key=True, required=True, help_text="Host inventory number")
    time = LongField(min_value=0, required=True, help_text="Last host profile time")
    meta = {"collection": "profile_stat"}


def get_profile_time(inv):
    try:
        return ProfileStat.objects.only("time").get(inv=inv).time
    except DoesNotExist:
        pass


def update_profile_time(inv, profile_time):
    if ProfileStat.objects(inv=inv, time__lt=profile_time).update(set__time=profile_time, multi=False):
        return

    try:
        ProfileStat(inv=inv, time=profile_time).save(force_insert=True)
    except NotUniqueError:
        pass


def sync_profile_stat():
    client = eine.get_client(eine.get_eine_provider(box_id=None))
    eine_profile_stat = client.get_profile_stat()

    walle_profile_stat = {}
    for stat in gevent_idle_iter(ProfileStat.objects(inv__in=list(eine_profile_stat.keys())).only("inv", "time")):
        walle_profile_stat[stat.inv] = stat.time

    for inv, profile_time in gevent_idle_iter(eine_profile_stat.items()):
        try:
            walle_profile_time = walle_profile_stat[inv]
        except KeyError:
            try:
                ProfileStat(inv=inv, time=profile_time).save(force_insert=True)
            except NotUniqueError:
                pass
        else:
            if profile_time > walle_profile_time:
                ProfileStat.objects(inv=inv, time__lt=profile_time).update(set__time=profile_time, multi=False)
