from sepelib.core import config

from walle import authorization, projects, default_cms, restrictions
from walle.clients import staff, inventory
from walle.constants import MTN_IP_METHOD_MAC, ROBOT_WALLE_OWNER, HostType
from walle.errors import BadRequestError, DNSAutomationValidationError
from walle.expert.automation import PROJECT_DNS_AUTOMATION
from walle.idm.project_role_managers import ProjectRole
from walle.projects import (
    NotificationRecipients,
    Notifications,
    CmsSettings,
    Project,
    AutomationSwitch,
    CauthSettingsDocument,
)
from walle.util.api import validate_user_string
from walle.util.misc import drop_none
from walle.util.notifications import SEVERITY_WARNING, SEVERITY_BOT, SEVERITY_CRITICAL
from walle.views.api.common import validated_tags
from walle.views.api.project_api import common
from walle.views.helpers import validators


class ProjectBuilder:
    def __init__(self, issuer):
        self._project_kwargs = {}
        self._project = Project()
        self._issuer = issuer
        self._auth_enabled = config.get_value("authorization.enabled")

    def set_fields(self, **kwargs):
        for field, val in kwargs.items():
            setattr(self._project, field, val)

    def build(self):
        return self._project

    def set_name(self, name):
        self.set_fields(name=validate_user_string(name, "project name"))

    def set_type(self, host_type):
        if not host_type:
            host_type = HostType.SERVER
        self.set_fields(type=host_type)

    def calculate_owners(self, owners):
        if owners is None:
            if not self._auth_enabled:
                owners = []
            else:
                owners = [authorization.get_issuer_login(self._issuer)]
        else:
            if not owners:
                raise BadRequestError("Project owners list mustn't be empty.")

            try:
                owners = staff.check_owners(owners)
            except staff.InvalidOwnerError as e:
                raise BadRequestError(str(e))

        return owners

    def set_notifications(self, recipients, owners):
        if not recipients:
            owners_emails = [authorization.get_login_email(owner) for owner in owners if not staff.is_group(owner)]
            recipients = {
                SEVERITY_WARNING: owners_emails,
                SEVERITY_BOT: owners_emails,
                SEVERITY_CRITICAL: owners_emails,
            }

        notifications = Notifications(recipients=NotificationRecipients())
        for severity, emails in recipients.items():
            setattr(notifications.recipients, severity, sorted(list(set(emails))))

        self.set_fields(notifications=notifications)

    def set_cms_settings(self, cms, cms_settings):
        if cms_settings:
            old_fields_cms_settings = None
            new_fields_cms_settings = []
            unique_cms_urls = set()
            for cms_setting in cms_settings:
                cms_settings_obj = CmsSettings.from_request(cms_setting)
                unique_cms_urls.add(cms_settings_obj.url)
                validators.check_redundant_tvm_app_id(cms_settings_obj)
                old_fields_cms_settings = cms_settings_obj
                new_fields_cms_settings.append(cms_settings_obj.to_cms_settings_document())

            if projects.DEFAULT_CMS_NAME in unique_cms_urls and len(unique_cms_urls) > 1:
                raise BadRequestError("Default cms can be the only one cms of project")

            self.set_fields(**old_fields_cms_settings.to_project())
            self.set_fields(cms_settings=new_fields_cms_settings)
        else:
            if cms is None:
                cms = {
                    "url": projects.DEFAULT_CMS_NAME,
                    "max_busy_hosts": default_cms.DEFAULT_MAX_BUSY_HOSTS,
                }

            cms_settings_obj = CmsSettings.from_request(cms)
            validators.check_redundant_tvm_app_id(cms_settings_obj)
            self.set_fields(**cms_settings_obj.to_project())
            self.set_fields(cms_settings=[cms_settings_obj.to_cms_settings_document()])

    def set_bot_project_id(self, bot_project_id, cms_settings=None):
        if bot_project_id:
            bot_project_id = validators.validated_bot_project_id(bot_project_id)

            if self._auth_enabled:
                issuer_login = authorization.get_issuer_login(self._issuer)
                if not self._is_walle_admin():
                    common.authenticate_user_by_bot_project_id(issuer_login, bot_project_id)
                    if cms_settings:
                        for cms_setting in cms_settings:
                            cms_settings = CmsSettings.from_cms_setting_document(cms_setting)
                            validators.check_cms_tvm_app_id_requirements(cms_settings, bot_project_id)
        else:
            if self._auth_enabled:
                # enable local testing
                raise BadRequestError("BOT project id must be set.")

        self.set_fields(bot_project_id=bot_project_id)

    def set_reboot_via_ssh(self, reboot_via_ssh):
        # this branch is meant only for local use, doing it in prod/testing will lead to inconsistency with IDM
        if not self._auth_enabled and reboot_via_ssh:  # we don't store false
            role_manager = ProjectRole.get_role_manager(ProjectRole.SSH_REBOOTER, project=self.build())
            role_manager.add_member(ROBOT_WALLE_OWNER)

        return reboot_via_ssh

    def set_enable_dns_automation(self, enable):
        enable = enable or False

        if enable:
            try:
                PROJECT_DNS_AUTOMATION.validate_project_dns_settings(self.build())
            except DNSAutomationValidationError as e:
                msg = "DNS healing automation can be enabled only for projects in MTN with DNS domain set: {}".format(e)
                raise BadRequestError(msg)

        self.set_fields(dns_automation=AutomationSwitch(enabled=enable))

    def set_enable_healing_automation(self, enable):
        enable = enable or False
        self.set_fields(healing_automation=AutomationSwitch(enabled=enable))

    def set_owned_vlans(self, owned_vlans):
        owned_vlans = owned_vlans or []

        if owned_vlans and not self._is_walle_admin():
            raise BadRequestError("Only Wall-e administrators can set project's owned VLANs")

        self.set_fields(owned_vlans=sorted(set(owned_vlans)))

    def set_automation_plot_id(self, automation_plot_id):
        self.set_fields(automation_plot_id=automation_plot_id)

    def set_project_tags(self, project_tags):
        self.set_fields(tags=validated_tags(project_tags or []))

    def set_profile(self, profile):
        if profile is not None:
            profile = common.get_validated_profile(profile)

        self.set_fields(profile=profile)

    def set_profile_tags(self, profile_tags):
        if profile_tags is not None:
            profile_tags = common.get_sorted_profile_tags(profile_tags)

        self.set_fields(profile_tags=profile_tags)

    def set_dns_domain(self, dns_domain):
        self.set_fields(dns_domain=dns_domain)

    def set_shortname_template(self, host_shortname_template):
        if not host_shortname_template:
            return

        if not self._project.dns_domain:
            raise BadRequestError("Host shortname can only be specified if dns domain is specified.")

        self.set_fields(host_shortname_template=validators.validated_host_shortname_template(host_shortname_template))

    def set_certificate_deploy(self, certificate_deploy):
        certificate_deploy = certificate_deploy or False
        if certificate_deploy:
            common.check_dns_domain_allowed_in_certificator(self._project.dns_domain)
        self.set_fields(certificate_deploy=certificate_deploy)

    def set_deploy_configuration(self, provisioner, deploy_config, deploy_config_policy, deploy_tags, deploy_network):
        self.set_fields(
            provisioner=provisioner,
            deploy_config=deploy_config,
            deploy_config_policy=deploy_config_policy,
            deploy_tags=deploy_tags,
            deploy_network=deploy_network,
        )

        inventory.check_deploy_configuration(
            provisioner,
            deploy_config,
            projects.get_eine_box(self._project.id),
            deploy_tags,
            self._project.certificate_deploy,
            deploy_network,
            deploy_config_policy,
        )

    def set_default_host_restrictions(self, default_host_restrictions):
        if default_host_restrictions is not None:
            default_host_restrictions = restrictions.strip_restrictions(default_host_restrictions, strip_to_none=True)

        self.set_fields(default_host_restrictions=default_host_restrictions)

    def set_manually_disabled_checks(self, manually_disabled_checks):
        self.set_fields(manually_disabled_checks=manually_disabled_checks)

    def set_hbf_project_id(self, hbf_project_id, ip_method):
        if hbf_project_id:
            hbf_project_id = validators.validated_hbf_project_id(hbf_project_id)
            ip_method = ip_method or MTN_IP_METHOD_MAC
            self.set_fields(**common.set_project_for_mtn_kwargs(hbf_project_id, ip_method))

    def set_cauth_settings(
        self,
        flow_type,
        trusted_sources,
        key_sources,
        secure_ca_list_url,
        insecure_ca_list_url,
        krl_url,
        sudo_ca_list_url,
    ):
        settings = drop_none(
            dict(
                key_sources=key_sources,
                secure_ca_list_url=secure_ca_list_url,
                insecure_ca_list_url=insecure_ca_list_url,
                krl_url=krl_url,
                sudo_ca_list_url=sudo_ca_list_url,
            )
        )
        flow_type, trusted_sources = validators.check_cauth_settings(flow_type, trusted_sources)
        if flow_type:
            settings["flow_type"] = flow_type
            settings["trusted_sources"] = trusted_sources

        if settings:
            self.set_fields(cauth_settings=CauthSettingsDocument(**settings))

    def set_logical_datacenter(self, logical_datacenter):
        self.set_fields(logical_datacenter=logical_datacenter)

    def _is_walle_admin(self):
        issuer_login = authorization.get_issuer_login(self._issuer)
        return authorization.is_admin(issuer_login)
