"""Project management."""

import logging
import typing as tp
from collections import namedtuple

import mongoengine
from mongoengine import (
    BooleanField,
    LongField,
    StringField,
    ListField,
    DictField,
    EmbeddedDocument,
    EmbeddedDocumentField,
    IntField,
    EmbeddedDocumentListField,
)

from sepelib.core import config
from sepelib.core.exceptions import Error, LogicalError
from sepelib.mongo.util import register_model
from walle import boxes
from walle import constants as walle_constants, restrictions
from walle.authorization import blackbox, has_iam
from walle.clients import dns as dns_clients
from walle.clients.cauth import CauthFlowType, CauthSource, CAuthKeySources
from walle.clients.cms import get_cms_client, CmsApiVersion
from walle.constants import NetworkTarget, ROBOT_WALLE_OWNER, DEFAULT_TIER_ID, HostType
from walle.errors import (
    ResourceNotFoundError,
    UnauthorizedError,
    CMSMigrationError,
    OperationRestrictedForHostTypeError,
)
from walle.expert.automation_plot import AUTOMATION_PLOT_ID_REGEX
from walle.expert.types import CheckType, get_walle_check_type
from walle.idm.project_role_managers import ProjectRole
from walle.maintenance_plot.constants import MAINTENANCE_PLOT_ID_REGEX
from walle.maintenance_plot.model import MaintenancePlotModel
from walle.models import Document
from walle.util.deploy_config import DeployConfigPolicies
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import drop_none

PERIOD = "period"

LIMIT = "limit"

DEFAULT_CMS_NAME = "default"

HEALING_AUTOMATION_LIMIT_NAMES = [
    "max_unreachable_failures",
    "max_ssh_failures",
    "max_memory_failures",
    "max_disk_failures",
    "max_link_failures",
    "max_cpu_failures",
    "max_cpu_caches_failures",
    "max_gpu_failures",
    "max_overheat_failures",
    "max_bmc_failures",
    "max_reboots_failures",
    "max_tainted_kernel_failures",
    "max_cpu_capping_failures",
    "max_fs_check_failures",
    "max_rack_failures",
    "max_checks_missing_failures",
    "max_dead_hosts",
    "max_infiniband_failures",
]
"""A list of available healing automation limits.

Attention: The order of limits determines their order in UI.
"""

MAX_DNS_FIXES_LIMIT = "max_dns_fixes"
"""The limit for automated DNS fixes."""

DNS_AUTOMATION_LIMIT_NAMES = [MAX_DNS_FIXES_LIMIT]
"""A list of available DNS automation limits.

Attention: The order of limits determines their order in UI.
"""

AUTOMATION_LIMIT_NAMES = HEALING_AUTOMATION_LIMIT_NAMES + DNS_AUTOMATION_LIMIT_NAMES
"""A list of available automation limits.

Attention: The order of limits determines their order in UI.
"""

HOST_LIMITS_NAMES_WITH_DEFAULT_VALUES = {
    "max_healing_cancellations": [
        {"period": "1h", "limit": 1},
        {"period": "1d", "limit": 2},
        {"period": "7d", "limit": 3},
    ]
}
"""Available host limits with default values."""


DEFAULT_AUTOMATION_LIMIT = 10
"""Default automation limit value."""


DEFAULT_RACK_FAILURE_LIMIT = 50
"""Default rack failure limit value."""


TAG_RE = r"[a-zA-Z0-9_.-]"
"""Validation regex for project tags."""

_SIMPLE_FQDN_RE_STR = r"(?: [a-z0-9]+ (?:-[a-z0-9]+)* \.)+ [a-z]{2,}(?: :[0-9]+ )?"
_URL_PATH_RE = r"""(?:
    /
    (?:[-._0-9a-zA-Z]+/)*
    (?:[-._0-9a-zA-Z]+)?
)*?
"""
# Use strict CMS URL validation to be able to easily compare CMS by comparing their URLs.
# Can't use re.VERBOSE here, because we use the pattern in API request validation schema.
CMS_RE_CUSTOM_URL = (r"^https?://" + _SIMPLE_FQDN_RE_STR + _URL_PATH_RE + "$").replace(" ", "").replace("\n", "")
SIMPLE_FQDN_RE = _SIMPLE_FQDN_RE_STR.replace(" ", "")

DNS_DOMAIN_RE = r"^{}$".format(SIMPLE_FQDN_RE)

STARTREK_QUEUE_PATTERN = r"^[A-Z]+$"
log = logging.getLogger(__name__)


class RepairRequestSeverity:
    HIGH = "high"
    MEDIUM = "medium"
    LOW = "low"

    ALL = [HIGH, MEDIUM, LOW]


class ProjectNotFoundError(ResourceNotFoundError):
    def __init__(self):
        super().__init__("The specified project ID doesn't exist.")


class NotificationRecipients(EmbeddedDocument):
    """Recipients by severity."""

    audit = ListField(StringField(), help_text="Emails for audit severity")
    info = ListField(StringField(), help_text="Emails for info severity")
    warning = ListField(StringField(), help_text="Emails for warning severity")
    bot = ListField(StringField(), help_text="Emails for bot severity")
    error = ListField(StringField(), help_text="Emails for error severity")
    critical = ListField(StringField(), help_text="Emails for critical severity")


class Notifications(EmbeddedDocument):
    """Project notifications configuration."""

    recipients = EmbeddedDocumentField(NotificationRecipients, required=True, help_text="Notification recipients")


class AutomationSwitch(EmbeddedDocument):
    """Project's automation switch."""

    enabled = BooleanField(required=True, help_text="Enable automated healing for the project")
    status_message = StringField(help_text="Reason for automation status change")
    credit = DictField(
        default=None,
        help_text="Credit for automated actions that was given by user when automation has been enabled. "
        "Automated actions spend credit when it exists and not expired. "
        "Automated actions started at the expense of a credit aren't checked against automation limits.",
    )
    credit_end_time = LongField(help_text="Time at which automation credit expires")
    failure_log_start_time = LongField(
        default=0, help_text="Time starting from which failure log will be checked against the automation limits"
    )


class StartrekReport(EmbeddedDocument):
    """Project's settings for daily host problems report."""

    enabled = BooleanField(required=True, help_text="Enable daily reports")
    queue = StringField(required=True, regex=STARTREK_QUEUE_PATTERN, help_text="Create tickets in this queue")
    summary = StringField(help_text="Optional substring to add to the ticket summary")
    extra = DictField(default=None, help_text="Other ticket parameters like task type or epic ticket")


class FsmHandbrake(EmbeddedDocument):
    issuer = StringField(required=True, help_text="The person who pulled the emergency brake handle")
    timestamp = LongField(required=True, help_text="Time when the person pulled the brake handle")
    timeout_time = LongField(required=True, help_text="Time when the emergency brake will be released")
    reason = StringField(required=False, help_text="Reason, emergency description")
    ticket_key = StringField(required=False, help_text="Ticket for the emergency")
    audit_log_id = StringField(help_text="ID of associated audit log entry")


class CmsSettingsDocument(EmbeddedDocument):
    cms = StringField(required=True, help_text="Project's CMS")
    cms_api_version = StringField(choices=CmsApiVersion.ALL, required=False)
    cms_max_busy_hosts = IntField(min_value=0, required=False)
    cms_tvm_app_id = IntField(required=False, help_text="TVM app id of CMS")
    temporary_unreachable_enabled = BooleanField(default=False, help_text="Support of soft-noc-maintenance")


class CauthSettingsDocument(EmbeddedDocument):
    flow_type = StringField(
        choises=CauthFlowType.ALL,
        default=CauthFlowType.CLASSIC,
        help_text="Trusted sources management type",
        required=False,
    )
    trusted_sources = ListField(
        StringField(choices=CauthSource.ALL),
        default=None,
        help_text="Trusted sources list",
        required=False,
    )
    key_sources = ListField(
        StringField(choices=[source.value for source in CAuthKeySources]),
        default=[CAuthKeySources.STAFF],
        help_text="Sources for keys",
        required=False,
    )
    secure_ca_list_url = StringField(required=False)
    insecure_ca_list_url = StringField(required=False)
    krl_url = StringField(required=False)
    sudo_ca_list_url = StringField(required=False)

    def to_request_params(self):
        extra = dict(
            key_sources=",".join(self.key_sources),
            secure_ca_list_url=self.secure_ca_list_url,
            insecure_ca_list_url=self.insecure_ca_list_url,
            krl_url=self.krl_url,
            sudo_ca_list_url=self.sudo_ca_list_url,
        )
        return drop_none(
            {
                "flow": self.flow_type,
                "trusted_sources": ",".join(self.trusted_sources) if self.trusted_sources else None,
                **extra,
            }
        )


@register_model
class Project(Document):
    """Represents a project."""

    id = StringField(
        regex=r"^[a-z][a-z0-9]*(?:-[a-z0-9]+)*$", max_length=32, primary_key=True, required=True, help_text="ID"
    )

    type = StringField(
        choices=HostType.get_choices(),
        required=False,
        help_text="The type of hosts that belong to the project",
        default=HostType.SERVER,
    )

    # Don't allow long names now to not confuse UI
    name = StringField(min_length=1, max_length=32, required=True, help_text="Name")
    tags = ListField(
        StringField(regex=TAG_RE, min_length=2, max_length=32), default=None, help_text="List of project's tags"
    )

    tier = IntField(default=DEFAULT_TIER_ID, help_text="Which tier should handle hosts of the project")

    # Can't make it required because MongoEngine forces required fields to be non-empty
    owners = ListField(StringField(regex=walle_constants.OWNER_RE), help_text="Owners")

    owned_vlans = ListField(
        IntField(min_value=walle_constants.VLAN_ID_MIN, max_value=walle_constants.VLAN_ID_MAX),
        help_text="VLAN IDs owned by the project",
    )
    vlan_scheme = StringField(choices=walle_constants.VLAN_SCHEMES, help_text="VLAN scheme")
    native_vlan = IntField(
        min_value=walle_constants.VLAN_ID_MIN, max_value=walle_constants.VLAN_ID_MAX, help_text="Native VLAN"
    )
    extra_vlans = ListField(
        IntField(min_value=walle_constants.VLAN_ID_MIN, max_value=walle_constants.VLAN_ID_MAX),
        help_text="A static list of VLAN IDs that should be assigned to each project's host",
        default=None,
    )
    host_shortname_template = StringField(
        help_text="Template for short names for the project's hosts. "
        "Python format string with mandatory {location}, {index} and "
        "optional {bucket} placeholders."
    )
    dns_domain = StringField(regex=DNS_DOMAIN_RE, help_text="Domain for DNS auto-configuration")
    yc_dns_zone_id = StringField(help_text="Binded YC DNS zone ID")
    yc_iam_folder_id = StringField(help_text="Binded YC floder ID for IAM")
    hbf_project_id = IntField(help_text="HBF project id")
    bot_project_id = IntField(help_text="BOT/OEBS project the host is assigned to")
    validate_bot_project_id = BooleanField(help_text="Check host bot_project_id when adding into project", default=None)
    automation_plot_id = StringField(regex=AUTOMATION_PLOT_ID_REGEX, max_length=32, help_text="Automation plot ID")
    maintenance_plot_id = StringField(regex=MAINTENANCE_PLOT_ID_REGEX, max_length=32, help_text="Maintenance plot ID")

    reboot_via_ssh = BooleanField(
        default=None, required=False, help_text="Flag to enable reboots via SSH (requires CAuth edits for robot-walle)"
    )

    # Attention:
    # * `profile` and `profile_tags` are optional fields and may be not set for project.
    # * For now we always force `profile` to be one of EINE_PROFILES_WITH_DC_SUPPORT when it's set and be very careful
    #   before allow some other profile: all automation logic has been made with assumption that it is a profile which
    #   backed up by DC engineers, which supported by our ProfileMode and so on.
    profile = StringField(help_text="Default profile")
    profile_tags = ListField(StringField(), default=None, help_text="Default profile's tags")

    provisioner = StringField(choices=walle_constants.PROVISIONERS, required=False, help_text="Default provisioner")
    deploy_config = StringField(min_length=1, required=False, help_text="Default deploy config")
    deploy_tags = ListField(StringField(), default=None, help_text="Default deploy tags")
    deploy_network = StringField(choices=NetworkTarget.DEPLOYABLE, help_text="Deploy network (service or project)")
    deploy_config_policy = StringField(
        choices=DeployConfigPolicies.get_all_names(), help_text="Deploy config policy", required=False
    )
    certificate_deploy = BooleanField(required=True, default=False, help_text="Enable deploy of certificates")

    cms = StringField(required=True, help_text="Project's CMS")
    cms_api_version = StringField(choices=CmsApiVersion.ALL, required=False)
    cms_max_busy_hosts = IntField(min_value=0, required=False)
    cms_tvm_app_id = IntField(required=False, help_text="TVM app id of CMS")
    cms_settings = EmbeddedDocumentListField(CmsSettingsDocument, required=False, help_text="Settings of project cms")

    notifications = EmbeddedDocumentField(Notifications, required=False, help_text="Notifications configuration")
    default_host_restrictions = ListField(
        StringField(choices=restrictions.ALL), default=None, help_text="Default restrictions for new hosts"
    )

    healing_automation = EmbeddedDocumentField(AutomationSwitch, required=True, help_text="Healing automation switch")
    dns_automation = EmbeddedDocumentField(AutomationSwitch, required=True, help_text="DNS automation switch")

    manually_disabled_checks = ListField(StringField(), help_text="Manually disabled checks")
    automation_limits = DictField(required=True, default=None, help_text="Automation limits configuration")
    host_limits = DictField(required=True, default=None, help_text="Host limits configuration")
    fsm_handbrake = EmbeddedDocumentField(
        FsmHandbrake, required=False, help_text="If present, contains the emergency brake info"
    )

    reports = EmbeddedDocumentField(StartrekReport, default=None, help_text="Parameters for broken host reports")
    repair_request_severity = StringField(
        choices=RepairRequestSeverity.ALL,
        default=RepairRequestSeverity.MEDIUM,
        help_text="Severity of ticket repairs in admin requests (BOT, EINE)",
        required=False,
    )
    cauth_settings = EmbeddedDocumentField(
        CauthSettingsDocument, default=None, required=False, help_text="CAuth settings"
    )

    # WALLESUPPORT-1806
    logical_datacenter = StringField(default=None, help_text="Logical datacenter identifier")

    default_api_fields = ("id", "name", "tags")

    iam_public_fields = (
        # NOTE(rocco66): UI needs
        "deploy_tags",
        "dns_automation",
        "dns_domain",
        "fsm_handbrake.timeout_time",
        "healing_automation",
        "id",
        "name",
        "reboot_via_ssh",
        "tags",
    )

    api_fields = default_api_fields + (
        "type",
        "owners",
        "owned_vlans",
        "vlan_scheme",
        "native_vlan",
        "extra_vlans",
        "profile",
        "profile_tags",
        "provisioner",
        "deploy_config",
        "deploy_tags",
        "deploy_network",
        "deploy_config_policy",
        "dns_domain",
        "yc_dns_zone_id",
        "host_shortname_template",
        "hbf_project_id",
        "bot_project_id",
        "reboot_via_ssh",
        "cms",
        "cms_max_busy_hosts",
        "cms_api_version",
        "cms_tvm_app_id",
        "certificate_deploy",
        "automation_limits",
        "host_limits",
        "automation_plot_id",
        "healing_automation.enabled",
        "healing_automation.status_message",
        "healing_automation.credit",
        "healing_automation.credit_end_time",
        "dns_automation.enabled",
        "dns_automation.status_message",
        "dns_automation.credit",
        "dns_automation.credit_end_time",
        "manually_disabled_checks",
        "fsm_handbrake.issuer",
        "fsm_handbrake.timestamp",
        "fsm_handbrake.timeout_time",
        "fsm_handbrake.reason",
        "fsm_handbrake.ticket_key",
        "fsm_handbrake.audit_log_id",
        "reports",
        "default_host_restrictions",
        "notifications.recipients",
        "repair_request_severity",
        "tier",
        "cms_settings",
        "cauth_settings.flow_type",
        "cauth_settings.trusted_sources",
        "cauth_settings.key_sources",
        "cauth_settings.secure_ca_list_url",
        "cauth_settings.insecure_ca_list_url",
        "cauth_settings.krl_url",
        "cauth_settings.sudo_ca_list_url",
        "maintenance_plot_id",
        "yc_iam_folder_id",
        "logical_datacenter",
    )

    meta = {
        "collection": "projects",
        "indexes": [
            {"name": "unique_name", "fields": ["name"], "unique": True},
            {"name": "tags", "fields": ["tags"]},
        ],
    }

    ALL_AVAILABLE_PROJECT_CHECKS_FIELD = "all_available_project_checks"
    YC_DNS_ZONE_CALCULATED_FIELD = "yc_dns_zone"

    def __unicode__(self):
        return "Project(id={})".format(self.id)

    @classmethod
    def api_query_fields(cls, requested_fields=None):
        if requested_fields:
            fields = list(requested_fields)
            # support for old client and temporary support for UI
            old_automation_field = "enable_automation"
            new_automation_field = "healing_automation.enabled"
            if old_automation_field in fields:
                fields.remove(old_automation_field)
                fields.append(new_automation_field)
            if cls.ALL_AVAILABLE_PROJECT_CHECKS_FIELD in fields:
                fields.remove(cls.ALL_AVAILABLE_PROJECT_CHECKS_FIELD)
                # NOTE(rocco66): we need tags for check infiniband
                fields.extend(["automation_plot_id", "tags"])
            if cls.YC_DNS_ZONE_CALCULATED_FIELD in fields:
                fields.append("yc_dns_zone_id")
            return super().api_query_fields(fields)
        else:
            return super().api_query_fields(requested_fields)

    def to_api_obj(self, requested_fields=None, extra_fields=None, iam_public_handler=False):
        # support for old client and temporary support for UI
        old_automation_field = "enable_automation"

        if requested_fields:
            if old_automation_field in requested_fields:
                automation_enabled = None
                if self.healing_automation:
                    automation_enabled = self.healing_automation.enabled

                if extra_fields is None:
                    extra_fields = {}
                extra_fields.update({old_automation_field: automation_enabled})

            if "hbf_project_id" in requested_fields and self.hbf_project_id is not None:
                if extra_fields is None:
                    extra_fields = {}
                extra_fields["hbf_project_id"] = hex(self.hbf_project_id)[2:]

        return super().to_api_obj(requested_fields, extra_fields, iam_public_handler)

    def has_infiniband(self) -> bool:
        return bool(self.tags and config.get_value("infiniband.involvement_tag") in self.tags)

    def has_tor_link_rule(self) -> bool:
        return bool(not self.tags or config.get_value("tor_link.deactivation_tag") not in self.tags)

    def get_dns_box_config(self) -> tp.Optional[boxes.DnsBoxConfig]:
        return boxes.get_dns_box_config(self.get_dns_box_id())

    def has_fully_enabled_automation(self) -> bool:
        return self.healing_automation.enabled and self.dns_automation.enabled

    def get_dns_box_id(self) -> tp.Optional[str]:
        return boxes.get_box(self.id, boxes.BoxType.dns)

    def get_dns(self) -> dns_clients.interface.DnsClientInterface:
        return dns_clients.get_client(self.get_dns_box_id(), self.yc_dns_zone_id)

    def get_cms(self, context_logger=None):
        query_params = {}

        if self.cms == DEFAULT_CMS_NAME:
            url = config.get_value("default_cms.url") + "/" + self.id
            query_params["strict"] = "true"
            api_version = DEFAULT_CMS_API_VERSION
        else:
            url = self.cms.rstrip("/")
            api_version = self.cms_api_version

        if api_version not in CmsApiVersion.ALL_CMS_API:
            return None

        client = get_cms_client(api_version)
        return client(
            self.get_cms_name(),
            self.id,
            url,
            query_params=query_params,
            context_logger=context_logger,
            use_tvm=self.cms_tvm_app_id is not None,
        )

    def get_cms_clients(self, context_logger=None, filter_condition=None):
        clients = []
        for cms_setting_document in self.cms_settings:
            if filter_condition and not filter_condition(cms_setting_document):
                continue

            cms_settings = CmsSettings.from_cms_setting_document(cms_setting_document)

            query_params = {}

            if cms_settings.url == DEFAULT_CMS_NAME:
                url = config.get_value("default_cms.url") + "/" + self.id
                query_params["strict"] = "true"
                api_version = DEFAULT_CMS_API_VERSION
            else:
                url = cms_settings.url.rstrip("/")
                api_version = cms_settings.api_version

            use_tvm = cms_settings.tvm_app_id is not None

            if api_version not in CmsApiVersion.ALL_CMS_API:
                log.info("%s: error getting cms settings, unknown api version: %s", self.id, api_version)
                return []

            client_cls = get_cms_client(api_version)
            client = client_cls(
                self.get_cms_name_from_url(cms_settings.url),
                self.id,
                url,
                query_params=query_params,
                context_logger=context_logger,
                use_tvm=use_tvm,
            )
            clients.append(client)

        if len(self.cms_settings) >= 2 and len(clients) == 0:
            raise CMSMigrationError(
                "Invalid cms settings, filtered two many cms: {} exists, {} get".format(
                    len(self.cms_settings), len(clients)
                )
            )

        return clients

    def get_cms_name(self):
        if self.cms == DEFAULT_CMS_NAME:
            return self.cms + "." + self.id
        else:
            return self.cms.rstrip("/")

    def get_cms_name_from_url(self, url):
        if url == DEFAULT_CMS_NAME:
            return url + "." + self.id
        else:
            return url.rstrip("/")

    def authorize(self, issuer, allowed_host_types=None):
        if has_iam():
            return

        # Protects us from using an object without required fields
        if self.id is None:
            raise LogicalError()

        if self.type and allowed_host_types and self.type not in allowed_host_types:
            raise OperationRestrictedForHostTypeError(self.type, allowed_host_types)

        blackbox.authorize(
            issuer,
            "You must be '{}' project owner to perform this request.".format(self.id),
            owners=get_project_owners(self),
            authorize_admins=True,
        )

    def authorize_owner(self, issuer):
        self.authorize(issuer)

    def authorize_user(self, issuer):
        if has_iam():
            return

        # owner > user
        try:
            self.authorize_owner(issuer)
        except UnauthorizedError:
            pass
        else:
            return

        users_and_superusers = []
        for role in [ProjectRole.USER, ProjectRole.SUPERUSER]:
            role_manager = ProjectRole.get_role_manager(role, self)
            users_and_superusers.extend(role_manager.list_members())

        blackbox.authorize(
            issuer,
            "You must be '{}' project user or owner to perform this request.".format(self.id),
            owners=users_and_superusers,
        )

    def get_maintenance_plot(self, fields=None) -> tp.Optional[MaintenancePlotModel]:
        if not self.maintenance_plot_id:
            return None

        objects = MaintenancePlotModel.objects(id=self.maintenance_plot_id)
        if fields is not None:
            objects = objects.only(*fields)

        try:
            return objects.get()
        except mongoengine.DoesNotExist:
            raise Error(
                "Internal error occurred: project {} have unknown maintenance plot {}.",
                self.id,
                self.maintenance_plot_id,
            )


def get_authorized(issuer, project_id, additional_fields=(), allowed_host_types=None):
    """Authorizes modification of the project."""

    fields = ["id", "owners"] + list(additional_fields)

    project = get_by_id(project_id, fields=fields)
    project.authorize(issuer, allowed_host_types=allowed_host_types)
    return project


def authorize_vlans(project_id, vlans, allow_not_found_error=False):
    """Authorizes usage of the specified VLANs in the specified project.

    :returns a list of owned VLANs that the project must have to be allowed to switch these VLANs (this may be useful
    for atomic update of the project).
    """

    vlans = set(vlans)

    try:
        owned_vlans = get_by_id(project_id, fields=("owned_vlans",)).owned_vlans
    except ProjectNotFoundError:
        if allow_not_found_error:
            raise
        raise Error("Failed to authorize VLAN usage for '{}' project: the project doesn't exist.", project_id)

    authorized_vlans = set(owned_vlans + walle_constants.SHARED_VLANS)

    unauthorized_vlans = vlans - authorized_vlans
    if unauthorized_vlans:
        raise UnauthorizedError(
            "Project {} doesn't own the following VLANs: {}.",
            project_id,
            ", ".join(str(vlan) for vlan in unauthorized_vlans),
        )

    return list(vlans - set(walle_constants.SHARED_VLANS))


def check_id(project_id):
    """Checks a project ID."""

    get_by_id(project_id, fields=("id",))


def get_by_id(project_id, fields=None) -> Project:
    """Returns a project by its ID."""

    objects = Project.objects(id=project_id)
    if fields is not None:
        objects = objects.only(*fields)

    try:
        return objects.get()
    except mongoengine.DoesNotExist:
        raise ProjectNotFoundError()


def get_project_owners(project):
    role_manager = ProjectRole.get_role_manager(ProjectRole.OWNER, project)
    return role_manager.list_members()


def is_reboot_via_ssh_enabled(project):
    role_manager = ProjectRole.get_role_manager(ProjectRole.SSH_REBOOTER, project)
    return ROBOT_WALLE_OWNER in role_manager.list_members()


def ipxe_support_enabled(project_id):
    disabled_projects = config.get_value("deployment.projects_without_ipxe_support")
    if not isinstance(disabled_projects, set):
        config.set_value("deployment.projects_without_ipxe_support", set(disabled_projects))

    return project_id not in disabled_projects


def get_default_project_automation_limits():
    limits = get_default_automation_limits()
    return {limit_name: limits for limit_name in AUTOMATION_LIMIT_NAMES}


def get_default_shadow_project_automation_limits():
    limits = [{PERIOD: "1d", LIMIT: 0}]
    return {limit_name: limits for limit_name in AUTOMATION_LIMIT_NAMES}


def get_default_host_limits():
    return HOST_LIMITS_NAMES_WITH_DEFAULT_VALUES


def get_default_automation_limits(failure=None):
    if failure == get_walle_check_type(CheckType.WALLE_RACK):
        limit = DEFAULT_RACK_FAILURE_LIMIT
    else:
        limit = DEFAULT_AUTOMATION_LIMIT
    return [{PERIOD: "1d", LIMIT: limit}]


def project_id_to_cms_tvm_alias(project_id, cms_url):
    return "cms:{}:{}".format(project_id, cms_url.rstrip("/"))


def map_cms_project_alias_to_tvm_app_id():
    project_iter = Project.objects().only("id", "cms_settings")
    map_alias_to_tvm_app_id = {}
    for prj in gevent_idle_iter(project_iter):
        for cms_setting in prj.cms_settings:
            if cms_setting.cms_tvm_app_id:
                key = project_id_to_cms_tvm_alias(prj.id, cms_setting.cms)
                map_alias_to_tvm_app_id[key] = cms_setting.cms_tvm_app_id
    return map_alias_to_tvm_app_id


# TODO(rocco66): move to walle.boxes
def get_eine_box(project_id: str) -> tp.Optional[str]:
    return boxes.get_box(project_id, boxes.BoxType.eine)


DEFAULT_CMS_API_VERSION = CmsApiVersion.V1_0


class CmsSettings(namedtuple("CmsSettings", "url max_busy_hosts api_version tvm_app_id temporary_unreachable_enabled")):
    @classmethod
    def from_project(cls, project):
        return cls(
            url=project["cms"],
            max_busy_hosts=project["cms_max_busy_hosts"],
            api_version=project["cms_api_version"],
            tvm_app_id=project["cms_tvm_app_id"],
            temporary_unreachable_enabled=False,
        )

    @classmethod
    def from_request(cls, request_cms):
        return cls(
            url=request_cms["url"],
            max_busy_hosts=request_cms.get("max_busy_hosts"),
            api_version=request_cms.get("api_version"),
            tvm_app_id=request_cms.get("tvm_app_id"),
            temporary_unreachable_enabled=request_cms.get("temporary_unreachable_enabled"),
        )

    @classmethod
    def from_cms_setting_document(cls, cms_setting_document):
        return cls(
            url=cms_setting_document.cms,
            max_busy_hosts=cms_setting_document.cms_max_busy_hosts,
            api_version=cms_setting_document.cms_api_version,
            tvm_app_id=cms_setting_document.cms_tvm_app_id,
            temporary_unreachable_enabled=cms_setting_document.temporary_unreachable_enabled,
        )

    def to_project(self):
        return dict(
            cms=self.url,
            cms_max_busy_hosts=self.max_busy_hosts,
            cms_api_version=self.api_version,
            cms_tvm_app_id=self.tvm_app_id,
            temporary_unreachable_enabled=self.temporary_unreachable_enabled,
        )

    def to_cms_settings_document(self):
        return CmsSettingsDocument(
            cms=self.url,
            cms_api_version=self.api_version,
            cms_max_busy_hosts=self.max_busy_hosts,
            cms_tvm_app_id=self.tvm_app_id,
            temporary_unreachable_enabled=self.temporary_unreachable_enabled,
        )

    def supports_tvm(self):
        if self.url == "default":
            return False
        elif self.api_version in CmsApiVersion.ALL_CMS_API:
            return True
        raise LogicalError()

    def to_default_cms_kwargs(self):
        return dict(cms=self.url, cms_max_busy_hosts=self.max_busy_hosts, temporary_unreachable_enabled=False)

    def to_user_cms_api_kwargs(self):
        return dict(
            cms=self.url,
            cms_api_version=self.api_version,
            cms_tvm_app_id=self.tvm_app_id,
            temporary_unreachable_enabled=self.temporary_unreachable_enabled,
        )


def is_temporary_unreachable_enabled(cms_settings: CmsSettingsDocument) -> bool:
    return cms_settings.temporary_unreachable_enabled
