import time
import uuid

from flask import Request, request, g
from six.moves.urllib.parse import unquote_plus

from walle.statbox.loggers import api_logger


class WalleRequest(Request):
    def __init__(self, *args, **kwargs):
        self.request_id = make_request_id()
        super().__init__(*args, **kwargs)


def _get_path_query_params(request):
    path = unquote_plus(request.path)
    split_url = request.url.split("?", 1)
    if len(split_url) > 1:
        return path, unquote_plus(split_url[1])
    else:
        return path, None


def make_request_id():
    return uuid.uuid4().hex


def log_request():
    path, query_params = _get_path_query_params(request)
    g.request_start_time = time.time()
    api_logger().log(path=path, query_params=query_params, http_method=request.method, walle_action="api_request")


def log_response(response):
    path, query_params = _get_path_query_params(request)
    ip = request.headers.get("X-Forwarded-For-Y") or request.remote_addr
    request_time = time.time() - g.request_start_time
    api_logger().log(
        path=path,
        ip=ip,
        query_params=query_params,
        http_method=request.method,
        walle_action="api_response",
        request_time=request_time,
        status_code=response.status_code,
    )
    return response
