"""Host restriction management."""

import copy
import logging

from walle.errors import OperationRestrictedError

log = logging.getLogger(__name__)


REBOOT = "reboot"
"""Prohibit reboot."""

PROFILE = "profile"
"""Prohibit profiling."""

REDEPLOY = "redeploy"
"""Prohibit redeployment."""

AUTOMATED_DNS = "automated-dns"
"""Prohibit automated DNS modification."""

AUTOMATED_REBOOT = "automated-reboot"
"""Prohibit automated reboot."""

AUTOMATED_PROFILE = "automated-profile"
"""Prohibit automated profiling."""

AUTOMATED_REDEPLOY = "automated-redeploy"
"""Prohibit automated redeployment."""

AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP = "automated-profile-with-full-disk-cleanup"
"""Prohibit automated profiling with using 'dangerous load' tag"""

AUTOMATED_MEMORY_CHANGE = "automated-memory-change"
"""Prohibit automated change of a corrupted memory."""

AUTOMATED_DISK_CHANGE = "automated-disk-change"
"""Prohibit automated change of a corrupted disk."""

AUTOMATED_DISK_CABLE_REPAIR = "automated-disk-cable-repair"
"""Prohibit automated repair of malfunctioning disk cable."""

AUTOMATED_LINK_REPAIR = "automated-link-repair"
"""Prohibit automated repair of malfunctioning link."""

AUTOMATED_BMC_REPAIR = "automated-bmc-repair"
"""Prohibit automated repair of problems with BMC."""

AUTOMATED_CPU_REPAIR = "automated-cpu-repair"
"""Prohibit automated CPU repairing."""

AUTOMATED_GPU_REPAIR = "automated-gpu-repair"
"""Prohibit automated GPU repairing."""

AUTOMATED_INFINIBAND_REPAIR = "automated-infiniband-repair"
"""Prohibit automated INFINIBAND repairing."""

AUTOMATED_MEMORY_REPAIR = "automated-memory-repair"
"""Prohibit automated memory repairing."""

AUTOMATED_OVERHEAT_REPAIR = "automated-overheat-repair"
"""Prohibit automated overheat repairing."""

AUTOMATED_CAPPING_REPAIR = "automated-capping-repair"
"""Prohibit automated capping repairing."""

AUTOMATED_HEALING = "automated-healing"
"""Prohibit automated host healing."""

AUTOMATED_RACK_REPAIR = "automated-rack-repair"
"""Prohibit automated rack repairing."""

AUTOMATED_REPAIRING = "automated-repairing"
"""Prohibit all automated hardware repairing."""

AUTOMATION = "automation"
"""Prohibit any automation."""


# It looks like restrictions have a one-to-one mapping with decisions,
# But actually some decisions trigger several restrictions and the list of `several other restrictions` is dynamic.
# Although, I think there is a way to avoid duplication, I just deposit it for the better time.
ALL = [
    REBOOT,
    PROFILE,
    REDEPLOY,
    AUTOMATION,  # this one triggers all restrictions below
    AUTOMATED_HEALING,  # this one triggers all restrictions below except DNS
    AUTOMATED_REBOOT,
    AUTOMATED_PROFILE,
    AUTOMATED_REDEPLOY,
    AUTOMATED_REPAIRING,  # this one triggers all restrictions below except DNS
    AUTOMATED_MEMORY_CHANGE,
    AUTOMATED_DISK_CHANGE,
    AUTOMATED_LINK_REPAIR,
    AUTOMATED_DISK_CABLE_REPAIR,
    AUTOMATED_BMC_REPAIR,
    AUTOMATED_GPU_REPAIR,
    AUTOMATED_CPU_REPAIR,
    AUTOMATED_MEMORY_REPAIR,
    AUTOMATED_OVERHEAT_REPAIR,
    AUTOMATED_CAPPING_REPAIR,
    AUTOMATED_RACK_REPAIR,
    AUTOMATED_DNS,
    AUTOMATED_INFINIBAND_REPAIR,
    AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP,
]
"""A list of available restrictions.

Attention: The order of restrictions determines their order in UI.
"""


def _expand_restriction_mapping(collapsed_mapping):
    changed = True
    expanded_mapping = copy.deepcopy(collapsed_mapping)

    while changed:
        changed = False

        for dependency, dependents in expanded_mapping.items():
            start_len = len(dependents)

            for dependent in dependents.copy():
                try:
                    expanded_dependent = expanded_mapping[dependent]
                except KeyError:
                    pass
                else:
                    dependents.update(expanded_dependent)

            changed |= len(dependents) != start_len

    return expanded_mapping


RESTRICTION_MAPPING = _expand_restriction_mapping(
    {
        REBOOT: {PROFILE, REDEPLOY, AUTOMATED_REBOOT},
        PROFILE: {AUTOMATED_PROFILE},
        REDEPLOY: {AUTOMATED_REDEPLOY},
        AUTOMATED_REBOOT: {
            AUTOMATED_PROFILE,
            AUTOMATED_REDEPLOY,
            AUTOMATED_MEMORY_CHANGE,
            AUTOMATED_OVERHEAT_REPAIR,
            AUTOMATED_CAPPING_REPAIR,
            AUTOMATED_GPU_REPAIR,
            AUTOMATED_CPU_REPAIR,
            AUTOMATED_BMC_REPAIR,
            AUTOMATED_MEMORY_REPAIR,
        },
        AUTOMATED_PROFILE: {AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP},
        AUTOMATED_REPAIRING: {
            AUTOMATED_MEMORY_CHANGE,
            AUTOMATED_DISK_CHANGE,
            AUTOMATED_LINK_REPAIR,
            AUTOMATED_BMC_REPAIR,
            AUTOMATED_GPU_REPAIR,
            AUTOMATED_CPU_REPAIR,
            AUTOMATED_MEMORY_REPAIR,
            AUTOMATED_OVERHEAT_REPAIR,
            AUTOMATED_CAPPING_REPAIR,
            AUTOMATED_RACK_REPAIR,
            AUTOMATED_DISK_CABLE_REPAIR,
            AUTOMATED_INFINIBAND_REPAIR,
        },
        AUTOMATED_HEALING: {AUTOMATED_REBOOT, AUTOMATED_PROFILE, AUTOMATED_REDEPLOY, AUTOMATED_REPAIRING},
        AUTOMATION: {AUTOMATED_DNS, AUTOMATED_HEALING},
    }
)
"""Maps restrictions to restrictions that they include.

In other words it's `$key -> $v[]`: $key prohibits all $v.
"""

EXCLUDE_FOR_MACS = {REBOOT, AUTOMATION, AUTOMATED_HEALING, AUTOMATED_REBOOT}


def _get_restriction_reverse_mapping():
    mapping = {}

    for parent_restriction, children_restrictions in RESTRICTION_MAPPING.items():
        for child_restriction in children_restrictions:
            mapping.setdefault(child_restriction, set()).add(parent_restriction)

    return mapping


_RESTRICTION_REVERSE_MAPPING = _get_restriction_reverse_mapping()
"""Maps restrictions to restrictions that are included by.

In other words it's `$key -> $v[]`: $key is prohibited by all $v.
"""


def check_restrictions(host, restrictions):
    """Checks host for presence of the specified restriction."""

    if isinstance(restrictions, str):
        restrictions = (restrictions,)

    expanded_restrictions = expand_restrictions(restrictions)
    applied_restrictions = host.applied_restrictions(*expanded_restrictions)
    if applied_restrictions:
        raise OperationRestrictedError(*_explain_restrictions(applied_restrictions, restrictions))

    return expanded_restrictions


def strip_restrictions(restrictions, strip_to_none=False):
    """Strips restrictions gotten from user."""

    if not restrictions:
        return None if strip_to_none else []

    restrictions = set(restrictions)
    for parent_restriction, children_restrictions in RESTRICTION_MAPPING.items():
        if parent_restriction in restrictions:
            restrictions.difference_update(children_restrictions)

    return sorted(restrictions)


def expand_restrictions(restrictions):
    """Expands stripped restrictions to a full list."""

    restrictions = set(restrictions)

    for restriction in restrictions & set(_RESTRICTION_REVERSE_MAPPING):
        restrictions.update(_RESTRICTION_REVERSE_MAPPING[restriction])

    return sorted(restrictions)


def _explain_restrictions(applied_restrictions, operation_restrictions):
    reason = "Operation restricted for this host. Restrictions applied: [{}]"
    reason_args = [", ".join(sorted(applied_restrictions))]

    stripped_restrictions = operation_restrictions
    if set(stripped_restrictions) != applied_restrictions:
        reason += ", operation restrictions: [{}]"
        reason_args += [", ".join(sorted(stripped_restrictions))]

    return _mk_tuple(reason, *reason_args)


def _mk_tuple(*args):
    return args
