from datetime import datetime, timedelta

from sepelib.core import constants
from walle.clients import calendar
from walle.errors import WalleError
from walle.hosts import Host
from walle.maintenance_plot.model import MaintenancePlot
from walle.projects import Project
from walle.scenario.data_storage.base import BaseScenarioDataStorage
from walle.scenario.errors import CorrectYpSlaDoesNotExists
from walle.util.mongo import MongoDocument

DEFAULT_OFFSET_FOR_YP_SLA = constants.WEEK_SECONDS * 5


def get_inv_to_tags_map(invs: list[int]) -> dict[int, list[str]]:
    hosts, project_tags = _get_hosts_and_tags(invs)
    return _hosts_to_tags(hosts, project_tags)


def _hosts_to_tags(hosts, tags) -> dict[int, list[str]]:
    result = {host.inv: tags[host.project] for host in hosts}

    return result


def _get_hosts_and_tags(invs: list[int]):
    hosts = list(
        MongoDocument.for_model(Host).find(
            {"inv": {"$in": invs}}, {"inv": 1, "project": 1, "location.short_datacenter_name": 1, "_id": 0}
        )
    )

    projects = {host.project for host in hosts}

    tags = {
        _item.id: _item.tags or []  # Handle projects w/o tags.
        for _item in MongoDocument.for_model(Project).find({"_id": {"$in": list(projects)}}, {"_id": 1, "tags": 1})
    }
    return hosts, tags


def get_request_to_maintenance_starting_time_using_yp_sla(maintenance_start_time: int) -> int:
    """
    So this is what we want to do here:
    S - start date for requesting CMS
    M - maintenance start date (ITDC/NOC trying to unload switch/rack by this date)
    X - either workday or holiday/weekend
    W - workday
    H - holiday/weekend

    We want to find such day closest to M to fulfill such a condition:
    SXXXM - among the X are two W and one H
    or
    SWWM - two W between S and M

    For now we have such conditions:
    - YP uses hard timeout (just forceful unload for host, drop all pods and go on) of 4 days
    - Nanny team (@alonger and etc) wants to have two workdays to fight the spoiled/bad pods
    """

    date_to = datetime.fromtimestamp(maintenance_start_time)
    date_from = datetime.fromtimestamp(maintenance_start_time - DEFAULT_OFFSET_FOR_YP_SLA)

    response = calendar.get_holidays(
        date_from=date_from.strftime("%Y-%m-%d"), date_to=date_to.strftime("%Y-%m-%d"), for_countries=["rus"]
    )
    holidays = {item["date"] for item in response["holidays"]}

    end_date = date_to
    start_date = None

    while end_date > date_from:
        working_days = 0
        for days in range(1, 4):
            date = end_date - timedelta(days=days)

            if date.strftime("%Y-%m-%d") not in holidays:
                working_days += 1

            if working_days >= 2:
                start_date = date - timedelta(days=1)
                break

        if start_date:
            break

        end_date = end_date - timedelta(days=1)

    if not start_date:
        raise CorrectYpSlaDoesNotExists(maintenance_start_time)

    return start_date.timestamp()


def get_host_group_maintenance_plot(host_group_id: int, data_storage: BaseScenarioDataStorage) -> MaintenancePlot:
    for host_group_source in data_storage.read_host_groups_sources():
        if host_group_source.group_id == host_group_id:
            return host_group_source.source.get_maintenance_plot()
    raise WalleError("Can not find host group source with ID '%d' in scenario's data storage")
